package com.tandbergtv.watchpoint.studio.ui.properties;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IProject;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.CheckboxCellEditor;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.jpdl.model.ProcessState;
import org.jbpm.gd.jpdl.model.StartState;

import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.ui.editor.expression.ExpressionEditor;
import com.tandbergtv.watchpoint.studio.ui.model.IExpression;
import com.tandbergtv.watchpoint.studio.ui.model.IValidatableElement;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

public class ProcessStateSectionComposite {

    private TabbedPropertySheetWidgetFactory widgetFactory;

    private Composite parent;
    private Composite composite;

    private Label subProcessNameLabel;
    private CCombo subProcessNameCombo;
    private Label messageLabel;
    private Button addButton;
    private Button removeButton;
    private Table table;
    private TableViewer tableViewer;

    private ProcessStateCellModifier cellModifier;
    private ExpressionEditor expressionEditor;

    private String[] columnTitles = new String[] { "Template Variable", "SubProcess Variable", "Read", "Write",
            "Required" };

    private TableColumn readColumn;

    private TableColumn writeColumn;

    private TableColumn requiredColumn;

    private Map<String, String> templatePathByProcessName;

    public ProcessStateSectionComposite(Composite parent, TabbedPropertySheetWidgetFactory widgetFactory) {
        this.widgetFactory = widgetFactory;
        this.parent = parent;

        createComponents();
    }

    private void createComponents() {
        composite = widgetFactory.createFlatFormComposite(parent);

        createCombo();
        createLabel();
        createButtons();
        createTable();
        createTableViewer();
    }

    private void createCombo() {
        subProcessNameLabel = widgetFactory.createLabel(composite, "Subprocess Name");
        FormData formData = new FormData();
        formData.left = new FormAttachment(0, 5);
        formData.top = new FormAttachment(0, 2);
        subProcessNameLabel.setLayoutData(formData);

        subProcessNameCombo = widgetFactory.createCCombo(composite, SWT.DROP_DOWN);
        formData = new FormData();
        formData.left = new FormAttachment(subProcessNameLabel, 0);
        formData.top = new FormAttachment(0, 0);
        formData.right = new FormAttachment(50, 0);
        subProcessNameCombo.setLayoutData(formData);

        subProcessNameCombo.addModifyListener(new ModifyListener() {
            @Override
            public void modifyText(ModifyEvent e) {
                String subprocname = subProcessNameCombo.getText().substring(0, subProcessNameCombo.getText().indexOf(" - v"));
                ProcessState processState = (ProcessState) tableViewer.getInput();
                processState.getSubProcess().setName(subprocname);

                updateOrCreteVariablesCombo(subprocname);
            }

            private void updateOrCreteVariablesCombo(String subprocname) {
                String subprocessPath = templatePathByProcessName.get(subprocname);

                StartState startState = Utility.extractStartState(subprocessPath);

                String[] variables = Utility.toNameArray(startState.getValidVariables());

                cellModifier.setVariables(Arrays.asList(variables));
                if (tableViewer.getCellEditors()[1] instanceof TextCellEditor) {
                    tableViewer.getCellEditors()[1] = new ComboBoxCellEditor(table, new String[0], SWT.BORDER | SWT.READ_ONLY);
                    expressionEditor.addFocusListener(tableViewer.getCellEditors()[1].getControl());
                }
                ((ComboBoxCellEditor)tableViewer.getCellEditors()[1]).setItems(variables);
            }
        });
    }

    private void createLabel() {
        messageLabel = widgetFactory.createLabel(composite, "Define the used variables :");
        FormData formData = new FormData();
        formData.top = new FormAttachment(subProcessNameLabel, 0, 5);
        formData.left = new FormAttachment(0, 5);
        formData.right = new FormAttachment(100, -5);
        messageLabel.setLayoutData(formData);
    }

    private void createButtons() {
        addButton = widgetFactory.createButton(composite, "+", SWT.PUSH);
        addButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                ProcessState processState = (ProcessState) tableViewer.getInput();
                WPVariable variable = processState.addEmptyVariable(Utility.getNextAvailableVariableName(table));

                tableViewer.add(variable);
            }
        });
        FormData formData = new FormData();
        formData.top = new FormAttachment(messageLabel, 0, 0);
        formData.right = new FormAttachment(100, 0);
        formData.width = 25;
        addButton.setLayoutData(formData);

        removeButton = widgetFactory.createButton(composite, "-", SWT.PUSH);
        formData = new FormData();
        formData.top = new FormAttachment(addButton, 0);
        formData.right = new FormAttachment(100, 0);
        formData.width = 25;
        removeButton.setLayoutData(formData);

        removeButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                if (((IStructuredSelection) tableViewer.getSelection()).getFirstElement() != null) {
                    ProcessState processState = (ProcessState) tableViewer.getInput();
                    WPVariable variable = (WPVariable) ((IStructuredSelection) tableViewer.getSelection())
                            .getFirstElement();

                    processState.removeVariable(variable);

                    tableViewer.remove(variable);
                }
            }
        });
    }

    private void createTable() {
        //table will be created in a exclusive parent in order to use TableColumnLayout
        Composite exclusiveTableParent = widgetFactory.createFlatFormComposite(composite);

        int tableStyle = SWT.SINGLE | SWT.BORDER | SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.HIDE_SELECTION;
        table = widgetFactory.createTable(exclusiveTableParent, tableStyle);
        table.setLinesVisible(true);
        table.setHeaderVisible(true);

        FormData layoutData = new FormData();
        layoutData.left = new FormAttachment(0, 0);
        layoutData.top = new FormAttachment(messageLabel, 0);
        layoutData.right = new FormAttachment(addButton, 0);
        layoutData.bottom = new FormAttachment(100, 0);
        layoutData.width = 600;
        exclusiveTableParent.setLayoutData(layoutData);

        TableColumn nameColumn = new TableColumn(table, SWT.LEFT);
        nameColumn.setText(columnTitles[0]);
        TableColumn mappedNameColumn = new TableColumn(table, SWT.LEFT);
        mappedNameColumn.setText(columnTitles[1]);
        readColumn = new TableColumn(table, SWT.CENTER);
        readColumn.setText(columnTitles[2]);
        readColumn.setResizable(false);
        writeColumn = new TableColumn(table, SWT.CENTER);
        writeColumn.setText(columnTitles[3]);
        writeColumn.setResizable(false);
        requiredColumn = new TableColumn(table, SWT.CENTER);
        requiredColumn.setText(columnTitles[4]);
        requiredColumn.setResizable(false);

        TableColumnLayout layout = new TableColumnLayout();
        exclusiveTableParent.setLayout(layout);
        layout.setColumnData(nameColumn, new ColumnWeightData(20, 100));
        layout.setColumnData(mappedNameColumn, new ColumnWeightData(20, 100));
        layout.setColumnData(readColumn, new ColumnWeightData(4, 60, false));
        layout.setColumnData(writeColumn, new ColumnWeightData(4, 60, false));
        layout.setColumnData(requiredColumn, new ColumnWeightData(4, 60, false));
    }

    private void createTableViewer() {
        tableViewer = new TableViewer(table);
        
        tableViewer.setUseHashlookup(true);
        tableViewer.setColumnProperties(columnTitles);
        tableViewer.setContentProvider(new ProcessStateContentProvider());
        tableViewer.setLabelProvider(new ProcessStateLabelProvider(readColumn, writeColumn, requiredColumn));

        CellEditor[] editors = new CellEditor[columnTitles.length];
        editors[0] = new TextCellEditor(table, SWT.BORDER);
        editors[1] = new TextCellEditor(table, SWT.BORDER);
        editors[2] = new CheckboxCellEditor(table);
        editors[3] = new CheckboxCellEditor(table);
        editors[4] = new CheckboxCellEditor(table);

        // instantiate expressionEditor
        expressionEditor = new ExpressionEditor(tableViewer, 0);

        // Assign the cell editors to the viewer 
        tableViewer.setCellEditors(editors);

        cellModifier = new ProcessStateCellModifier(tableViewer, columnTitles);

        // Set the cell modifier for the viewer
        tableViewer.setCellModifier(cellModifier);

        TableViewerColumn viewerColumn = new TableViewerColumn(tableViewer, readColumn);
        viewerColumn.setLabelProvider((CellLabelProvider) tableViewer.getLabelProvider());
        viewerColumn = new TableViewerColumn(tableViewer, writeColumn);
        viewerColumn.setLabelProvider((CellLabelProvider) tableViewer.getLabelProvider());
        viewerColumn = new TableViewerColumn(tableViewer, requiredColumn);
        viewerColumn.setLabelProvider((CellLabelProvider) tableViewer.getLabelProvider());
        
        table.pack();
    }

    public void setInput(ProcessState processState, List<WorkflowTemplateDTO> templateList, IProject project) {
        tableViewer.setInput(processState);
        templatePathByProcessName = getTemplatePathByProcessName(processState, templateList);
        selectCurrentComboOption(processState, updateComboOptions(processState, templateList));
        expressionEditor.setupAndInitialize(Utility.getVariables(processState), project, new CustomExpression());
    }

    private Map<String, String> getTemplatePathByProcessName(ProcessState processState, List<WorkflowTemplateDTO> templateList) {
        Map<String,String> templatePathByProcessName = new HashMap<String, String>();
        String currentTemplateName = Utility.getTemplate(processState).getName();
        for (WorkflowTemplateDTO dto: templateList) {
            if (dto.getSubprocesses().isEmpty() && !dto.getName().equals(currentTemplateName)) {
                templatePathByProcessName.put(dto.getName(), dto.getPath());
            }
        }
        
        return templatePathByProcessName;
    }

    private List<String> updateComboOptions(ProcessState processState, List<WorkflowTemplateDTO> templateList) {
        List<String> comboOptions = new ArrayList<String>();
        for (WorkflowTemplateDTO dto : templateList) {
            if (dto.getSubprocesses().isEmpty() && !dto.getName().equals(Utility.getTemplate(processState).getName())) {
                comboOptions.add(dto.toString());
            }
        }

        //Upper case goes before lower case
        Collections.sort(comboOptions);
        subProcessNameCombo.setItems(comboOptions.toArray(new String[0]));
        return comboOptions;
    }

    private void selectCurrentComboOption(ProcessState processState, List<String> comboOptions) {
        int currentSelected = -1;
        String option = null;
        for (int i = 0; i < comboOptions.size(); i++) {
            option = comboOptions.get(i);
            if (processState.getSubProcess() != null && option.substring(0, option.indexOf(" - v")).equals(processState.getSubProcess().getName())) {
                currentSelected = i;
                break;
            }
        }

        if (currentSelected >= 0) {
            subProcessNameCombo.select(currentSelected);
        }
    }

    class CustomExpression implements IExpression {
        private IValidatableElement model;

        @Override
        public void setModel(IValidatableElement model) {
            this.model = model;
        }

        @Override
        public IValidatableElement getModel() {
            return model;
        }

        @Override
        public String getText() {
            WPVariable variable = (WPVariable) model;
            return variable.getName();
        }

        @Override
        public DataType getReturnType() {
            WPVariable variable = (WPVariable) model;
            if (subProcessNameCombo.getSelectionIndex() > -1 && variable.getMappedName() != null && !variable.getMappedName().isEmpty()) {
                String path = templatePathByProcessName.get(((ProcessState)tableViewer.getInput()).getSubProcess().getName());
                StartState startState = Utility.extractStartState(path);
                WPVariable selectedVariable = startState.getVariable(variable.getMappedName());
                if (selectedVariable != null && selectedVariable.getType() != null && !selectedVariable.getType().isEmpty()) {
                    return DataType.valueOf(selectedVariable.getType());
                }
            }

            return DataType.OBJECT;
        }

        @Override
        public boolean isValidatable() {
            WPVariable variable = (WPVariable) model;
            if (!variable.isWritable()) {
                return true;
            }
            return false;
        }
    };
}