package com.tandbergtv.watchpoint.studio.ui.properties;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.draw2d.ColorConstants;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.common.model.GenericElement;
import org.jbpm.gd.jpdl.model.Action;
import org.jbpm.gd.jpdl.util.AutoResizeTableLayout;

import com.tandbergtv.watchpoint.studio.ui.actionconfig.ActionClassConfiguration;
import com.tandbergtv.watchpoint.studio.ui.actionconfig.ActionConfigurationManager;
import com.tandbergtv.watchpoint.studio.ui.actionconfig.ActionVariableConfiguration;
import com.tandbergtv.watchpoint.studio.ui.model.AbstractTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants;
import com.tandbergtv.watchpoint.studio.ui.properties.template.ArgumentTableItemWrapper;

public class TaskNodeActionElementComposite implements SelectionListener, FocusListener {

	// ========================================================================
	// ===================== DATA MANAGED BY COMPOSITE
	// ========================================================================

	private AbstractTaskNode taskNode;
	// ========================================================================
	// ===================== THE CONTROLS USED
	// ========================================================================

	private Composite parent;

	private TabbedPropertySheetWidgetFactory widgetFactory;

	private Label classNameLabel, argumentsLabel;

	private CCombo classNameCombo;

	private Table argumentsTable;

	private Button addButton, removeButton;

	private Map<String, String> arguments;

	// ========================================================================
	// ===================== CONSTANTS USED
	// ========================================================================

	private static final String DEFAULT_CLASS_NAME = "";

	private static final String CLASS_NAME_LABEL = "Class Name: ";

	private static final String ARGUMENTS_LABEL = "Arguments: ";

	private static final String[] COLUMN_TITLES = new String[] { "Name", "Value" };

	// ========================================================================
	// ===================== CONSTRUCTOR
	// ========================================================================

	/**
	 * Creates a new ActionConfigurationComposite and initializes it.
	 * 
	 * @param parent
	 *            a widget which will be the parent of the new instance (cannot be null)
	 * @param widgetFactory
	 *            The Factory to use to create widgets / child composites for this composite.
	 */
	public TaskNodeActionElementComposite(Composite parent,
			TabbedPropertySheetWidgetFactory widgetFactory) {
		this.parent = parent;
		this.widgetFactory = widgetFactory;

		this.createControls();
	}

	// ========================================================================
	// ===================== CREATE AND INITIALIZE CONTROLS
	// ========================================================================

	/*
	 * Create and initialize the controls.
	 */
	private void createControls() {
		this.arguments = new LinkedHashMap<String, String>();
		this.create();

		// Initialize the Layout for the controls
		this.initializeArgumentsTable();
	}

	/*
	 * Creates the widgets for this composite.
	 * 
	 */
	private void create() {
		Composite composite = widgetFactory.createFlatFormComposite(parent);
		classNameLabel = new Label(composite, SWT.NONE);
		classNameLabel.setText(CLASS_NAME_LABEL);
		classNameLabel.setLayoutData(createClassLabelLayoutData());
		classNameLabel.setBackground(ColorConstants.white);
		classNameLabel.pack();

		classNameCombo = widgetFactory.createCCombo(parent, SWT.DROP_DOWN);
		classNameCombo.setLayoutData(createClassNameLayoutData(0, 105, 320));
		classNameCombo.setItems(this.getConfiguredClassNames());
		classNameCombo.add(DEFAULT_CLASS_NAME, 0);
		classNameCombo.setEditable(true);
		classNameCombo.addFocusListener(this);
		classNameCombo.pack();
		
		argumentsLabel = new Label(composite, SWT.NONE);
		argumentsLabel.setText(ARGUMENTS_LABEL);
		argumentsLabel.setLayoutData(createArgumentLabelLayoutData());
		argumentsLabel.setBackground(ColorConstants.white);
		argumentsLabel.pack();
		argumentsTable = widgetFactory.createTable(parent, SWT.V_SCROLL | SWT.H_SCROLL
				| SWT.FULL_SELECTION);
		argumentsTable.setLayoutData(createArgumentsTableLayoutData());
		argumentsTable.addSelectionListener(this);
		argumentsTable.pack();

		addButton = widgetFactory.createButton(parent, "Add", SWT.PUSH);
		removeButton = widgetFactory.createButton(parent, "Remove", SWT.PUSH);
		addButton.setLayoutData(createAddButtonLayoutData());
		addButton.addSelectionListener(this);
		removeButton.setLayoutData(createRemoveButtonLayoutData());
		removeButton.addSelectionListener(this);
	}

	// ========================================================================
	// ===================== THE TASK NODE AND ACTION ELEMENT
	// ========================================================================

	/**
	 * Set the Task Node for which the action element is displayed.
	 * 
	 * @param taskNode
	 *            The Task Node
	 */
	public void setTaskNode(AbstractTaskNode taskNode) {
		// No change to the task node, do nothing
		if (this.taskNode == taskNode)
			return;

		unhookListeners();
		clearControls();

		this.taskNode = taskNode;
		if (this.taskNode != null) {
			updateControls();
			hookListeners();
		}
	}

	/**
	 * Get the Task Node associated with this composite.
	 * 
	 * @return The Task Node associated with this composite.
	 */
	public AbstractTaskNode getTaskNode() {
		return this.taskNode;
	}

	/*
	 * Get the Action contained in the Task Node or null if no action / task node exists.
	 */
	private Action getAction() {
		return (this.taskNode != null) ? this.taskNode.getAction() : null;
	}

	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/*
	 * Add the event listeners for the Combo Box and the Expression Text
	 */
	private void hookListeners() {
		this.classNameCombo.addSelectionListener(this);
		this.removeButton.addSelectionListener(this);
		this.argumentsTable.addSelectionListener(this);
	}

	/*
	 * Remove the event listeners for the Combo Box and the Expression Text
	 */
	private void unhookListeners() {
		this.classNameCombo.removeSelectionListener(this);
		this.removeButton.removeSelectionListener(this);
		this.argumentsTable.removeSelectionListener(this);
	}

	/*
	 * Update the data being displayed in the controls.
	 */
	private void updateControls() {
		Action action = taskNode.getAction();
		if (action != null) {
			if (action.getClassName() != null) {
				classNameCombo.setText(action.getClassName());
			} else {
				classNameCombo.setText("");
			}
			GenericElement[] genericElements = action.getGenericElements();
			arguments = new LinkedHashMap<String, String>();
			for (GenericElement genericElement : genericElements) {
				arguments.put(genericElement.getName(), genericElement.getValue());
			}

			List<ActionVariableConfiguration> args = this.getArgumentNames(action.getClassName());
			if (args != null && args.size() > 0) {
				argumentsTable.removeAll();
				for (GenericElement genericElement : genericElements) {
					new ArgumentTableItemWrapper(this, argumentsTable, genericElement, false);
				}
				
				setAddButtonEnable(false);
				setRemoveButtonEnable(false);
			} else {
				argumentsTable.removeAll();
				for (GenericElement genericElement : genericElements) {
					new ArgumentTableItemWrapper(this, argumentsTable, genericElement, true);
				}
		
				setAddButtonEnable(true);
			}
		}
		else {
			action = createAction();
			taskNode.setAction(action);
		}
	}
	
	/**
	 * 		HACK method to mark the template as dirty when the user change the GenericElement "name" property. 
	 * 		For the GenericElement "value" property this method is not required, since the call to setValue already marks the template as dirty.  
	 */
	public void updateGenericElement() {
		GenericElement[] elements = taskNode.getAction().getGenericElements();
		
		for (GenericElement element : elements) {
			taskNode.getAction().removeGenericElement(element);
		}
		for (GenericElement element : elements) {
			taskNode.getAction().addGenericElement(element);
		}
		
	}

	/*
	 * Clear all data displayed in the controls and hide the controls if required.
	 */
	private void clearControls() {
		argumentsTable.removeAll();
		addButton.setEnabled(false);
		removeButton.setEnabled(false);
	}

	/* Refresh the Arguments Table, displaying the arguments */
	private void refreshTable(boolean isEditable) {
		argumentsTable.removeAll();
		removeAllGenericElements();
		Set<String> keys = this.arguments.keySet();
		if (keys != null)
			for (String key : keys) {
				GenericElement genericElement = createGenericElement();
				genericElement.setName(key);
				genericElement.setValue(arguments.get(key));
				new ArgumentTableItemWrapper(this, argumentsTable, genericElement, isEditable);
				addGenericElement(genericElement);
			}
	}

	/**
	 * Handle the Focus Lost event for the Script Expression text field.
	 * 
	 * @see org.eclipse.swt.events.FocusListener#focusLost(org.eclipse.swt.events.FocusEvent)
	 */
	public void focusLost(FocusEvent e) {
		if (e.widget == this.classNameCombo) {
			Action action = this.getAction();
			if (action != null) {
				if (!this.isConfiguredClassName(action.getClassName())) {				
					setAddButtonEnable(true);
				}
			}
			updateClassNameText();
		}
	}

	/**
	 * The Focus Gained event does nothing.
	 * 
	 * @see org.eclipse.swt.events.FocusListener#focusGained(org.eclipse.swt.events.FocusEvent)
	 */
	public void focusGained(FocusEvent e) {
	}

	/*
	 * Set the Action Handler Class
	 */
	private void updateClassNameText() {
		Action action = this.getAction();
		if (action != null)
			action.setClassName(this.classNameCombo.getText());
	}

	// ========================================================================
	// ===================== ACTION TYPE COMBO EVENT HANDLING
	// ========================================================================

	/**
	 * Listens for changes in the selection of the Action Type Combo Box
	 * 
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) {
	}

	/**
	 * Listens for the selection of items in the combo box.
	 * 
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) {
		if (e.widget == this.classNameCombo) {
			handleClassNameComboSelected();
		}
		else if (e.widget == addButton) {
			handleAddButtonSelected();
		}
		else if (e.widget == removeButton) {
			handleRemoveButtonSelected();
		}
		else if (e.widget == argumentsTable) {

		}
	}

	/**
	 * Enables/Disables the remove button based on the value of the parameter.
	 * 
	 * @param enable
	 *            the boolean indicating whether to enable the remove button or not
	 */
	public void setRemoveButtonEnable(boolean enable) {
		this.removeButton.setEnabled(enable);
	}

	/**
	 * Enables/Disables the Add button based on the value of the parameter.
	 * 
	 * @param enable
	 *            the boolean indicating whether to enable the Add button or not
	 */
	public void setAddButtonEnable(boolean enable) {
		this.addButton.setEnabled(enable);
	}

	/*
	 * Handle the Class Name combo box selection
	 */
	private void handleClassNameComboSelected() {
		String selection = this.classNameCombo.getText();
		Action action = this.getAction();
		if (this.getAction() != null) {
			action.setClassName(selection);
		}
		else {
			action = createAction();
			taskNode.setAction(action);
			this.getAction().setClassName(selection);
		}
		arguments = new LinkedHashMap<String, String>();
		List<ActionVariableConfiguration> args = this.getArgumentNames(selection);

		if (this.isConfiguredClassName(selection)) {
			for (ActionVariableConfiguration argument : args) {
				arguments.put(argument.getName(), "");
			}
			refreshTable(false);
			setAddButtonEnable(false);
			setRemoveButtonEnable(false);
		}
		else {
			refreshTable(true);
			setAddButtonEnable(true);
		}
	}

	/*
	 * Initializes the Argument Table Columns
	 */
	private void initializeTableColumns() {
		TableLayout layout = (TableLayout) argumentsTable.getLayout();
		TableColumn nameColumn = new TableColumn(argumentsTable, SWT.LEFT);
		nameColumn.setText(COLUMN_TITLES[0]);
		nameColumn.pack();
		ColumnWeightData nameColumnData = new ColumnWeightData(5);
		layout.addColumnData(nameColumnData);
		TableColumn valueColumn = new TableColumn(argumentsTable, SWT.LEFT);
		valueColumn.setText(COLUMN_TITLES[1]);
		valueColumn.setAlignment(SWT.CENTER);
		valueColumn.pack();
		ColumnWeightData valueColumnData = new ColumnWeightData(5);
		layout.addColumnData(valueColumnData);
	}

	/**
	 * Initializes the table for Arguments
	 */
	private void initializeArgumentsTable() {
		argumentsTable.setHeaderVisible(true);
		argumentsTable.setLinesVisible(true);
		argumentsTable.setLayout(new AutoResizeTableLayout(argumentsTable));
		this.initializeTableColumns();
		argumentsTable.pack();
	}

	/*
	 * Creates layout data for the Class label.
	 * 
	 * @return
	 */
	private FormData createClassLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.top = new FormAttachment(0, 0);
		return data;
	}

	/*
	 * Creates layout data for the Argument label.
	 * 
	 * @return
	 */
	private FormData createArgumentLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.top = new FormAttachment(classNameLabel, 0);
		return data;
	}

	/*
	 * Creates layout data for the Class Name Combo Box.
	 * 
	 * @return
	 */
	private FormData createClassNameLayoutData(int x, int y, int z) {
		FormData data = new FormData();
		data.left = new FormAttachment(x, y);
		data.top = new FormAttachment(0, 0);
		data.width = z;
		return data;
	}

	/*
	 * Creates layout data for the Argumetns Table.
	 * 
	 * @return
	 */
	private FormData createArgumentsTableLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 105);
		data.top = new FormAttachment(classNameCombo, 2);
		data.width = 400;
		data.height = 100;
		return data;
	}

	/*
	 * Creates layout data for the Add Button.
	 * 
	 * @return
	 */
	private FormData createAddButtonLayoutData() {

		FormData result = new FormData();
		result.top = new FormAttachment(argumentsTable, 0);
		result.top.alignment = SWT.TOP;
		result.left = new FormAttachment(removeButton, 0);
		result.left.alignment = SWT.LEFT;
		result.right = new FormAttachment(75, -5);
		return result;
	}

	/*
	 * Creates layout data for the Remove Button.
	 * 
	 * @return
	 */
	private FormData createRemoveButtonLayoutData() {
		FormData result = new FormData();
		result.top = new FormAttachment(addButton, 5);
		result.left = new FormAttachment(argumentsTable, 5);
		result.right = new FormAttachment(75, -5);
		return result;

	}

	/*
	 * Handles when Add button is clicked
	 */
	private void handleAddButtonSelected() {
		int counter = 1;
		String name = "Name" + counter;
		while (this.arguments.containsKey(name))
			name = "Name" + (++counter);
		String value = "Value" + counter;
		GenericElement genericElement = createGenericElement();
		genericElement.setName(name);
		genericElement.setValue(value);
		arguments.put(name, value);
		ArgumentTableItemWrapper wrapper = new ArgumentTableItemWrapper(this, argumentsTable,
				genericElement, true);
		argumentsTable.setSelection(wrapper.getTableItem());
		removeButton.setEnabled(argumentsTable.getSelectionIndex() != -1);
		addGenericElement(genericElement);
	}

	/*
	 * Handles when Remove button is clicked
	 */
	private void handleRemoveButtonSelected() {
		TableItem item = argumentsTable.getItem(argumentsTable.getSelectionIndex());
		ArgumentTableItemWrapper wrapper = (ArgumentTableItemWrapper) item.getData();
		arguments.remove(wrapper.getArgumentName());
		removeGenericElement(wrapper.getArgumentName(), wrapper.getArgumentValue());
		refreshTable(true);
		removeButton.setEnabled(argumentsTable.getSelectionIndex() != -1);
	}

	/*
	 * Gets Class Names
	 */
	private String[] getConfiguredClassNames() {
		ActionConfigurationManager manager = ActionConfigurationManager.getInstance();
		List<ActionClassConfiguration> classes = manager.getAllActionClasses();
		int classCount = (classes != null) ? classes.size() : 0;

		String[] classNames = new String[classCount];
		for (int i = 0; i < classCount; i++)
			classNames[i] = classes.get(i).getClassName();

		return classNames;
	}

	/* Checks if the Class Name is a configured class */
	private boolean isConfiguredClassName(String className) {
		return (ActionConfigurationManager.getInstance().getActionClass(className) != null);
	}

	/*
	 * Gets Argument Names for a class name
	 */
	private List<ActionVariableConfiguration> getArgumentNames(String className) {
		ActionConfigurationManager manager = ActionConfigurationManager.getInstance();
		ActionClassConfiguration actionClass = manager.getActionClass(className);
		if (actionClass != null) {
			return actionClass.getVariables();
		}

		return null;
	}

	/*
	 * Adds a Generic Element to the Action
	 */
	private void addGenericElement(GenericElement genericElement) {
		Action action = this.getAction();
		if (action != null) {
			action.addGenericElement(genericElement);
		}
	}

	/*
	 * Removes a Generic Element from the Action
	 */
	private void removeGenericElement(String argName, String argValue) {
		Action action = this.getAction();
		if (action != null) {
			GenericElement[] genericElements = action.getGenericElements();
			for (GenericElement genericElement : genericElements) {
				if (argName != null && argName.equals(genericElement.getName()))
					action.removeGenericElement(genericElement);
			}
		}
	}

	/*
	 * Removes all Generic Elements from the Action
	 */
	private void removeAllGenericElements() {
		Action action = this.getAction();
		if (action != null) {
			GenericElement[] genericElements = action.getGenericElements();
			for (GenericElement genericElement : genericElements) {
				action.removeGenericElement(genericElement);
			}
		}
	}

	/*
	 * Creates a new instance of GenericElement
	 * 
	 * @return a new instance of GenericElement
	 */
	private GenericElement createGenericElement() {
		return (GenericElement) this.taskNode.getFactory().createById(
				SemanticElementConstants.GENERIC_ELEMENT_SEID);
	}

	/*
	 * Creates a new instance of Action
	 * 
	 * @return a new instance of Action
	 */
	private Action createAction() {
		return (Action) this.taskNode.getFactory().createById(SemanticElementConstants.ACTION_SEID);
	}
	
}
