/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.properties;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.InvalidPropertiesFormatException;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.eclipse.jface.bindings.keys.KeyStroke;
import org.eclipse.jface.bindings.keys.ParseException;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.fieldassist.ContentProposalAdapter;
import org.eclipse.jface.fieldassist.IContentProposal;
import org.eclipse.jface.fieldassist.IContentProposalListener;
import org.eclipse.jface.fieldassist.SimpleContentProposalProvider;
import org.eclipse.jface.fieldassist.TextContentAdapter;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.jbpm.gd.common.notation.AbstractNotationElement;
import org.jbpm.gd.common.part.NotationElementGraphicalEditPart;
import org.jbpm.gd.common.part.OutlineEditPart;
import org.jbpm.gd.common.properties.AbstractPropertySection;
import org.jbpm.gd.jpdl.model.Condition;
import org.jbpm.gd.jpdl.model.NodeElementContainer;
import org.jbpm.gd.jpdl.model.Transition;

import com.tandbergtv.watchpoint.studio.ui.UIException;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.util.ValidationMessages;
import com.tandbergtv.watchpoint.studio.validation.IValidationService;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationServiceFactory;

public class WPConditionSection extends AbstractPropertySection implements FocusListener,
		KeyListener, IContentProposalListener {
	private static final Logger logger = Logger.getLogger(WPConditionSection.class);

	private Label label;

	private Text expressionText;

	private Transition transition;

	private ContentProposalAdapter adapter;

	private String keyPressedValue = "";

	public void createControls(Composite parent, TabbedPropertySheetPage aTabbedPropertySheetPage) {
		super.createControls(parent, aTabbedPropertySheetPage);
		Composite clientArea = getWidgetFactory().createFlatFormComposite(parent);
		label = getWidgetFactory().createLabel(clientArea, "Condition");
		expressionText = getWidgetFactory().createText(clientArea, "");
		label.setLayoutData(createLabelLayoutData());
		expressionText.setLayoutData(createExpressionTextLayoutData());
		adapter = createAdapter(expressionText, getTextProposals(), true);
		hookListeners();
		refresh();
	}

	private void hookListeners() {
		expressionText.addFocusListener(this);
		expressionText.addKeyListener(this);
		adapter.addContentProposalListener(this);
	}

	private FormData createLabelLayoutData() {
		FormData result = new FormData();
		result.top = new FormAttachment(10, 0);
		result.left = new FormAttachment(0, 0);
		return result;
	}

	private FormData createExpressionTextLayoutData() {
		FormData result = new FormData();
		result.top = new FormAttachment(10, 0);
		result.left = new FormAttachment(label, 0);
		result.right = new FormAttachment(100, 0);
		return result;
	}

	/**
	 * @see org.jbpm.gd.common.properties.AbstractPropertySection#setInput(org.eclipse.ui.IWorkbenchPart,
	 *      org.eclipse.jface.viewers.ISelection)
	 */
	public void setInput(IWorkbenchPart part, ISelection selection) {
		super.setInput(part, selection);
		if (!(selection instanceof IStructuredSelection))
			return;
		Object input = ((IStructuredSelection) selection).getFirstElement();
		if (input instanceof NotationElementGraphicalEditPart) {
			AbstractNotationElement notationElement = ((NotationElementGraphicalEditPart) input).getNotationElement();
			input = notationElement.getSemanticElement();
		}
		else if (input instanceof OutlineEditPart) {
			input = ((OutlineEditPart) input).getModel();
		}
		if (input instanceof Transition) {
			transition = (Transition) input;
			refresh();
		}
	}

	/**
	 * @see org.eclipse.ui.views.properties.tabbed.AbstractPropertySection#refresh()
	 */
	public void refresh() {
		/* refresh() is called from createControls(), and setInput() hasn't been called */
		if (transition == null)
			return;
		
		Condition condition = transition.getCondition();
		if (condition != null && condition.getExpression() != null)
			expressionText.setText(condition.getExpression());
		else
			expressionText.setText("");
	}

	public boolean shouldUseExtraSpace() {
		return true;
	}

	/**
	 * @see org.eclipse.swt.events.FocusListener#focusGained(org.eclipse.swt.events.FocusEvent)
	 */
	public void focusGained(FocusEvent e) {
		adapter.setEnabled(true);
	}

	/**
	 * @see org.eclipse.swt.events.FocusListener#focusLost(org.eclipse.swt.events.FocusEvent)
	 */
	public void focusLost(FocusEvent e) {
		if (e.widget == expressionText) {
			handleExpressionTextChanged();
		}
		if (Utility.getNodeElementContainer(this.getPart()) != null)
			validate();
	}

	private void handleExpressionTextChanged() {
		if (expressionText.getText().trim().length() == 0)
			this.transition.setCondition(null);
		else {
			if (transition.getCondition() == null) {
				Condition condition = (Condition) transition.getFactory().createById(
						SemanticElementConstants.CONDITION_SEID);
				transition.setCondition(condition);
			}
			transition.getCondition().setExpression(expressionText.getText());
		}
	}

	/*
	 * Validates the Condition Expression
	 */
	private void validate() {
		ValidationServiceFactory serviceFactory = ValidationServiceFactory.createFactory();
		IValidationService validationService = serviceFactory.createValidationService();
		NodeElementContainer container = Utility.getNodeElementContainer(this.getPart());
		List<ValidationMessage> messages = null;
		if (container instanceof WorkflowTemplate)
			messages = validationService.validateCondition(transition.getSource(), transition,
					(WorkflowTemplate) container);
		else if (container instanceof NodeDefinition)
			messages = validationService.validateCondition(transition.getSource(), transition,
					(NodeDefinition) container);

		if (messages != null && messages.size() > 0) {

			try {
				String msg = ValidationMessages.getInstance().getMessage(messages.get(0));
				MessageDialog.openError(Display.getCurrent().getActiveShell(),
						"Error in Condition Expression", msg);
			}
			catch (InvalidPropertiesFormatException e1) {
				logger.error("Error occured while opening the Message Dialog", e1);
				throw new UIException("Error occured while opening the Message Dialog");
			}
			catch (IOException e1) {
				logger.error("Error occured while opening the Message Dialog", e1);
				throw new UIException("Error occured while opening the Message Dialog");
			}

		}
	}

	/*
	 * creates an adpater for ContentAssist
	 */
	private ContentProposalAdapter createAdapter(Text text, String[] proposals, boolean propogate) {
		char[] autoActivationCharacters = new char[] {};
		KeyStroke keyStroke = null;
		try {
			keyStroke = KeyStroke.getInstance("Ctrl+Space");
		}
		catch (ParseException e1) {
			logger.error("Error occured while Creating instance of KeyStroke", e1);
			throw new UIException("Error occured while Creating instance of KeyStroke");

		}
		SimpleContentProposalProvider scp = new SimpleContentProposalProvider(proposals);
		ContentProposalAdapter adapter = new ContentProposalAdapter(text, new TextContentAdapter(),
				scp, keyStroke, autoActivationCharacters);
		adapter.setProposalAcceptanceStyle(ContentProposalAdapter.PROPOSAL_INSERT);
		adapter.setFilterStyle(ContentProposalAdapter.FILTER_CUMULATIVE);
		adapter.setPropagateKeys(propogate);
		adapter.setEnabled(false);
		return adapter;
	}

	/*
	 * Gets the Array of Proposals
	 */
	private String[] getTextProposals() {
		List<WPVariable> variables = Utility.getVariables(Utility.getNodeElementContainer(this.getPart()));
		Set<String> variableNames = new HashSet<String>();
		for (WPVariable variable : variables) {
			variableNames.add(variable.getName());
		}
		String proposals[] = variableNames.toArray(new String[variableNames.size()]);

		return proposals;
	}

	/**
	 * @see org.eclipse.swt.events.KeyListener#keyPressed(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyPressed(KeyEvent arg0) {
		if (arg0.character == 0) {
			keyPressedValue = getFilter();
		}
		else if (arg0.character == 8 && keyPressedValue.length() > 0) {
			keyPressedValue = keyPressedValue.substring(0, keyPressedValue.length() - 1);
		}
		else if (Character.isLetterOrDigit(arg0.character)) {
			keyPressedValue = keyPressedValue + arg0.character;
		}
		SimpleContentProposalProvider scp = new SimpleContentProposalProvider(
				filterProposals(keyPressedValue));
		adapter.setContentProposalProvider(scp);

	}

	/**
	 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyReleased(KeyEvent arg0) {
	}

	/**
	 * @see org.eclipse.jface.fieldassist.IContentProposalListener#proposalAccepted(org.eclipse.jface.fieldassist.IContentProposal)
	 */
	public void proposalAccepted(IContentProposal arg0) {
		String textValue = expressionText.getText();
		String tempValue = keyPressedValue + arg0.getContent();
		int index = textValue.lastIndexOf(tempValue);
		int cursorPosition = expressionText.getCaretPosition();
		StringBuffer newTextValue = new StringBuffer();
		if (index != -1) {
			newTextValue.append(textValue.substring(0, cursorPosition - tempValue.length()));
			newTextValue.append(arg0.getContent());
			newTextValue.append(textValue.substring(cursorPosition));
		}
		else {
			newTextValue.append(textValue);
			newTextValue.append(arg0.getContent());
		}
		expressionText.setText(newTextValue.toString());
		expressionText.setSelection(cursorPosition);
		keyPressedValue = "";

	}

	/*
	 * Filters the proposals based on user input
	 */
	private String[] filterProposals(String filter) {
		java.util.List<String> filterProposals = new ArrayList<String>();
		for (String proposal : getTextProposals()) {
			if (proposal.startsWith(filter)) {
				filterProposals.add(proposal);
			}
		}
		return filterProposals.toArray(new String[filterProposals.size()]);
	}

	/*
	 * Gets the Value for filtering the proposals
	 */
	private String getFilter() {
		String str = expressionText.getText();
		int cursorPosition = expressionText.getCaretPosition();
		StringBuffer filterString = new StringBuffer();
		for (int i = cursorPosition; i > 0; i--) {
			char c = str.charAt(i - 1);
			if (c != ' ' && c != '|' && c != '&' && c != '=' && c != '(') {
				filterString.insert(0, c);
			}
			else {
				break;
			}
		}
		return filterString.toString();
	}
}
