/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.properties;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.InvalidPropertiesFormatException;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.eclipse.jface.bindings.keys.KeyStroke;
import org.eclipse.jface.bindings.keys.ParseException;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.fieldassist.ContentProposalAdapter;
import org.eclipse.jface.fieldassist.IContentProposal;
import org.eclipse.jface.fieldassist.IContentProposalListener;
import org.eclipse.jface.fieldassist.SimpleContentProposalProvider;
import org.eclipse.jface.fieldassist.TextContentAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.views.properties.tabbed.AbstractPropertySection;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.jpdl.model.Decision;
import org.jbpm.gd.jpdl.model.NodeElementContainer;

import com.tandbergtv.watchpoint.studio.ui.UIException;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.util.ValidationMessages;
import com.tandbergtv.watchpoint.studio.validation.IValidationService;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationServiceFactory;

public class WPDecisionConfigurationComposite implements FocusListener,
		KeyListener, IContentProposalListener {
	private static final String EXPRESSION_LABEL = "Expression";
	private static final Logger logger = Logger.getLogger(WPDecisionConfigurationComposite.class);

	public static WPDecisionConfigurationComposite create(AbstractPropertySection parentSection, 
			TabbedPropertySheetWidgetFactory widgetFactory, Composite parent) {
		WPDecisionConfigurationComposite result = new WPDecisionConfigurationComposite();
		result.parentPropertySection = parentSection;
		result.widgetFactory = widgetFactory;
		result.parent = parent;
		result.create();
		return result;
	}

	private TabbedPropertySheetWidgetFactory widgetFactory;
	private Composite parent;
	private Decision decision;

	private Composite expressionComposite;
	private Label expressionLabel;
	private Text expressionText;
	private String keyPressedValue = "";

	private ContentProposalAdapter adapter;

	private AbstractPropertySection parentPropertySection;

	public void setDecision(Decision decision) {
		if (this.decision == decision)
			return;
		unhookListeners();
		clearControls();
		this.decision = decision;
		if (decision != null) {
			updateControls();
			hookListeners();
		}
	}

	private void hookListeners() {
		expressionText.addFocusListener(this);
		expressionText.addKeyListener(this);
		adapter.addContentProposalListener(this);
	}

	private void unhookListeners() {
		expressionText.removeFocusListener(this);
		expressionText.removeKeyListener(this);
		adapter.removeContentProposalListener(this);
	}

	private void clearControls() {
		expressionText.setText("");
	}

	private void updateControls() {
		if (decision.getExpression() == null)
			expressionText.setText("");
		else
			expressionText.setText(decision.getExpression());
	}

	private void create() {
		createExpression();
		adapter = createAdapter(expressionText, getTextProposals(), true);
		initializeLayouts();
	}

	private void createExpression() {
		expressionComposite = widgetFactory.createFlatFormComposite(parent);
		expressionLabel = widgetFactory.createLabel(expressionComposite, EXPRESSION_LABEL);
		expressionText = widgetFactory.createText(expressionComposite, "");
	}

	private void initializeLayouts() {
		expressionComposite.setLayoutData(createExpressionCompositeLayoutData());
		expressionLabel.setLayoutData(createExpressionLabelLayoutData());
		expressionText.setLayoutData(createExpressionTextLayoutData());
	}

	private FormData createExpressionLabelLayoutData() {
		FormData result = new FormData();
		result.top = new FormAttachment(0, 0);
		result.left = new FormAttachment(0, 0);
		return result;
	}

	private FormData createExpressionTextLayoutData() {
		FormData result = new FormData();
		result.top = new FormAttachment(0, 0);
		result.left = new FormAttachment(expressionLabel, 0);
		result.right = new FormAttachment(100, 0);
		return result;
	}

	private FormData createExpressionCompositeLayoutData() {
		FormData result = new FormData();
		result.top = new FormAttachment(10, 0);
		result.left = new FormAttachment(0, 0);
		result.right = new FormAttachment(100, 0);
		return result;
	}

	private void updateExpressionText() {
		if(expressionText.getText().trim().length() == 0){
			decision.setExpression(null);
			return;			
		}
		decision.setExpression(expressionText.getText());
	}

	public void focusLost(FocusEvent e) {
		if (e.widget == expressionText) {
			updateExpressionText();
			if(Utility.getNodeElementContainer(this.parentPropertySection.getPart())!=null)
				validate();
		}

	}

	public void focusGained(FocusEvent e) {
		adapter.setEnabled(true);

	}

	/*
	 * Creates an adapter for Content Assist
	 */
	private ContentProposalAdapter createAdapter(Text text, String[] proposals, boolean propogate) {
		char[] autoActivationCharacters = new char[] {};
		KeyStroke keyStroke = null;
		try {
			keyStroke = KeyStroke.getInstance("Ctrl+Space");
		} catch (ParseException e1) {
			logger.error("Error occured while Creating instance of KeyStroke", e1);
			throw new UIException("Error occured while Creating instance of KeyStroke");
		}
		SimpleContentProposalProvider scp = new SimpleContentProposalProvider(proposals);
		ContentProposalAdapter adapter = new ContentProposalAdapter(text, new TextContentAdapter(),
				scp, keyStroke, autoActivationCharacters);
		adapter.setProposalAcceptanceStyle(ContentProposalAdapter.PROPOSAL_INSERT);
		adapter.setFilterStyle(ContentProposalAdapter.FILTER_CUMULATIVE);
		adapter.setPropagateKeys(propogate);
		adapter.setEnabled(false);
		return adapter;
	}

	/*
	 * Gets the proposals
	 */
	private String[] getTextProposals() {
		List<WPVariable> variables = Utility.getVariables(Utility.getNodeElementContainer(this.parentPropertySection.getPart()));
		Set<String> variableNames = new HashSet<String>();
		for (WPVariable variable : variables) {
			String name = variable.getName();
			if (name != null && name.trim().length() > 0)
				variableNames.add(variable.getName());
		}
		String proposals[] = variableNames.toArray(new String[variableNames.size()]);

		return proposals;
	}

	/*
	 * Validates the Decision Expression
	 */
	private void validate() {
		ValidationServiceFactory serviceFactory = ValidationServiceFactory.createFactory();
		IValidationService validationService = serviceFactory.createValidationService();
		NodeElementContainer container = Utility.getNodeElementContainer(this.parentPropertySection.getPart());
		List<ValidationMessage> messages = null;
		if (container instanceof WorkflowTemplate)
			messages = validationService.validateDecisionExpression(decision,
					(WorkflowTemplate) container);
		else if (container instanceof NodeDefinition)
			messages = validationService.validateDecisionExpression(decision,
					(NodeDefinition) container);
			

		if (messages != null && messages.size() > 0) {

			try {
				String msg = ValidationMessages.getInstance().getMessage(messages.get(0));
				MessageDialog.openError(Display.getCurrent().getActiveShell(),
						"Error in Decision Expression", msg);

			} catch (InvalidPropertiesFormatException e1) {
				logger.error("Error occured while opening the Message Dialog", e1);
				throw new UIException("Error occured while opening the Message Dialog");
			} catch (IOException e1) {
				logger.error("Error occured while opening the Message Dialog", e1);
				throw new UIException("Error occured while opening the Message Dialog");
			}

		}
	}


	/**
	 * @see org.eclipse.swt.events.KeyListener#keyPressed(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyPressed(KeyEvent arg0) {
		if (arg0.character == 0) {
			keyPressedValue = getFilter();
		}else if(arg0.character == 8 && keyPressedValue.length()>0){
			keyPressedValue = keyPressedValue.substring(0, keyPressedValue.length()-1);
		}else if (Character.isLetterOrDigit(arg0.character)){
			keyPressedValue= keyPressedValue + arg0.character;
		}
		SimpleContentProposalProvider scp = new SimpleContentProposalProvider(
				filterProposals(keyPressedValue));
		adapter.setContentProposalProvider(scp);
	}

	/**
	 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyReleased(KeyEvent arg0) {
	}

	/**
	 * @see org.eclipse.jface.fieldassist.IContentProposalListener#proposalAccepted(org.eclipse.jface.fieldassist.IContentProposal)
	 */
	public void proposalAccepted(IContentProposal arg0) {
		String textValue = expressionText.getText();
		String tempValue = keyPressedValue + arg0.getContent();
		int index = textValue.lastIndexOf(tempValue);
		int cursorPosition = expressionText.getCaretPosition();
		StringBuffer newTextValue =  new StringBuffer();
		if (index != -1) {
			newTextValue.append(textValue.substring(0, cursorPosition-tempValue.length()));
			newTextValue.append(arg0.getContent());
			newTextValue.append(textValue.substring(cursorPosition));
		} else {
			newTextValue.append(textValue);
			newTextValue.append(arg0.getContent());
		}
		expressionText.setText(newTextValue.toString());
		expressionText.setSelection(cursorPosition);
		keyPressedValue = "";
	}

	/*
	 * Filters the proposals based on user input
	 */
	private String[] filterProposals(String filter) {
		java.util.List<String> filterProposals = new ArrayList<String>();
		for (String proposal : getTextProposals()) {
			if (proposal.startsWith(filter)) {
				filterProposals.add(proposal);
			}
		}
		return filterProposals.toArray(new String[filterProposals.size()]);
	}

	/*
	 * Gets the Value for filtering the proposals
	 */
	private String getFilter() {
		String str = expressionText.getText();
		int cursorPosition = expressionText.getCaretPosition();
		StringBuffer filterString =  new StringBuffer();
		for (int i = cursorPosition; i > 0; i--) {
			char c = str.charAt(i - 1);
			if (c != ' ' && c != '|' && c != '&' && c != '=' && c != '(') {
				filterString.insert(0,c);
			} else {
				break;
			}
		}
		return filterString.toString();
	}
}
