package com.tandbergtv.watchpoint.studio.ui.properties;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.draw2d.ColorConstants;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.views.properties.tabbed.AbstractPropertySection;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.common.model.GenericElement;
import org.jbpm.gd.common.model.SemanticElementFactory;
import org.jbpm.gd.jpdl.model.Action;
import org.jbpm.gd.jpdl.model.ActionElement;
import org.jbpm.gd.jpdl.model.ExceptionHandler;
import org.jbpm.gd.jpdl.model.ExceptionHandlerContainer;
import org.jbpm.gd.jpdl.util.AutoResizeTableLayout;

import com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer;
import com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants;
import com.tandbergtv.watchpoint.studio.ui.properties.template.ExceptionArgumentTableItemWrapper;

public class WPExceptionHandlerContainerComposite implements SelectionListener, FocusListener {
	
	private Composite parent;

	private TabbedPropertySheetWidgetFactory widgetFactory;

	private Label classNameLabel, argumentsLabel;

	private CCombo classNameCombo; 

	private Table argumentsTable;
	
	private ExceptionHandlerContainer exceptionHandlerContainer;
	
	private ExceptionHandler exceptionHandler;
	
	private AbstractPropertySection parentPropertySection;

	private Map<String, GenericElement> arguments;
	
	private static final String CLASS_NAME_LABEL = "Handler: ";

	private static final String ARGUMENTS_LABEL = "Arguments: ";

	private static final String[] COLUMN_TITLES = new String[] { "Name", "Value" };
	
	private static final String ROLLBACK = "Rollback";
	
	// ========================================================================
	// ===================== CONSTRUCTOR
	// ========================================================================

	/**
	 * Creates a new ActionConfigurationComposite and initializes it.
	 * 
	 * @param parentPropertySection 
	 *            The parent property section
	 * @param parent
	 *            a widget which will be the parent of the new instance (cannot be null)
	 * @param widgetFactory
	 *            The Factory to use to create widgets / child composites for this composite.
	 */
	public WPExceptionHandlerContainerComposite(AbstractPropertySection parentPropertySection, Composite parent,
			TabbedPropertySheetWidgetFactory widgetFactory) {
		this.parentPropertySection = parentPropertySection;
		this.parent = parent;
		this.widgetFactory = widgetFactory;
		
		this.createControls();
	}

	// ========================================================================
	// ===================== CREATE AND INITIALIZE CONTROLS
	// ========================================================================

	/*
	 * Create and initialize the controls.
	 */
	private void createControls() {
		this.arguments = new LinkedHashMap<String, GenericElement>();
		this.create();
		// Initialize the Layout for the controls
		this.initializeArgumentsTable();
	}

	/*
	 * Creates the widgets for this composite.
	 * 
	 */
	private void create() {
		Composite composite = widgetFactory.createFlatFormComposite(parent);
		classNameLabel = new Label(composite, SWT.NONE);
		classNameLabel.setText(CLASS_NAME_LABEL);
		classNameLabel.setLayoutData(createClassLabelLayoutData());
		classNameLabel.setBackground(ColorConstants.white);
		classNameLabel.pack();

		classNameCombo = widgetFactory.createCCombo(parent, SWT.DROP_DOWN);
		classNameCombo.setLayoutData(createClassNameLayoutData(0, 105, 320));
		String[] item = new String[6];
		item[0]="Fail";
		item[1]="Ignore";
		item[2]="Email";
		item[3]="Retry";
		item[4]="Catch";
		item[5]=ROLLBACK;
		
		classNameCombo.setItems(item);
		classNameCombo.setEditable(true);
		classNameCombo.addFocusListener(this);
		classNameCombo.pack();
		
		argumentsLabel = new Label(composite, SWT.NONE);
		argumentsLabel.setText(ARGUMENTS_LABEL);
		argumentsLabel.setLayoutData(createArgumentLabelLayoutData());
		argumentsLabel.setBackground(ColorConstants.white);
		argumentsLabel.pack();
		argumentsTable = widgetFactory.createTable(parent, SWT.V_SCROLL | SWT.H_SCROLL
				| SWT.FULL_SELECTION);
		argumentsTable.setLayoutData(createArgumentsTableLayoutData());
		argumentsTable.addSelectionListener(this);
		argumentsTable.pack();

	}
	
	public void setExceptionHandlerContainer(ExceptionHandlerContainer exceptionHandlerContainer) {
		if (this.exceptionHandlerContainer == exceptionHandlerContainer) 
			return;
		unhookSelectionListeners();
		clearControls();
		this.exceptionHandlerContainer = exceptionHandlerContainer;

		if (exceptionHandlerContainer != null) {
			ExceptionHandler[] exceptionHandlers = exceptionHandlerContainer.getExceptionHandlers();
			if (exceptionHandlers.length >0 && exceptionHandlers[0] instanceof ExceptionHandler)
				exceptionHandler = (ExceptionHandler)exceptionHandlers[0];
			else {
				SemanticElementFactory factory = this.exceptionHandlerContainer.getFactory();
				exceptionHandler = (ExceptionHandler)factory.createById("org.jbpm.gd.jpdl.exceptionHandler");
				exceptionHandler.setExceptionClass("java.lang.Exception");
				this.exceptionHandlerContainer.addExceptionHandler(exceptionHandler);
			}
			
			updateControls();
			hookSelectionListeners();
		}		
	}  		
	
	private void hookSelectionListeners() {
		this.classNameCombo.addSelectionListener(this);
		this.argumentsTable.addSelectionListener(this);
	}

	/*
	 * Remove the event listeners for the Combo Box and the Expression Text
	 */
	private void unhookSelectionListeners() {
		this.classNameCombo.removeSelectionListener(this);
		this.argumentsTable.removeSelectionListener(this);
	}

	/*
	 * Update the data being displayed in the controls.
	 */
	private void updateControls() {
		Action action = this.getAction();
		if (action != null) {
			if (action.getClassName() != null)
				classNameCombo.setText(action.getClassName());
			
			GenericElement[] genericElements = action.getGenericElements();				
			arguments = new LinkedHashMap<String, GenericElement>();
			for (GenericElement genericElement : genericElements) {
				arguments.put(genericElement.getName(), genericElement);
			}

			List<String> args = this.getArgumentNames(action.getClassName());
			if (args != null && args.size() > 0) {
				argumentsTable.removeAll();

				for (String name : args) {
					GenericElement genericElement = arguments.get(name);
					new ExceptionArgumentTableItemWrapper(this, argumentsTable, genericElement, false);
				}
			}			
		}
		else {
			action = createAction();
			action.setClassName("Fail");
			classNameCombo.setText(action.getClassName());
			exceptionHandler.addAction(action);
		}
	}

	/*
	 * Clear all data displayed in the controls and hide the controls if required.
	 */
	private void clearControls() {
		argumentsTable.removeAll();
	}

	/* Refresh the Arguments Table, displaying the arguments */
	private void refreshTable(boolean isEditable) {
		argumentsTable.removeAll();
		removeAllGenericElements();
		Set<String> keys = this.arguments.keySet();
		if (keys != null)
			for (String key : keys) {
				GenericElement genericElement = arguments.get(key);
				new ExceptionArgumentTableItemWrapper(this, argumentsTable, genericElement, isEditable);
				addGenericElement(genericElement);
			}
	}

	/**
	 * Handle the Focus Lost event for the Script Expression text field.
	 * 
	 * @see org.eclipse.swt.events.FocusListener#focusLost(org.eclipse.swt.events.FocusEvent)
	 */
	public void focusLost(FocusEvent e) {
		if (e.widget == this.classNameCombo) {
			updateClassNameText();
		} 
	}

	/**
	 * The Focus Gained event does nothing.
	 * 
	 * @see org.eclipse.swt.events.FocusListener#focusGained(org.eclipse.swt.events.FocusEvent)
	 */
	public void focusGained(FocusEvent e) {
	}

	/*
	 * Set the Action Handler Class
	 */
	private void updateClassNameText() {
		Action action = this.getAction();
		if (action != null) {
			action.setClassName(this.classNameCombo.getText());			
		}
		
	}

	// ========================================================================
	// ===================== ACTION TYPE COMBO EVENT HANDLING
	// ========================================================================

	/**
	 * Listens for changes in the selection of the Action Type Combo Box
	 * 
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) {
	}

	/**
	 * Listens for the selection of items in the combo box.
	 * 
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) {
		if (e.widget == this.classNameCombo) {
			handleClassNameComboSelected();
		}		
		else if (e.widget == argumentsTable) {

		}
	}
	/*
	 * Handle the Class Name combo box selection
	 */
	private void handleClassNameComboSelected() {
		String selection = this.classNameCombo.getText();

		// Removes the Rollback node if it has been set previously
		if (exceptionHandlerContainer instanceof IMessageContainer) {
			IMessageContainer container = (IMessageContainer) exceptionHandlerContainer;
			container.removeRollbackNode();
		}
		
		Action action = this.getAction();
		if (this.getAction() != null) {
			action.setClassName(selection);
		}
		else {
			action = createAction();
			exceptionHandler.addAction(action);
			this.getAction().setClassName(selection);
		}
		arguments = new LinkedHashMap<String, GenericElement>();

		List<String> args = this.getArgumentNames(selection);
		
		if (args.size() > 0) {
			for (String name : args) {
				GenericElement ge = createGenericElement();
				ge.setName(name);
				ge.setValue("");
				arguments.put(name, ge);				
			}
			refreshTable(false);
		}
		else {
			refreshTable(false);
		}
	}

	/*
	 * Initializes the Argument Table Columns
	 */
	private void initializeTableColumns() {
		TableLayout layout = (TableLayout) argumentsTable.getLayout();
		TableColumn nameColumn = new TableColumn(argumentsTable, SWT.LEFT);
		nameColumn.setText(COLUMN_TITLES[0]);
		nameColumn.pack();
		ColumnWeightData nameColumnData = new ColumnWeightData(5);
		layout.addColumnData(nameColumnData);
		TableColumn valueColumn = new TableColumn(argumentsTable, SWT.LEFT);
		valueColumn.setText(COLUMN_TITLES[1]);
		valueColumn.setAlignment(SWT.CENTER);
		valueColumn.pack();
		ColumnWeightData valueColumnData = new ColumnWeightData(5);
		layout.addColumnData(valueColumnData);
	}

	/**
	 * Initializes the table for Arguments
	 */
	private void initializeArgumentsTable() {
		argumentsTable.setHeaderVisible(true);
		argumentsTable.setLinesVisible(true);
		argumentsTable.setLayout(new AutoResizeTableLayout(argumentsTable));
		this.initializeTableColumns();
		argumentsTable.pack();
	}

	/*
	 * Creates layout data for the Class label.
	 * 
	 * @return
	 */
	private FormData createClassLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.top = new FormAttachment(0, 0);
		return data;
	}

	/*
	 * Creates layout data for the Argument label.
	 * 
	 * @return
	 */
	private FormData createArgumentLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.top = new FormAttachment(classNameLabel, 0);
		return data;
	}

	/*
	 * Creates layout data for the Class Name Combo Box.
	 * 
	 * @return
	 */
	private FormData createClassNameLayoutData(int x, int y, int z) {
		FormData data = new FormData();
		data.left = new FormAttachment(x, y);
		data.top = new FormAttachment(0, 0);
		data.width = z;
		return data;
	}

	/*
	 * Creates layout data for the Argumetns Table.
	 * 
	 * @return
	 */
	private FormData createArgumentsTableLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 105);
		data.top = new FormAttachment(classNameCombo, 2);
		data.width = 400;
		data.height = 100;
		return data;
	}

	private void addGenericElement(GenericElement genericElement) {
		Action action = this.getAction();
		action.addGenericElement(genericElement);
	}

	/*private void removeGenericElement(String argName, String argValue) {
		Action action = this.getAction();
		if (action != null) {
			GenericElement[] genericElements = action.getGenericElements();
			for (GenericElement genericElement : genericElements) {
				if (argName != null && argName.equals(genericElement.getName()))
					action.removeGenericElement(genericElement);
			}
		}
	}*/

	private void removeAllGenericElements() {
		Action action = this.getAction();
		if (action != null) {
			GenericElement[] genericElements = action.getGenericElements();
			for (GenericElement genericElement : genericElements) {
				action.removeGenericElement(genericElement);
			}
		}
	}


	private GenericElement createGenericElement() {
		GenericElement genericElement = (GenericElement) this.exceptionHandlerContainer.getFactory().createById(SemanticElementConstants.GENERIC_ELEMENT_SEID);
		return genericElement;
	}
	
	private Action getAction() {
		ActionElement[] actions = exceptionHandler.getActionElements();
		Action action = null;
		if (actions != null && actions.length > 0) {
			if (actions[0] instanceof Action)
				action = (Action)actions[0];
		}
		return action;
	}
	
	private List<String> getArgumentNames(String actionClass) {
	    List<String> argumentNames = new ArrayList<String>();
	    
		if (actionClass.equals("Retry"))
			argumentNames.add("retry");
		else if (actionClass.equals("Email"))
			argumentNames.add("email");
		else if (actionClass.equals("Catch")) {
			argumentNames.add("errorCodeVariable");
			argumentNames.add("errorMessageVariable");
			argumentNames.add("errorHintVariable");
		}
		
		return argumentNames;
	}
	
	/*
	 * Creates a new instance of Action
	 * 
	 * @return a new instance of Action
	 */
	private Action createAction() {
		return (Action) this.exceptionHandlerContainer.getFactory().createById(SemanticElementConstants.ACTION_SEID);
	}
	
	public AbstractPropertySection getParentPropertySection() {
		return parentPropertySection;
	}
}
