/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.properties.template;

import java.util.Set;

import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.views.properties.tabbed.AbstractPropertySection;
import org.jbpm.gd.jpdl.util.AutoResizeTableLayout;

import com.tandbergtv.watchpoint.studio.ui.model.IProtectionKeyElement;
import com.tandbergtv.watchpoint.studio.ui.model.ProtectionKey;
import com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants;

/**
 * Composite for adding and removing protection keys from a protection key container element.
 * Defines a table in which protection keys can be added and removed.
 * 
 * @author Imran Naqvi
 * 
 */
public class PKContainerConfigurationComposite implements SelectionListener {
	
	/**
	 * Static Create method.
	 * 
	 * @param widgetFactory
	 *            Factory to create widgets
	 * @param parent
	 *            Parent composite in which to paint all controls
	 * @param columnTitles
	 *            titles of the columns to display
	 * @return instance of PKContainerConfigurationComposite
	 */
	public static PKContainerConfigurationComposite create(AbstractPropertySection parentPropertySection, FormToolkit widgetFactory,
			Composite parent, String[] columnTitles) {
		PKContainerConfigurationComposite result = new PKContainerConfigurationComposite();
		result.parentPropertySection = parentPropertySection;
		result.widgetFactory = widgetFactory;
		result.parent = parent;
		result.columnTitles = columnTitles;
		result.create();
		return result;
	}

	/**
	 * Static Create method which specifies the column titles.
	 * 
	 * @param widgetFactory
	 *            Factory to create widgets
	 * @param parent
	 *            Parent composite in which to paint all controls
	 * @return instance of PKContainerConfigurationComposite
	 */
	public static PKContainerConfigurationComposite create(AbstractPropertySection parentPropertySection, FormToolkit widgetFactory,
			Composite parent) {
		return create(parentPropertySection, widgetFactory, parent, new String[] { "Name" });
	}

	private FormToolkit widgetFactory;

	private Composite parent;

	private Label pkLabel;

	private Table pkTable;

	private Button addButton, removeButton;

	private IProtectionKeyElement protectionKeyElement;

	private String[] columnTitles = null;
	
	private AbstractPropertySection parentPropertySection;

	private PKContainerConfigurationComposite() {
	}

	/**
	 * Sets the protectionKeyElement whose properties are shown by this control.
	 * 
	 * @param protectionKeyElement
	 */
	public void setProtectionKeyElement(IProtectionKeyElement protectionKeyElement) {
		unhookListeners();
		this.protectionKeyElement = protectionKeyElement;
		clearControls();
		if (protectionKeyElement != null) {
			updateControls();
			hookListeners();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) {
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) {
		if (e.widget == addButton) {
			handleAddButtonSelected();
		}
		else if (e.widget == removeButton) {
			handleRemoveButtonSelected();
		}
		else if (e.widget == pkTable) {
			handlePKTableSelected(e);
		}
	}

	/**
	 * Enables/Disables the remove buttong based on the value of the parameter.
	 * 
	 * @param enable
	 *            the boolean indicating whether to enable the remove button or not
	 */
	public void setRemoveButtonEnable(boolean enable) {
		this.removeButton.setEnabled(enable);
	}

	/*
	 * Adds listeners to all sub controls that are part of this composite.
	 */
	private void hookListeners() {
		addButton.addSelectionListener(this);
		removeButton.addSelectionListener(this);
		pkTable.addSelectionListener(this);
	}

	/*
	 * Removes listeners to all sub controls that are part of this composite.
	 */
	private void unhookListeners() {
		addButton.removeSelectionListener(this);
		removeButton.removeSelectionListener(this);
		pkTable.removeSelectionListener(this);
	}

	/*
	 * Clear data from all controls.
	 */
	private void clearControls() {
		pkTable.removeAll();
		addButton.setEnabled(false);
		removeButton.setEnabled(false);
	}

	/*
	 * Update data in all controls.
	 */
	private void updateControls() {
		addButton.setEnabled(true);
		refreshTable();
	}

	/*
	 * Creates the widgets that are included in this composite.
	 */
	private void create() {
		pkLabel = widgetFactory.createLabel(parent, "Define the protection keys :");
		pkTable = widgetFactory.createTable(parent, SWT.V_SCROLL | SWT.H_SCROLL
				| SWT.FULL_SELECTION);
		addButton = widgetFactory.createButton(parent, "Add", SWT.PUSH);
		removeButton = widgetFactory.createButton(parent, "Remove", SWT.PUSH);
		removeButton.setEnabled(false);
		pkLabel.setLayoutData(createMessageLabelLayoutData());
		pkTable.setLayoutData(createPKTableLayoutData());
		addButton.setLayoutData(createAddButtonLayoutData());
		removeButton.setLayoutData(createRemoveButtonLayoutData());
		initializePKTable();
	}

	/*
	 * Initializes the protection key table.
	 */
	private void initializePKTable() {
		pkTable.setHeaderVisible(true);
		pkTable.setLinesVisible(true);
		pkTable.setLayout(new AutoResizeTableLayout(pkTable));
		initializeTaskTableColumns();
	}

	/*
	 * Initializes the columns for the table of protection keys.
	 */
	private void initializeTaskTableColumns() {
		TableLayout layout = (TableLayout) pkTable.getLayout();
		TableColumn dummyColumn = new TableColumn(pkTable, SWT.CENTER);
		dummyColumn.setText("");
		ColumnWeightData dummyColumnData = new ColumnWeightData(0);
		layout.addColumnData(dummyColumnData);
		TableColumn nameColumn = new TableColumn(pkTable, SWT.LEFT);
		nameColumn.setText(columnTitles[0]);
		ColumnWeightData nameColumnData = new ColumnWeightData(15);
		nameColumnData.resizable = false;
		layout.addColumnData(nameColumnData);
	}

	private FormData createMessageLabelLayoutData() {
		FormData result = new FormData();
		result.top = new FormAttachment(0, 5);
		result.left = new FormAttachment(0, 0);
		result.right = new FormAttachment(100, -5);
		return result;
	}

	private FormData createPKTableLayoutData() {
		FormData result = new FormData();
		result.top = new FormAttachment(pkLabel, 5);
		result.left = new FormAttachment(0, 100);
		result.bottom = new FormAttachment(100, -5);
		result.right = new FormAttachment(removeButton, -5);
		return result;
	}

	private FormData createAddButtonLayoutData() {
		FormData result = new FormData();
		result.top = new FormAttachment(pkTable, 0);
		result.top.alignment = SWT.TOP;
		result.left = new FormAttachment(removeButton, 0);
		result.left.alignment = SWT.LEFT;
		result.right = new FormAttachment(90, -5);
		return result;
	}

	private FormData createRemoveButtonLayoutData() {
		FormData result = new FormData();
		result.top = new FormAttachment(addButton, 5);
		result.right = new FormAttachment(90, -5);
		return result;
	}

	private void handleAddButtonSelected() {
		ProtectionKey key = createProtectionKey();
		protectionKeyElement.addProtectionKey(key);
		ProtectionKeyTableItemWrapper wrapper = new ProtectionKeyTableItemWrapper(this.parentPropertySection, this, pkTable,
				key);
		pkTable.setSelection(wrapper.getTableItem());
		wrapper.editCell(1);
		removeButton.setEnabled(pkTable.getSelectionIndex() != -1);
	}

	private void handleRemoveButtonSelected() {
		TableItem item = pkTable.getItem(pkTable.getSelectionIndex());
		ProtectionKeyTableItemWrapper wrapper = (ProtectionKeyTableItemWrapper) item.getData();
		protectionKeyElement.removeProtectionKey(wrapper.getProtectionKey());
		refreshTable();
		removeButton.setEnabled(pkTable.getSelectionIndex() != -1);
	}

	private void refreshTable() {
		pkTable.removeAll();
		Set<ProtectionKey> keys = protectionKeyElement.getProtectionKeys();
		if (keys != null)
			for (ProtectionKey key : keys) {
				new ProtectionKeyTableItemWrapper(this.parentPropertySection, this, pkTable, key);
			}
	}

	private void handlePKTableSelected(SelectionEvent e) {
		removeButton.setEnabled(pkTable.getSelectionIndex() != -1);
	}

	private ProtectionKey createProtectionKey() {
		return (ProtectionKey) this.protectionKeyElement.getFactory().createById(
				SemanticElementConstants.PROTECTIONKEY_SEID);
	}
}
