/**
 * ResourceTypeUpdateCommand.java
 * Created Apr 23, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.sync.resource;

import static com.tandbergtv.watchpoint.studio.dto.ParameterType.ADAPTOR;
import static com.tandbergtv.watchpoint.studio.dto.ParameterType.HEARTBEAT;
import static com.tandbergtv.watchpoint.studio.dto.ParameterType.INITIALIZATION;

import java.util.Iterator;

import com.tandbergtv.watchpoint.studio.dto.ParameterType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.dto.ResourceTypeParameter;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiff;
import com.tandbergtv.watchpoint.studio.ui.sync.IUpdateElement;
import com.tandbergtv.watchpoint.studio.ui.sync.command.AbstractDiffCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.util.ISynchronizationContext;

/**
 * Updates a resource type
 * 
 * @author Sahil Verma
 */
public class ResourceTypeUpdateCommand extends AbstractDiffCommand {

	protected ResourceType localModel;
	
	protected ResourceType model;
	
	/**
	 * @param name
	 * @param diff
	 * @param context
	 */
	public ResourceTypeUpdateCommand(String name, IDiff diff, ISynchronizationContext context) {
		super(name, diff, context);
		this.model = (ResourceType) diff.getModel();
		this.localModel = (ResourceType) IUpdateElement.class.cast(diff).getLocalModel();
	}

	/**
	 * {@inheritDoc}
	 */
	public void execute() {
		updateProperties();
		updateAdaptor();
		updateManagement();
		updateMessages();
		
		context.updateResourceType(localModel);
	}
	
	/**
	 * Copies basic properties
	 * @param model
	 */
	protected void updateProperties() {
		localModel.setName(model.getName());
		localModel.setVersion(model.getVersion());
		localModel.setDescription(model.getDescription());
		localModel.setConnectionType(model.getConnectionType());
		localModel.setCreateDate(model.getCreateDate());
		localModel.setOutOfTheBox(model.isOutOfTheBox());
	}

	/**
	 * Copies management parameters
	 */
	protected void updateManagement() {
		localModel.setHeartbeatRequired(model.isHeartbeatRequired());
		localModel.setHeartbeatConnectionType(model.getHeartbeatConnectionType());
		localModel.setHeartbeatStrategyClass(model.getHeartbeatStrategyClass());
		
		mergeParameters(HEARTBEAT);
		
		localModel.setInitializationRequired(model.isInitializationRequired());
		localModel.setInitializationStrategyClass(model.getInitializationStrategyClass());
		
		mergeParameters(INITIALIZATION);
	}

	/**
	 * Copies adaptor information
	 */
	protected void updateAdaptor() {
		localModel.setAdaptorType(model.getAdaptorType());
		localModel.setAdaptorClassName(model.getAdaptorClassName());
		localModel.setAdaptorScope(model.getAdaptorScope());
		localModel.setMessageIdentificationClassName(model.getMessageIdentificationClassName());
		
		mergeParameters(ADAPTOR);
	}
	
	protected void updateMessages() {
		localModel.setMessageSequenceNumber(model.getMessageSequenceNumber());
	}
	
	/**
	 * Merges parameters. The application needs to take care of this nonsense explicitly because JPA 
	 * doesn't handle deleted collection items.
	 * 
	 * @param type
	 */
	protected void mergeParameters(ParameterType type) {
		if (localModel.getParameters() == null)
			return;
		
		Iterator<ResourceTypeParameter> i = localModel.getParameters().iterator();
		
		while (i.hasNext()) {
			ResourceTypeParameter parameter = i.next();
			
			if (parameter.getType() != type)
				continue;
			
			if (!model.hasParameter(type, parameter)) {
				parameter.setResourceType(null);
				i.remove();
				context.deleteParameter(parameter);
			}
		}
		
		if (model.getParameters() == null)
			return;
		
		for (ResourceTypeParameter parameter : model.getParameters()) {
			if (parameter.getType() == type)
				localModel.addParameter(type, parameter.getName(), parameter.getValue());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return diff.getElement().getLabel() + " " + model + " " + diff.getKind();
	}
}
