package com.tandbergtv.watchpoint.studio.ui.view.resourcetype.actions;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.ui.IObjectActionDelegate;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.navigator.CommonViewer;
import org.eclipse.ui.part.ViewPart;
import org.eclipse.ui.statushandlers.StatusManager;
import org.jbpm.gd.common.xml.XmlAdapter;
import org.jbpm.gd.jpdl.model.Description;
import org.jbpm.gd.jpdl.model.NodeElement;

import com.tandbergtv.watchpoint.studio.dataaccess.DataAccessFactory;
import com.tandbergtv.watchpoint.studio.dataaccess.DataAccessInterface;
import com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext;
import com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI;
import com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI;
import com.tandbergtv.watchpoint.studio.dataaccess.jpa.JPAPersistenceContextFactory;
import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.TemplateMessageDTO;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.service.INodeDefinitionService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.editor.WatchPointTemplateEditor;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.NodeGroup;
import com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;

public abstract class AbstractNodeDefinitionAction implements IObjectActionDelegate{
	protected ViewPart viewPart;
	protected XmlAdapter adapter;
	protected WatchPointTemplateEditor editor = null;
	protected List<IWatchPointDTO> nodeReferences;

	@Override
	public void run(IAction action) {
	}

	@Override
	public void selectionChanged(IAction action, ISelection selection) {
	}

	@Override
	public void setActivePart(IAction action, IWorkbenchPart targetPart) {
		this.viewPart = (ViewPart) targetPart;
	}
	
	protected TreeViewer getTreeViewer() {
		return ((TreeViewer) viewPart.getAdapter(CommonViewer.class));
	}
	
	protected void refreshTreeElement(NodeDefinitionDTO nodeDefinitionDTO) {
		TreeViewer tree = getTreeViewer();
		Object[] expandedItems = tree.getExpandedElements();
		List<Object> itemsToExpand = new ArrayList<Object>(Arrays.asList(expandedItems));
		
		/* Check if the current node is already expanded */
		if (!itemsToExpand.contains(nodeDefinitionDTO)) {
			itemsToExpand.add(nodeDefinitionDTO);
		}

		tree.getTree().setRedraw(false);
		/* Refreshes the resource type in order to keep the object in tree in synch with the file system */
		tree.refresh(nodeDefinitionDTO.getResourceType());
		
		INodeDefinitionService service = ServiceFactory.createFactory().createNodeDefinitionService();
		
		/* Expands the tree elements */
		for (Object item : itemsToExpand) {
			if (item instanceof NodeDefinitionDTO) {
				NodeDefinitionDTO node = (NodeDefinitionDTO) item;
				NodeDefinitionDTO refreshedNode = service.getNodeDefinitionDTOByPath(node.getPath());

				/* Could be null while builder is running in the background and re-creating this node */
				if (refreshedNode != null)
					tree.setExpandedState(refreshedNode, true);
			}
		}

		tree.getTree().setRedraw(true);
	}
	
	protected void updateSuperstateNodeDefinition(NodeDefinitionDTO nodeDTO, NodeDefinition node, String nodeName){
		NodeDefinition superstateNodeDefinition = null;
		try {
			superstateNodeDefinition = SemanticElementUtil.createNodeDefinition(nodeDTO.getXml());
        } catch (Exception e) {
			handleException("Error creating message node from xml: " + nodeDTO.getName(), e);
			return;
		}
		
		XmlAdapter adapter = Utility.getAdapterFromString(nodeDTO.getXml(), node.getFactory());
		adapter.initialize(superstateNodeDefinition);
		
		NodeGroup nodeGroup = (NodeGroup)superstateNodeDefinition.getNode();
		
		for (NodeElement nodeElement : nodeGroup.getNodeElements()) {
			if(nodeElement instanceof NodeDefinition && ((NodeDefinition) nodeElement).getDefinitionName().equals(nodeName) ){
				NodeDefinition nodeSuperstate = (NodeDefinition)nodeElement;
				updateNodeInSuperstate(nodeSuperstate);
			}
		}
		
		String xml = Utility.getXMLFromDocument(adapter.getNode().getOwnerDocument());
		nodeDTO.setXml(xml);
		
		INodeDefinitionService service = ServiceFactory.createFactory().createNodeDefinitionService();
		service.updateNodeDefinitionInFileSystem(nodeDTO);
	}
	
	protected void updateNodeInSuperstate(NodeDefinition nodeSuperstate) {
		
	}

	/**
	 * Build a map which key is the template and the value is a list containing the template messages
	 * which uses the given node definition.
	 * @param nodeDefinition The node definition to check for references
	 * @return A map which key is a template and the value is a list containing template messages
	 */
	protected Map<WorkflowTemplateDTO, List<TemplateMessageDTO>> buildReferencesMap(NodeDefinitionDTO nodeDefinitionDTO){
		Map<WorkflowTemplateDTO, List<TemplateMessageDTO>> templateMessagesMap = new HashMap<WorkflowTemplateDTO, List<TemplateMessageDTO>>();
		
		for (IWatchPointDTO referenceDTO : this.nodeReferences) {
			/* Check if the reference is a workflow template */
			if(referenceDTO instanceof WorkflowTemplateDTO){
				WorkflowTemplateDTO templateDTO = (WorkflowTemplateDTO)referenceDTO;
				List<TemplateMessageDTO> templateMessages = new ArrayList<TemplateMessageDTO>();
				for (TemplateMessageDTO messageDTO : templateDTO.getMessages()) {
					/* Get the message DTO if it's name match the node definition name */
		    		if(nodeDefinitionDTO.getName().equals( messageDTO.getNodeDefinitionName() )){
		    			templateMessages.add(messageDTO);
		    		}
				}
				/* Adds the template only if there is messages matching */
				if(!templateMessages.isEmpty()){
					templateMessagesMap.put(templateDTO, templateMessages);
				}
			}else if(referenceDTO instanceof NodeDefinitionDTO){
				/* Check if the reference is a node definition */
				NodeDefinitionDTO nodeDTO = (NodeDefinitionDTO) referenceDTO;
				NodeDefinition node = initializeNodeDefinition(nodeDTO);
				/* Get the templates using this node definition (SuperState) */
				for (IWatchPointDTO nodeReference : checkNodeInUse(node)) {
					if(nodeReference instanceof WorkflowTemplateDTO){
						WorkflowTemplateDTO templateDTO = (WorkflowTemplateDTO)nodeReference;
						List<TemplateMessageDTO> templateMessages = new ArrayList<TemplateMessageDTO>();
						for (TemplateMessageDTO messageDTO : templateDTO.getMessages()) {
							/* Get the message DTO if it's name match the node definition name */
							if(nodeDTO.getName().equals( messageDTO.getNodeDefinitionName() )){
				    			templateMessages.add(messageDTO);
				    		}
						}
						/* Adds the template only if there is messages matching */
						if(!templateMessages.isEmpty()){
							/* If the template is on the map, add the message to it's list */
							if(templateMessagesMap.containsKey(templateDTO)){
								templateMessagesMap.get(templateDTO).addAll(templateMessages);
							} else {
								templateMessagesMap.put(templateDTO, templateMessages);
							}
						}
					}
				} 
			}
		}
		
		return templateMessagesMap;
	}
	
	protected NodeDefinition initializeNodeDefinition(NodeDefinitionDTO nodeDTO){
		NodeDefinition node = null;
		try {
			node = SemanticElementUtil.createNodeDefinition(nodeDTO.getXml());
			node.setId(nodeDTO.getId());
        } catch (Exception e) {
			handleException("Error creating message node from xml: " + nodeDTO.getName(), e);
        }
		return node;
	}
	
	/* ------------------------------------------------------------------------------------------ */
	
	protected void handleException(String errorMessage, Exception e){
		Status status = new Status(IStatus.ERROR, "WatchPoint Studio",IStatus.ERROR, errorMessage, e);
		StatusManager.getManager().handle(status, StatusManager.SHOW);
	}
	
	protected void setNewNodeDefDescription(NodeDefinition node, String desc){
		Description description = (Description) node.getFactory().createById(
				SemanticElementConstants.DESCRIPTION_SEID);
		description.setDescription(desc);
		node.setDescription(description);
	}
	
	protected List<IWatchPointDTO> checkNodeInUse(NodeDefinition node) {
		
		IPersistenceContext context = JPAPersistenceContextFactory.createFactory().createPersistenceContext();
		context.initialize();
		Class<WorkflowTemplateDTO> entityClass = WorkflowTemplateDTO.class;
		DataAccessFactory daFactory = DataAccessFactory.createFactory();
		DataAccessInterface<?, ?> templatedao = daFactory.createDataAccessObject(entityClass, context);	
		Class<NodeDefinitionDTO> nodeentityClass = NodeDefinitionDTO.class;
		DataAccessInterface<?, ?> nodedao = daFactory.createDataAccessObject(nodeentityClass, context);
		
		NodeDefinitionDTO nodeDefinitionDTO = ((NodeDefinitionDTODAI)nodedao).find(node.getId());
		
		List<WorkflowTemplateDTO> templateList = new ArrayList<WorkflowTemplateDTO>();
		if (nodeDefinitionDTO.getType().equals(NodeDefinitionType.MessageNode)) {
			Collection<String> messageUIDs = new ArrayList<String>(); 
			for (Message msg : nodeDefinitionDTO.getMessages()) {
				messageUIDs.add(msg.getUid());
			}
			templateList = ((WorkflowTemplateDTODAI)templatedao).findByMessageUsage(messageUIDs, NodeDefinitionType.MessageNode);		
		} else {
			for (Message msg : nodeDefinitionDTO.getMessages()) {
				String systemID = Utility.getSystemIDByMessageUID(msg.getUid());
				templateList = ((WorkflowTemplateDTODAI)templatedao).findBySuperStateUsage(nodeDefinitionDTO.getName(), systemID);
				break;
			}
		}
		
		String messageUid = "";
		for (Message msg : nodeDefinitionDTO.getMessages()) {
			messageUid = msg.getUid();
		}
		List<NodeDefinitionDTO> nodeDefinitionList = ((NodeDefinitionDTODAI)nodedao).findByMessageUID(messageUid,  NodeDefinitionType.SuperState);
		List<IWatchPointDTO> references = new ArrayList<IWatchPointDTO>();
		references.addAll(templateList);
		references.addAll(nodeDefinitionList);

		return references;
	}

	protected String buildReferenceModifyingErrorMsg(
			List<IWatchPointDTO> references) {
		String message = "";
		String LINE_SEPRATOR = "line.separator";
		Set<String> templateSet = new HashSet<String>();
		Set<String> nodeDefSet = new HashSet<String>();
		/* Add DTO objects to sets to remove duplicate names */
		for (IWatchPointDTO dto : references) {
			if (dto instanceof NodeDefinitionDTO)
				nodeDefSet.add(((NodeDefinitionDTO) dto).getName());
			else if (dto instanceof WorkflowTemplateDTO)
				templateSet.add(((WorkflowTemplateDTO) dto).getName());
		}
		if (templateSet.size() > 0)
			message += "The node definition is being used by the following templates: "
					+ templateSet + "." + System.getProperty(LINE_SEPRATOR);
		if (nodeDefSet.size() > 0)
			message += "The node definition is being used by the following superstates: "
					+ nodeDefSet + "." + System.getProperty(LINE_SEPRATOR);
		return message;
	}
	
	protected String buildReferenceDeletionErrorMsg( List<IWatchPointDTO> references) {
		String message = "";
		String LINE_SEPRATOR = "line.separator";
		Set<String> templateSet = new HashSet<String>();
		Set<String> nodeDefSet = new HashSet<String>();
		/* Add DTO objects to sets to remove duplicate names */
		for (IWatchPointDTO dto : references) {
			if (dto instanceof NodeDefinitionDTO)
				nodeDefSet.add(((NodeDefinitionDTO) dto).getName());
			else if (dto instanceof WorkflowTemplateDTO)
				templateSet.add(((WorkflowTemplateDTO) dto).getName());
		}
		if (templateSet.size() > 0)
			message += "The node definition is being used by the following templates: "
					+ templateSet + "." + System.getProperty(LINE_SEPRATOR);
		if (nodeDefSet.size() > 0)
			message += "The node definition is being used by the following superstates: "
					+ nodeDefSet + "." + System.getProperty(LINE_SEPRATOR);
		message += "Are you sure you want to delete?";
		return message;
	}

	protected String parseDescription(String desc, String varName, String varDesc) {
		if (desc == null || desc.length() == 0)
			return desc;

		String line = System.getProperty("line.separator");
		int lineLength = line.length();

		String matchVar = "@param " + varName + " :";
		int inx = desc.indexOf(matchVar);
		int varNameIndex = -1;

		if (inx >= 0) {
			int localIndex = matchVar.indexOf(varName);
			varNameIndex = inx + localIndex;
			inx += localIndex;

			String subdesc = desc.substring(inx + varName.length() + ": ".length());
			inx = subdesc.indexOf("@");
			if (varDesc != null)
				desc = desc.substring(0, varNameIndex + varName.length() + ": ".length()) + " "
						+ varDesc + ((inx >= 0) ? (line + subdesc.substring(inx)) : "");
			else {
				desc = ((desc.length() > 7) ? desc.substring(0, varNameIndex - 7) : "");
				if (inx >= 0) {
					if ((desc.length() > 0)
							&& !desc.substring(desc.length() - lineLength).equals(line)) {
						desc = desc += line;
					}
					desc = desc += subdesc.substring(inx);
				} else {
					desc = desc += "";
				}
			}

		} else {
			if (!desc.substring(desc.length() - lineLength).equals(line)) {
				desc = desc += line;
			}
			desc += "@param " + varName + " : " + (varDesc == null ? "" : varDesc);
		}
		return desc;
	}

}
