package com.tandbergtv.watchpoint.studio.ui.wizard;

import static com.tandbergtv.watchpoint.studio.application.ApplicationPropertyKeys.WORKFLOW_SYSTEM_ID;
import static com.tandbergtv.watchpoint.studio.ui.wizard.RTPConstants.SCHEMA_FILE_NAME;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.WizardNewProjectCreationPage;
import org.eclipse.ui.part.ISetSelectionTarget;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.tandbergtv.watchpoint.studio.application.ApplicationProperties;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer;
import com.tandbergtv.watchpoint.studio.util.ProjectPreferencesUtil;
import com.tandbergtv.watchpoint.studio.util.XMLDocumentUtility;

/**
 * @author Patrik Araujo
 */
public class NewResourceTypeProjectWizard extends Wizard implements INewWizard {
	private static final String SCHEMA_IMPORT_ELEMENT = "xs:import";
	
	WizardNewProjectCreationPage projectPage;
	
	NewResourceTypeProjectWizardPage resourceTypePage;
	IResourceTypeService rtService;
	ResourceType resourceType;
	
	private IProject newProject;
	private IWorkbench workbench;
	
	public NewResourceTypeProjectWizard() {
		this.workbench = PlatformUI.getWorkbench();
		setWindowTitle("New Resource Type Project");
		setNeedsProgressMonitor(true);
		rtService = ServiceFactory.createFactory().createResourceTypeService();
	}
	
	@Override
	public void init(IWorkbench arg0, IStructuredSelection arg1) {
		// This method is required to this wizard be able to be shown
		// in the New Project dialog in eclipse. 
	}

	/**
	 * Get the Resource Type created by this wizard
	 * @return the resourceType
	 */
	public ResourceType getResourceType() {
		return this.resourceType;
	}
	
	public String getProjectName(){
		return projectPage.getProjectName();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jface.wizard.Wizard#performFinish()
	 */
	@Override
	public boolean performFinish() {
		try {
			if(!validateFormPage()){
				return false;
			}
			
			ResourceType newResourceType = this.resourceTypePage.getResourceTypeObject();
			
			this.resourceType = newResourceType;
			
			// Creates the java project related to the resource type 
			createJavaProject();
			selectAndReveal(newProject);
			
			// Refresh resource type explorer and resource group explorer
			refreshExtPointView("com.tandbergtv.gd.ResourceTypeExplorer");
			refreshExtPointView("com.tandbergtv.gd.ResourceGroup");
		}
		catch(Exception e) {
			Utility.reportError("Resource Type Creation Failed.", e);
			return false;
		}
		
		return true;
	}
	
	private boolean validateFormPage() {
		/* Check if the name is unique */
		if(!this.rtService.isResourceTypeNameUnique(this.resourceTypePage.getResourceTypeName())) {
			this.resourceTypePage.setErrorMessage("A Resource Type with this name already exists");
			return false;
		}
		/* Check if the systemId is unique */
		if(!this.rtService.isResourceTypeSystemIdUnique(this.resourceTypePage.getResourceTypeSystemID())) {
			this.resourceTypePage.setErrorMessage("A Resource Type with this System ID already exists");
			return false;
		}
		/* Check if the systemId is equals to the workflow systemId */
		ApplicationProperties properties = ApplicationProperties.getInstance();
		String workflowSystemId = properties.getProperty(WORKFLOW_SYSTEM_ID);
		if(this.resourceTypePage.getResourceTypeSystemID().equals(workflowSystemId)){
			this.resourceTypePage.setErrorMessage("The System ID can't be the same as the Workflow System");
			return false;
		}
		
		return true;
	}

	private void refreshExtPointView(String viewId){
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		AbstractTreeViewExplorer explorer = null;
		try {
			IViewPart view = window.getActivePage().showView(viewId);
			explorer = (AbstractTreeViewExplorer) view;
			explorer.refresh();
		} catch (PartInitException e) {
			Utility.reportError("Impossible to refresh resource type view.", e);
		}
	}
	
	private IProject getWorkspaceProject(String projectName){
		return ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
	}

	/**
	 * Add pages to the wizard
	 */
	public void addPages() {
		addProjectPage();
		addResourceTypePage();
	}
	
	private void addProjectPage() {
		super.addPages();
		setWindowTitle("New Resource Type Project");
		projectPage = new WizardNewProjectCreationPage("basicNewProjectPage");
		projectPage.setTitle("New Resource Type Project");
		projectPage.setDescription("Create a new Resource Type Project.");		
		addPage(projectPage);
	}
	
	private void addResourceTypePage() {
		resourceTypePage = new NewResourceTypeProjectWizardPage();
		addPage(resourceTypePage);
	}
	
	/**
	 * Creates the java project related to the resource type
	 */
	private void createJavaProject() { 
		try {
			ResourceTypeJavaProjectCreator projectCreator = new ResourceTypeJavaProjectCreator(
					resourceType, resourceType.getName(), null);
			this.newProject = projectCreator.createJavaProject();
			createInitialContent(this.newProject);
		} catch (RuntimeException e) {
			Utility.reportError("Error while creating Resource Type project", e);
		}
	}
	
	private void handleProjectException( String message, Exception e){
		Utility.reportError("Resource Type Creation Failed.", e);
		try {
			if(getWorkspaceProject(projectPage.getProjectName()).exists()){
				getWorkspaceProject(projectPage.getProjectName()).delete(true, null);
			}
		} catch (CoreException ex) {
			ErrorDialog.openError(getShell(), "Problem deleting java project", null, ex.getStatus());
		}
		
	}
	
	void createProject(IProjectDescription description, IProject projectHandle,
			IProgressMonitor monitor) throws CoreException,
			OperationCanceledException {
		try {
			monitor.beginTask("", 2000);
			projectHandle.create(description, new SubProgressMonitor(monitor,
					1000));
			if (monitor.isCanceled()) {
				throw new OperationCanceledException();
			}
			projectHandle.open(IResource.BACKGROUND_REFRESH,
					new SubProgressMonitor(monitor, 1000));
		} finally {
			monitor.done();
		}
	}
	
	protected void selectAndReveal(IResource newResource) {
		selectAndReveal(newResource, workbench.getActiveWorkbenchWindow());
	}
	
	private void selectAndReveal(IResource resource,
	           IWorkbenchWindow window) {
		if (!inputValid(resource, window)) return;   
		Iterator<IWorkbenchPart> itr = getParts(window.getActivePage()).iterator();
		while (itr.hasNext()) {
		    selectAndRevealTarget(
					window, 
					new StructuredSelection(resource), 
					getTarget((IWorkbenchPart)itr.next()));
		}
	}
	
	private boolean inputValid(IResource resource, IWorkbenchWindow window) {
		if (window == null || resource == null) return false;
		else if (window.getActivePage() == null) return false;
		else return true;
	}

	private void selectAndRevealTarget(IWorkbenchWindow window, final ISelection selection, ISetSelectionTarget target) {
		if (target == null) return;
		final ISetSelectionTarget finalTarget = target;
		window.getShell().getDisplay().asyncExec(new Runnable() {
		    public void run() {
		        finalTarget.selectReveal(selection);
		    }
		});
	}
	
	private ISetSelectionTarget getTarget(IWorkbenchPart part) {
        ISetSelectionTarget target = null;
        if (part instanceof ISetSelectionTarget) {
            target = (ISetSelectionTarget)part;
        }
        else {
            target = (ISetSelectionTarget)part.getAdapter(ISetSelectionTarget.class);
        }
		return target;		
	}

	private List<IWorkbenchPart> getParts(IWorkbenchPage page) {
		List<IWorkbenchPart> result = new ArrayList<IWorkbenchPart>();
		addParts(result, page.getViewReferences());
		addParts(result, page.getEditorReferences());
		return result;
	}
	
	private void addParts(List<IWorkbenchPart> parts, IWorkbenchPartReference[] refs) {
		for (int i = 0; i < refs.length; i++) {
           IWorkbenchPart part = refs[i].getPart(false);
           if (part != null) {
               parts.add(part);
           }
	    }		
	}
	
	private void createInitialContent(IProject javaProject) {
		try {
			copySchemaFile(javaProject);
		} catch (Exception e) {
			handleProjectException( "Resource Type Creation Failed.", e );
		}
	}
	
	private void copySchemaFile(IProject javaProject) throws Exception {
		File schemaFile = resourceTypePage.getFile();
		if(schemaFile != null){
			String schemaFileName = schemaFile.getName();
			// Create schema file
			createProjectFile(javaProject, schemaFileName, schemaFile);
			IFile file = javaProject.getProject().getFile(schemaFileName);
			try {
				ArrayList<String> importPaths = getSchemaImportFilePath(file.getLocation().toOSString());
				if(importPaths.size() > 0){
					for (String importPath : importPaths) {
						File schemaImportFile = getImportFile(schemaFile, importPath);
						if(schemaImportFile != null){
							// Copy each import schema into the project
							createProjectFile(javaProject, schemaImportFile.getName(), schemaImportFile);
						}else{
							MessageDialog.openError(Display.getCurrent().getActiveShell(), "Problem",
							"Impossible to copy schema import file.\n" +"File path : " + importPath);
						}
					}
					updateSchemaImports(file);
				}
				// Set the schema file name in the project.
				ProjectPreferencesUtil.addProjectPreference(javaProject.getProject(), SCHEMA_FILE_NAME,
						schemaFileName);
			} catch (FileNotFoundException e) {
				ErrorDialog.openError(getShell(), "Problem creating schema files", null, null);	
			}
		}
	}
	
	private void createProjectFile(IProject javaProject, String relativePath, File file) throws FileNotFoundException, CoreException{
		createProjectFileFromInputStream(javaProject, relativePath, new FileInputStream(file));
	}
	
	private void createProjectFileFromInputStream(IProject javaProject, String relativePath, InputStream fileInputStream) throws CoreException{
		IFile projectFile = javaProject.getProject().getFile(relativePath);
		if(!projectFile.exists()){
			projectFile.create(fileInputStream, true, null);
		}
	}
	
	private File getImportFile(File schema, String importPath){
		IPath importSchemaPath = new Path(importPath);
		File importSchemaFile = null;
		if( !importSchemaPath.isAbsolute() ){
			URI importSchemaURI = schema.toURI().resolve(importPath);
			importSchemaFile = new File(importSchemaURI.getPath());
		}else{
			importSchemaFile = new File(importPath);
		}
		return importSchemaFile.exists() ? importSchemaFile : null ;
	}
		
	private ArrayList<String> getSchemaImportFilePath(String filePath) throws Exception {
		Document schema = XMLDocumentUtility.loadFile(new FileInputStream(filePath), false);
		NodeList importElements = schema.getElementsByTagName(SCHEMA_IMPORT_ELEMENT);
		ArrayList<String> filePaths = new ArrayList<String>();
		for (int i = 0; i < importElements.getLength(); i++) {
			Node importLocation = importElements.item(i).getAttributes().getNamedItem("schemaLocation");
			String importPath = importLocation.getNodeValue();
			filePaths.add(importPath);
		}
		return filePaths;
	}
	
	private void updateSchemaImports(IFile file) throws Exception {
		Document schema = XMLDocumentUtility.loadFile(file.getContents(), false);
		NodeList importElements = schema.getElementsByTagName(SCHEMA_IMPORT_ELEMENT);
		for (int i = 0; i < importElements.getLength(); i++) {
			Node importLocation = importElements.item(i).getAttributes().getNamedItem("schemaLocation");
			File temp = new File(importLocation.getNodeValue());
			importLocation.setNodeValue(temp.getName());
		}
		ByteArrayOutputStream baos = XMLDocumentUtility.createByteArray(schema);
		byte[] contentArray = baos.toByteArray();
		file.setContents(new ByteArrayInputStream(contentArray), IFile.KEEP_HISTORY, null);
	}

}
