package com.tandbergtv.watchpoint.studio.ui.wizard;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants.RESOURCE_TYPE_BUILD_FILE_NAME;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants.RESOURCE_TYPE_BUILD_PROPERTIES_FILENAME;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants.RESOURCE_TYPE_PLUGIN_FILE_NAME;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Properties;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.ui.PreferenceConstants;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.jbpm.gd.jpdl.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;

import com.tandbergtv.watchpoint.studio.builder.WatchpointResourceTypeNature;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.ui.wizard.resourcegroup.NewResourceGroupWizard;
import com.tandbergtv.watchpoint.studio.util.PluginFileGenerationException;
import com.tandbergtv.watchpoint.studio.util.ResourceTypePluginFileGenerator;
import com.tandbergtv.watchpoint.studio.util.XMLDocumentUtility;

public class ResourceTypeJavaProjectCreator {
	private ResourceType resourceType;
	private String projectName;
	private String locationPath;
	
	public ResourceTypeJavaProjectCreator(ResourceType resourceType, String projectName, String locationPath) {
		this.resourceType = resourceType;
		this.projectName = projectName;
		this.locationPath = locationPath;
	}
	
	public IProject createJavaProject() throws RuntimeException{
		IProject newProject = null;
		try {
			newProject = createNewProject();
			IJavaProject javaProject = JavaCore.create(newProject);
			createOutputLocation(javaProject);
			addJavaBuilder(javaProject);
			setClasspath(javaProject);
			createInitialContent(javaProject);
			addNature(newProject);
		} catch (Exception e) {
			throw new RuntimeException("Error while creating " + projectName + " project", e);
		}
		return newProject;
	}
	
	private void addNature(IProject project) throws CoreException {
			IProjectDescription description = project.getDescription();
			String[] natures = description.getNatureIds();

			// Add the nature
			String[] newNatures = new String[natures.length + 1];
			System.arraycopy(natures, 0, newNatures, 0, natures.length);
			newNatures[natures.length] = WatchpointResourceTypeNature.NATURE_ID;
			description.setNatureIds(newNatures);
			project.setDescription(description, null);
	}
	
	private IProject createNewProject() {
		final IProject newProjectHandle = ResourcesPlugin.getWorkspace().getRoot().getProject(this.projectName);
		final IProjectDescription description = createProjectDescription(newProjectHandle);
		WorkspaceModifyOperation op = new WorkspaceModifyOperation() {
			protected void execute(IProgressMonitor monitor)
					throws CoreException {
				createProject(description, newProjectHandle, monitor);
			}
		};
		runProjectCreationOperation(op, newProjectHandle);
		return newProjectHandle;
	}
	
	private void createProject(IProjectDescription description, IProject projectHandle,
			IProgressMonitor monitor) throws CoreException,
			OperationCanceledException {
		try {
			monitor.beginTask("", 2000);
			projectHandle.create(description, new SubProgressMonitor(monitor,
					1000));
			if (monitor.isCanceled()) {
				throw new OperationCanceledException();
			}
			projectHandle.open(IResource.BACKGROUND_REFRESH,
					new SubProgressMonitor(monitor, 1000));
		} finally {
			monitor.done();
		}
	}
	
	private void runProjectCreationOperation(WorkspaceModifyOperation op,
			IProject newProjectHandle) {
		try {
			op.run(null);
		} catch (InterruptedException e) {
			Logger.logError("InterruptedException while creating project", e);
		} catch (InvocationTargetException e) {
			Throwable t = e.getTargetException();
			if (t instanceof CoreException) {
				throw new RuntimeException("Project creation interrupted. " + t.getMessage());
			} else {
				throw new RuntimeException("Error while creating project " + newProjectHandle.getName());
			}
		}
	}
	
	private IProjectDescription createProjectDescription(
			IProject newProjectHandle) {
		IPath newPath = null;
		if (this.locationPath != null)
			newPath = new Path(locationPath);
		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		IProjectDescription description = workspace
				.newProjectDescription(newProjectHandle.getName());
		description.setLocation(newPath);
		addJavaNature(description);
		return description;
	}
	
	private void addJavaNature(IProjectDescription description) {
		ArrayList<String> natures = new ArrayList<String>();
		natures.addAll(Arrays.asList(description.getNatureIds()));
		natures.add(JavaCore.NATURE_ID);
		description.setNatureIds((String[])natures.toArray(new String[natures.size()]));
	}
	
	private void createOutputLocation(IJavaProject javaProject) throws JavaModelException, CoreException {
		IFolder binFolder = javaProject.getProject().getFolder("bin");
		createFolder(binFolder);
		IPath outputLocation = binFolder.getFullPath();
		javaProject.setOutputLocation(outputLocation, null);
	}
	
	private void addJavaBuilder(IJavaProject javaProject) throws CoreException {
		IProjectDescription desc = javaProject.getProject().getDescription();
		ICommand command = desc.newCommand();
		command.setBuilderName(JavaCore.BUILDER_ID);
		desc.setBuildSpec(new ICommand[] { command });
		javaProject.getProject().setDescription(desc, null);
	}
	
	private void createFolder(IFolder folder) throws CoreException {
		IContainer parent = folder.getParent();
		if (parent != null && !parent.exists() && parent instanceof IFolder) {
			createFolder((IFolder)parent);
		}
		folder.create(true, true, null);
	}
	
	private void setClasspath(IJavaProject javaProject) throws JavaModelException, CoreException {
		javaProject.setRawClasspath(new IClasspathEntry[0], null);
		addSourceFolders(javaProject);
		addJRELibraries(javaProject);
//		addJbpmLibraries(javaProject);
		// Hack to overcome the problems of the classpath container not being created in the classpath.
		javaProject.getRawClasspath();
	}
	
	private void addJRELibraries(IJavaProject javaProject) throws JavaModelException {		
		ArrayList<IClasspathEntry> entries = new ArrayList<IClasspathEntry>();
		entries.addAll(Arrays.asList(javaProject.getRawClasspath()));
		entries.addAll(Arrays.asList(PreferenceConstants.getDefaultJRELibrary()));
		javaProject.setRawClasspath((IClasspathEntry[])entries.toArray(new IClasspathEntry[entries.size()]), null);
	}
	
	private void addSourceFolders(IJavaProject javaProject) throws JavaModelException, CoreException {
		ArrayList<IClasspathEntry> entries = new ArrayList<IClasspathEntry>();
		entries.addAll(Arrays.asList(javaProject.getRawClasspath()));
		addSourceFolder(javaProject, entries, "src");
		javaProject.setRawClasspath((IClasspathEntry[])entries.toArray(new IClasspathEntry[entries.size()]), null);
	}
	
	private void addSourceFolder(IJavaProject javaProject, ArrayList<IClasspathEntry> entries, String path) throws CoreException {
		IFolder folder = javaProject.getProject().getFolder(path);
		createFolder(folder);
		IPackageFragmentRoot root = javaProject.getPackageFragmentRoot(folder);
		entries.add(JavaCore.newSourceEntry(root.getPath()));
	}
	
	private void createInitialContent(IJavaProject javaProject) throws TransformerException, ParserConfigurationException, IOException, CoreException {
		createFolders(javaProject);
		try {
			copyBuildFiles(javaProject);
			copyPluginFile(javaProject);
			createDefaultResourceGroup(javaProject);
		} catch (Exception e) {
//			handleProjectException( "Resource Type Creation Failed.", e );
		}
	}
	
	/*
	 * 	Creates the default ResourceGroup with the same name as the ResourceType
	 */
	private void createDefaultResourceGroup(IJavaProject javaProject) throws CoreException {
		NewResourceGroupWizard.createResourceGroup(javaProject.getProject().getFolder("groups"), 
				resourceType.getName(), resourceType.getSystemId());
	}

	private void copyPluginFile(IJavaProject javaProject) throws PluginFileGenerationException, CoreException {
		// Copy plugin.xml
		Document pluginFile = ResourceTypePluginFileGenerator.generatePluginDocument(resourceType);
		File pluginSourcePath = javaProject.getProject().getLocation().toFile();
		ByteArrayOutputStream plugin = ResourceTypePluginFileGenerator.createWritablePluginDocument(pluginFile, pluginSourcePath);
		byte[] pluginContentArray = plugin.toByteArray();
		createProjectFileFromInputStream(javaProject, RESOURCE_TYPE_PLUGIN_FILE_NAME, new ByteArrayInputStream(pluginContentArray));
	}

	private void copyBuildFiles(IJavaProject javaProject) throws Exception {
		// Copy build.xml
		InputStream buildFile = this.getClass().getResourceAsStream(RESOURCE_TYPE_BUILD_FILE_NAME);
		Document buildDocument = XMLDocumentUtility.loadFile(buildFile, false);
		String projectName = javaProject.getProject().getName();
		changeBuildAttributeValue(buildDocument, "project", "name", projectName);
		ByteArrayOutputStream baos = XMLDocumentUtility.createByteArray(buildDocument);
		byte[] buildContentArray = baos.toByteArray();
		createProjectFileFromInputStream(javaProject, RESOURCE_TYPE_BUILD_FILE_NAME, new ByteArrayInputStream(buildContentArray));

		// Copy build.properties
		Properties prop = new Properties();
		prop.load(this.getClass().getResourceAsStream(RESOURCE_TYPE_BUILD_PROPERTIES_FILENAME));
		prop.put("project.name", javaProject.getProject().getName());
		ByteArrayOutputStream propFile = new ByteArrayOutputStream();
		prop.store(propFile, "Properties for the Resource Type build.xml");
		byte[] fileContentArray = propFile.toByteArray();
		createProjectFileFromInputStream(javaProject, RESOURCE_TYPE_BUILD_PROPERTIES_FILENAME, new ByteArrayInputStream(fileContentArray));
	}
	
	/**
	 * This method change the value of a given attribute in a given element in a given document.<br>
	 * This method assume that the given element occur just once in the given document.
	 * @param doc The Document object
	 * @param element The name of the element to be changed
	 * @param attribute The attribute to be changed
	 * @param value The new attribute value
	 */
	private void changeBuildAttributeValue(Document doc, String element, String attribute, String value){
		NodeList elements = doc.getElementsByTagName(element);
		elements.item(0).getAttributes().getNamedItem(attribute).setNodeValue(value);
	}
	
//	private void createProjectFile(IJavaProject javaProject, String relativePath, File file) throws FileNotFoundException, CoreException{
//		createProjectFileFromInputStream(javaProject, relativePath, new FileInputStream(file));
//	}
	
	private void createProjectFileFromInputStream(IJavaProject javaProject, String relativePath, InputStream fileInputStream) throws CoreException{
		IFile projectFile = javaProject.getProject().getFile(relativePath);
		if(!projectFile.exists()){
			projectFile.create(fileInputStream, true, null);
		}
	}
	
	private void createFolders(IJavaProject javaProject) throws CoreException {
		ArrayList<String> entries = new ArrayList<String>();
		entries.add("resources");
		entries.add("lib");
		entries.add("resources/adaptor");
		entries.add("groups");
		for (String path : entries) {
			IFolder folder = javaProject.getProject().getFolder(path);
			createFolder(folder);
		}
	}

}
