/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.tandbergtv.watchpoint.studio.ui.wizard.export;

import static com.tandbergtv.watchpoint.studio.ui.wizard.export.ResourceTypeExportConstants.RESOURCE_TYPE_IMAGE_NAME;

import java.util.ArrayList;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.pde.core.IModel;
import org.eclipse.pde.internal.ui.PDEUIMessages;
import org.eclipse.pde.internal.ui.parts.WizardCheckboxTablePart;
import org.eclipse.pde.internal.ui.wizards.ListUtil;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;

import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

@SuppressWarnings("restriction")
public abstract class BaseExportWizardPage extends AbstractExportWizardPage {

	protected ExportPart fExportPart;
	private IStructuredSelection fSelection;
	protected ExportDestinationTab fDestinationTab;
	protected TabFolder fTabFolder;

	class ExportListProvider implements IStructuredContentProvider {
		public Object[] getElements(Object parent) {
			return getListElements();
		}

		public void dispose() {
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}
	}
	
	class ResourceTypeLabelProvider extends LabelProvider implements ITableLabelProvider{
		
		@Override
		public String getText(Object element) {
			if (element instanceof IProject) {
				String name = ((IProject) element).getName();
				return name;
			}
			return super.getText(element);
		}

		@Override
		public Image getImage(Object element) {
			return super.getImage(element);
		}

		@Override
		public Image getColumnImage(Object element, int columnIndex) {
			Image image = this.createImage(RESOURCE_TYPE_IMAGE_NAME);
			return image;
		}

		@Override
		public String getColumnText(Object element, int columnIndex) {
			if (element instanceof IProject) {
				String name = ((IProject) element).getName();
				return name;
			}
			return super.getText(element);
		}
		
		private Image createImage(String path) {
			if (path == null)
				return null;

			ImageDescriptor descriptor = Utility.getImageDescriptor(path);
			return new Image(Display.getCurrent(), descriptor.getImageData());
		}
		
	}

	class ExportPart extends WizardCheckboxTablePart {
		public ExportPart(String label, String[] buttonLabels) {
			super(label, buttonLabels);
		}

		public void updateCounter(int count) {
			super.updateCounter(count);
			pageChanged();
		}

		protected void buttonSelected(Button button, int index) {
			switch (index) {
				case 0 :
					handleSelectAll(true);
					break;
				case 1 :
					handleSelectAll(false);
			}
		}
	}

	public BaseExportWizardPage(IStructuredSelection selection, String name, String choiceLabel) {
		super(name);
		fSelection = selection;
		fExportPart = new ExportPart(choiceLabel, new String[] {PDEUIMessages.WizardCheckboxTablePart_selectAll, PDEUIMessages.WizardCheckboxTablePart_deselectAll});
		setDescription("Export the selected Resource Type projects");
	}

	public void createControl(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		GridLayout layout = new GridLayout();
		layout.verticalSpacing = 10;
		container.setLayout(layout);

		createViewer(container);

		fTabFolder = new TabFolder(container, SWT.NONE);
		fTabFolder.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		createTabs(fTabFolder, getDialogSettings());

		initializeViewer();
		if (getErrorMessage() != null) {
			setMessage(getErrorMessage());
			setErrorMessage(null);
		}
		setControl(container);
		Dialog.applyDialogFont(container);
	}

	protected void createTabs(TabFolder folder, IDialogSettings settings) {
		createDestinationTab(folder);

		fDestinationTab.initialize(settings);
	}

	protected void createDestinationTab(TabFolder folder) {
		fDestinationTab = new ExportDestinationTab(this);
		TabItem item = new TabItem(folder, SWT.NONE);
		item.setControl(fDestinationTab.createControl(folder));
		item.setText(PDEUIMessages.ExportWizard_destination);
		
	}

	protected void createViewer(Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		composite.setLayout(layout);
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));

		fExportPart.createControl(composite);
		GridData gd = (GridData) fExportPart.getControl().getLayoutData();
		gd.heightHint = 125;
		gd.widthHint = 150;
		gd.horizontalSpan = 2;

		TableViewer viewer = fExportPart.getTableViewer();
		viewer.setContentProvider(new ExportListProvider());
		viewer.setLabelProvider(new ResourceTypeLabelProvider());
		viewer.setComparator(ListUtil.PLUGIN_COMPARATOR);
		viewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				if (event.getSource() instanceof CheckboxTableViewer) {
					IStructuredSelection selection = (IStructuredSelection) event.getSelection();
					((CheckboxTableViewer) event.getSource()).setChecked(selection.getFirstElement(), !((CheckboxTableViewer) event.getSource()).getChecked(selection.getFirstElement()));
					fExportPart.updateCounter(((CheckboxTableViewer) event.getSource()).getCheckedElements().length);
				}
			}
		});
		fExportPart.getTableViewer().setInput(getInput());
	}

	protected abstract Object getInput();

	protected void initializeViewer() {
		Object[] elems = fSelection.toArray();
		ArrayList<IProject> checked = new ArrayList<IProject>(elems.length);

		for (int i = 0; i < elems.length; i++) {
			Object elem = elems[i];
			IProject project = null;

			if (elem instanceof IFile) {
				IFile file = (IFile) elem;
				project = file.getProject();
			} else if (elem instanceof IProject) {
				project = (IProject) elem;
			} else if (elem instanceof IJavaProject) {
				project = ((IJavaProject) elem).getProject();
			} else if (elem instanceof ResourceType) {
				ResourceType rtype = (ResourceType) elem;
				project = ResourcesPlugin.getWorkspace().getRoot().getProject(rtype.determineNameForProject());
			}
			if (project != null) {
				if(!checked.contains(project)){
					checked.add(project);
				}
			}
		}
		fExportPart.setSelection(checked.toArray());
		if (checked.size() > 0)
			fExportPart.getTableViewer().reveal(checked.get(0));
	}

	public Object[] getSelectedItems() {
		return fExportPart.getSelection();
	}

	protected void pageChanged() {
		if (getMessage() != null)
			setMessage(null);
		String error = fExportPart.getSelectionCount() > 0 ? null : PDEUIMessages.ExportWizard_status_noselection;
		if (error == null)
			error = validateTabs();
		setErrorMessage(error);
		setPageComplete(error == null);
	}

	protected String validateTabs() {
		String message = fDestinationTab.validate();
		return message;
	}

	protected abstract boolean isValidModel(IModel model);

	public abstract Object[] getListElements();

	protected abstract IModel findModelFor(IAdaptable object);

	protected void saveSettings(IDialogSettings settings) {
		fDestinationTab.saveSettings(settings);
	}

	protected String getDestination() {
		return fDestinationTab.getDestination();
	}
	
	protected boolean getExportGroups() {
		return fDestinationTab.getAutoExportGroups();
	}

	protected boolean doExportSource() {
		return false;
	}

	protected String getQualifier() {
		return null;
	}
}
