/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.tandbergtv.watchpoint.studio.ui.wizard.export;

import static com.tandbergtv.watchpoint.studio.ui.preferences.PreferenceConstants.EXPORT_DIR;

import java.io.File;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.pde.internal.ui.PDEUIMessages;
import org.eclipse.pde.internal.ui.SWTFactory;
import org.eclipse.pde.internal.ui.util.SWTUtil;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.DirectoryDialog;

import com.tandbergtv.watchpoint.studio.ui.UI;
import com.tandbergtv.watchpoint.studio.ui.WatchPointStudioPlugin;
import com.tandbergtv.watchpoint.studio.ui.preferences.PreferenceConstants;

@SuppressWarnings("restriction")
public class ExportDestinationTab extends AbstractExportTab {

	public static final String PLUGINS_FOLDER_NAME = "plugins";
	public static final String SUBSYSTEMS_FOLDER_NAME = "subsystems";
	
	protected static final String S_DESTINATION = "destination"; //$NON-NLS-1$

	protected Combo fDirectoryCombo;
	protected Button fBrowseDirectory;
	protected Button autoExportGroups;
	

	public ExportDestinationTab(AbstractExportWizardPage page) {
		super(page);
	}

	public Control createControl(Composite parent) {
		Composite composite = SWTFactory.createComposite(parent, 2, 1, GridData.FILL_HORIZONTAL);

		fDirectoryCombo = SWTFactory.createCombo(composite, SWT.BORDER, 1, null);
		((GridData) fDirectoryCombo.getLayoutData()).horizontalIndent = 15;

		fBrowseDirectory = SWTFactory.createPushButton(composite, PDEUIMessages.ExportWizard_browse, null);
		SWTUtil.setButtonDimensionHint(fBrowseDirectory);

		IPreferenceStore prefs = WatchPointStudioPlugin.getDefault().getPreferenceStore();
		boolean exportRG = prefs.getBoolean(PreferenceConstants.AUTO_EXPORT_RESOURCE_GROUPS);

		autoExportGroups = SWTFactory.createCheckButton(composite, UI.ExportPage_AutoExportResourceGroup, null, exportRG, 2);
		SWTUtil.setButtonDimensionHint(autoExportGroups);

		return composite;
	}

	protected void initialize(IDialogSettings settings) {
		updateExportType();
		initializeCombo(settings, S_DESTINATION, fDirectoryCombo);
		hookListeners();
	}

	private String getSuggestedLocation() {
		IPreferenceStore prefs = WatchPointStudioPlugin.getDefault().getPreferenceStore();
		String suggestedName = prefs.getString(EXPORT_DIR);

		if (suggestedName != null && !suggestedName.equals("")) {
			suggestedName = suggestedName.trim();
			if (!suggestedName.endsWith(File.separator)) {
				suggestedName += File.separator;
			}
			suggestedName += PLUGINS_FOLDER_NAME + File.separator + 
							 SUBSYSTEMS_FOLDER_NAME + File.separator;
		} else {
			suggestedName = "";
		}
		return suggestedName;
	}

	
	protected void initializeCombo(IDialogSettings settings, String key, Combo combo) {
		super.initializeCombo(settings, key, combo);
		
		String suggestedLocation = getSuggestedLocation();
		if (!isValidLocation(combo.getText().trim()) || combo.getText().trim().isEmpty()) {
			combo.setText(suggestedLocation);
		} 
		
		// If suggestedLocation is not present in the combo, adds an entry.
		boolean found = false;
		for (String item : combo.getItems()) {
			if (item.equals(suggestedLocation)) {
				found = true;
				break;
			}
		}
		if (!found) {
			combo.add(suggestedLocation);
			// Workaround to not truncate the combo box. 
			combo.setText(combo.getText());
		}
	}

	protected void updateExportType() {
		fDirectoryCombo.setEnabled(true);
		fBrowseDirectory.setEnabled(true);
	}

	protected void hookListeners() {
		fDirectoryCombo.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				fPage.pageChanged();
			}
		});

		fBrowseDirectory.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				chooseDestination(fDirectoryCombo);
			}
		});
	}

	private void chooseDestination(Combo combo) {
		DirectoryDialog dialog = new DirectoryDialog(fPage.getShell(), SWT.SAVE);
		String path = combo.getText();
		if (path.trim().length() == 0)
			path = ResourcesPlugin.getWorkspace().getRoot().getLocation().toString();
		dialog.setFilterPath(path);
		dialog.setText(PDEUIMessages.ExportWizard_dialog_title);
		dialog.setMessage(PDEUIMessages.ExportWizard_dialog_message);
		String res = dialog.open();
		if (res != null) {
			if (combo.indexOf(res) == -1)
				combo.add(res, 0);
			combo.setText(res);
		}
	}

	protected void saveSettings(IDialogSettings settings) {
		saveCombo(settings, S_DESTINATION, fDirectoryCombo);
	}

	protected String validate() {
		if (fDirectoryCombo.getText().trim().length() == 0)
			return PDEUIMessages.ExportWizard_status_nodirectory;
		else if (!isValidLocation(fDirectoryCombo.getText().trim()))
			return UI.ExportDestinationTab_status_invaliddirectory;

		return null;
	}

	protected String getDestination() {
		File dir = new File(fDirectoryCombo.getText().trim());
		return dir.getAbsolutePath();
	}
	
	protected boolean getAutoExportGroups() {
		return autoExportGroups.getSelection();
	}
}
