package com.tandbergtv.watchpoint.studio.ui.wizard.resourcegroup;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;
import org.eclipse.ui.model.WorkbenchContentProvider;
import org.eclipse.ui.model.WorkbenchLabelProvider;

import com.tandbergtv.watchpoint.studio.builder.WatchpointResourceTypeNature;
import com.tandbergtv.watchpoint.studio.builder.WatchpointTemplateNature;
import com.tandbergtv.watchpoint.studio.service.IResourceGroupService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;

/**
 * 		Wizard for new ResourceGroups
 * 
 * @author <a href="francisco.neto@venturus.org.br">vntfrne Francisco Bento da Silva Neto</a>
 *
 */
public class NewResourceGroupWizardPage extends WizardPage {
	
	private Text containerText;
	private Text processText;
	private Button browseButton;
	
	private IWorkspaceRoot workspaceRoot;
	private String containerName;
	private String suggestedFolder = "";

	public NewResourceGroupWizardPage(String suggestedFolder) {
		super("Resource Group");
		setTitle("Create Resource Group");
		setDescription("Create a new Resource Group");	
		workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
		if (suggestedFolder != null) {
			this.suggestedFolder = suggestedFolder;	
		}
	}

	public void init(IStructuredSelection selection) {
		IContainer container = null;
		if (selection != null && !selection.isEmpty()) {
			Object object = selection.getFirstElement();
			if (IFile.class.isInstance(object) && !IContainer.class.isInstance(object)) {
				container = ((IFile)object).getParent();
			} else if (IContainer.class.isInstance(object)) {
				container = (IContainer)object;
			}
		}
		initContainerName(container);
	}
	
	public void createControl(Composite parent) {
		initializeDialogUnits(parent);
		Composite composite = createClientArea(parent);		
		createLabel(composite);	
		createContainerField(composite);
		createProcessField(composite);
		setControl(composite);
		Dialog.applyDialogFont(composite);		
		setPageComplete(false);
	}

	private void createLabel(Composite composite) {
		Label label= new Label(composite, SWT.WRAP);
		label.setText("Choose a source folder and a Resource Group name.");
		GridData gd= new GridData();
		gd.widthHint= convertWidthInCharsToPixels(80);
		gd.horizontalSpan= 3;
		label.setLayoutData(gd);
	}

	private Composite createClientArea(Composite parent) {
		Composite composite= new Composite(parent, SWT.NONE);
		GridLayout layout= new GridLayout();
		layout.marginWidth= 0;
		layout.marginHeight= 0;
		layout.numColumns= 3;
		composite.setLayout(layout);
		return composite;
	}
	
	private void createContainerField(Composite parent) {
		Label label = new Label(parent, SWT.NONE);
		label.setText("Source folder : ");
		containerText = new Text(parent, SWT.BORDER);
		containerText.setText(containerName);
		containerText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				verifyContentsValid();
			}
		});
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		containerText.setLayoutData(gd);
		browseButton = new Button(parent, SWT.PUSH);
		browseButton.setText("Browse...");
		browseButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				chooseContainer();
			}			
		});
		gd = new GridData();
		gd.widthHint = convertWidthInCharsToPixels(15);
		browseButton.setLayoutData(gd);
	}
	
	private void createProcessField(Composite parent) {
		Label label = new Label(parent, SWT.NONE);
		label.setText("Resource Group name : ");
		processText = new Text(parent, SWT.BORDER);
		processText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				verifyContentsValid();
			}
		});
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		processText.setLayoutData(gd);		
	}
	
	private void chooseContainer() {
		WorkbenchContentProvider provider= new WorkbenchContentProvider();
		ILabelProvider labelProvider= new WorkbenchLabelProvider(); 
		ElementTreeSelectionDialog dialog= new ElementTreeSelectionDialog(getShell(), labelProvider, provider);
		dialog.setTitle("Folder Selection");
		dialog.setMessage("Choose a folder");
		dialog.setInput(ResourcesPlugin.getWorkspace());
		dialog.addFilter(createViewerFilter());
		dialog.open();
		initContainerName((IContainer)dialog.getFirstResult());
		containerText.setText(containerName);
	}

	private ViewerFilter createViewerFilter() {
		ViewerFilter filter= new ViewerFilter() {
			public boolean select(Viewer viewer, Object parent, Object element) {
				boolean filter = false;
				
				IProject project = null;
				if (IFolder.class.isInstance(element)) {
					IFolder folder = (IFolder) element;
					project = folder.getProject();
				} else if (IProject.class.isInstance(element)){
					project = (IProject) element;
				}
				if (project != null) {
					try {
						// Will only show projects that have the Watchpoint template nature or RType nature.
						filter = project.hasNature(WatchpointTemplateNature.NATURE_ID)
						      || project.hasNature(WatchpointResourceTypeNature.NATURE_ID);
					} catch (CoreException e) {}
				}
				return filter;
			}
		};
		return filter;
	}
	
	
	private void initContainerName(IContainer elem) {
		containerName = (elem == null) ? suggestedFolder : elem.getFullPath().makeRelative().toString(); 
	}
	
	private void verifyContentsValid() {
		if (!checkContainerPathValid()) {
			setErrorMessage("The folder does not exist.");
			setPageComplete(false);
		} else if (isProcessNameEmpty()) {
			setErrorMessage("Enter a name for the Resource Group.");
			setPageComplete(false);
		} else if (processExists()){
			setErrorMessage("A Resource Group with this name already exists.");
			setPageComplete(false);
		} else {
			setErrorMessage(null);
			setPageComplete(true);
		}
	}
	
	private boolean processExists() {
		IResourceGroupService service = ServiceFactory.createFactory().createResourceGroupService();
		return service.isNameExists(processText.getText());
	}
	
	private boolean isProcessNameEmpty() {
		String str = processText.getText();
		return str == null || "".equals(str);
	}
	
	private boolean checkContainerPathValid() {
		if ("".equals(containerText.getText())) {
			return false;
		}
		IPath path = new Path(containerText.getText());
		return workspaceRoot.getFolder(path).exists();
	}
	
	public String getProcessName() {
		return processText.getText(); // + ".par";
	}
	
	public IFolder getProcessFolder() {
		IPath path = new Path(containerText.getText());
		return workspaceRoot.getFolder(path);
	}
	
}
