/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package com.tandbergtv.watchpoint.studio.ui.wizard.template;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.wizards.newresource.BasicNewResourceWizard;

import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.util.FileUtil;

public class NewWatchPointTemplateWizard extends Wizard implements INewWizard {

	// The Logger
	private static final Logger logger = Logger.getLogger(NewWatchPointTemplateWizard.class);

	private static final String TEMPLATE_NAME_ATTRIBUTE_REGEX = "name=\"(.*?)\"";
	
	private IStructuredSelection selection;
	private NewProcessDefinitionWizardPage page;
	
	public NewWatchPointTemplateWizard() {
		setWindowTitle("New Template");
	}

	public void init(IWorkbench workbench, IStructuredSelection currentSelection) {
		this.selection= currentSelection;
	}
	
	public void addPages() {
		page = new NewProcessDefinitionWizardPage();
		addPage(page);
		page.init(selection);
	}
	
	public boolean performFinish() {
		try {
			IFolder folder = page.getProcessFolder();
			folder.create(true, true, null);
			IFile processDefinitionFile = folder.getFile(folder.getName() + "." + WorkflowTemplateDTO.TEMPLATE_DEFINITION_FILE_EXTENSION_NAME);
			processDefinitionFile.create(createInitialProcessDefinition(), true, null);
			IFile gpdFile = folder.getFile(WorkflowTemplateDTO.GPD_FILE_NAME);
			gpdFile.create(createInitialGpdInfo(), true, null);
			IFile imgFile = folder.getFile(WorkflowTemplateDTO.TEMPLATE_IMAGE_FILE_NAME);
			imgFile.create(createInitialImage(), true, null);
			IDE.openEditor(getActivePage(), processDefinitionFile);
			openPropertiesView();
			BasicNewResourceWizard.selectAndReveal(processDefinitionFile, getActiveWorkbenchWindow());
			return true;
		} catch (CoreException e) {
			e.printStackTrace();
			return false;
		}
	}

	private void openPropertiesView() {
		try {
			if (getActivePage().findView("org.eclipse.ui.views.PropertySheet") == null) {
				getActivePage().showView("org.eclipse.ui.views.PropertySheet");
			}
		} catch (PartInitException e) {
			e.printStackTrace();
		}
	}

	private IWorkbenchPage getActivePage() {
		return getActiveWorkbenchWindow().getActivePage();
	}

	private IWorkbenchWindow getActiveWorkbenchWindow() {
		return PlatformUI.getWorkbench().getActiveWorkbenchWindow();
	}
	
	private ByteArrayInputStream createInitialProcessDefinition() throws JavaModelException {
		byte[] templateBytes = new byte[0];
		String parName = page.getProcessFolder().getName();
		String processName = parName; //.substring(0, parName.indexOf(".par"));
		processName = "name=\"" + processName + "\"";
		InputStream in = this.getClass().getResourceAsStream(WorkflowTemplateDTO.TEMPLATE_DEFINITION_FILE_NAME);
		try {
			String template = new String(FileUtil.readFile(in));
			// Replaces the template name included in the "sample" file. 
			template = template.replaceFirst(TEMPLATE_NAME_ATTRIBUTE_REGEX, processName);
			templateBytes = template.getBytes();
		} catch (IOException e) {
			logger.error("Error loading bundled template file", e);
		}
		
		return new ByteArrayInputStream(templateBytes);
	}
	
	private ByteArrayInputStream createInitialGpdInfo() {
		byte[] gpdBytes = new byte[0];
		InputStream in = this.getClass().getResourceAsStream(WorkflowTemplateDTO.GPD_FILE_NAME);
		try {
			gpdBytes = FileUtil.readFile(in);
		} catch (IOException e) {
			logger.error("Error loading bundled template gpd", e);
		}
		return new ByteArrayInputStream(gpdBytes);
	}
	
	private ByteArrayInputStream createInitialImage() {
		byte[] imgBytes = new byte[0];
		InputStream in = this.getClass().getResourceAsStream(WorkflowTemplateDTO.TEMPLATE_IMAGE_FILE_NAME);
		try {
			imgBytes = FileUtil.readFile(in);
		} catch (IOException e) {
			logger.error("Error loading bundled image file", e);
		}
		return new ByteArrayInputStream(imgBytes);
	}
}
