package com.tandbergtv.watchpoint.studio.ui.wizard.template;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jdt.core.IClasspathContainer;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.ui.PreferenceConstants;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.WizardNewProjectCreationPage;
import org.eclipse.ui.part.ISetSelectionTarget;
import org.jbpm.gd.jpdl.Logger;
import org.jbpm.gd.jpdl.prefs.JbpmInstallation;
import org.jbpm.gd.jpdl.prefs.PreferencesManager;
import org.jbpm.gd.jpdl.util.JbpmClasspathContainer;

import com.tandbergtv.watchpoint.studio.builder.WatchpointTemplateNature;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;

/**
 * 	Wizard based on JBPM's org.jbpm.gd.jpdl.wizard.NewProcessDefinitionWizard.
 *  Code replication is required to change labels and messages.
 * 
 * @author <a href="francisco.neto@venturus.org.br">xfranet - Francisco Bento da Silva Neto</a>
 * @see org.jbpm.gd.jpdl.wizard.NewProcessDefinitionWizard
 *
 */
public class NewWatchpointTemplateProjectWizard extends Wizard implements INewWizard {

	private WizardNewProjectCreationPage mainPage;
	private ConfigureRuntimePage configureRuntimePage;
	private NewProcessProjectDetailsWizardPage coreJbpmPage;
	private IProject newProject;
	private IWorkbench workbench;
	
	public boolean canFinish() {
		return super.canFinish() && coreJbpmPage.combo.getItemCount() > 0 && coreJbpmPage.combo.getSelectionIndex() != -1;
	}
	   
	public void init(IWorkbench w, IStructuredSelection currentSelection) {
	    this.workbench = w;
		setNeedsProgressMonitor(true);
		setWindowTitle("New Template Project");
	}

	public void addPages() {
		addMainPage();
		if (!isRuntimeAvailable()) {
			addConfigureRuntimePage();
		}
		addChooseRuntimePage();
	}
	
	private boolean isRuntimeAvailable() {
		return !PreferencesManager.INSTANCE.getJbpmInstallationMap().isEmpty();
	}
	
	private void addConfigureRuntimePage() {
		configureRuntimePage = new ConfigureRuntimePage();
		addPage(configureRuntimePage);
	}

	private void addChooseRuntimePage() {
		coreJbpmPage = new NewProcessProjectDetailsWizardPage();
		addPage(coreJbpmPage);
	}

	private void addMainPage() {
		super.addPages();
		setWindowTitle("New Template Project");
		mainPage = new WizardNewProjectCreationPage("basicNewProjectPage");
		mainPage.setTitle("New Template Project");
		mainPage.setDescription("Create a new Template project.");		
		addPage(mainPage);
	}
	
	private IProject createNewProject() {
		final IProject newProjectHandle = mainPage.getProjectHandle();
		final IProjectDescription description = createProjectDescription(newProjectHandle);
		WorkspaceModifyOperation op = new WorkspaceModifyOperation() {
			protected void execute(IProgressMonitor monitor)
					throws CoreException {
				createProject(description, newProjectHandle, monitor);
			}
		};
		runProjectCreationOperation(op, newProjectHandle);
		return newProjectHandle;
	}
	
	private void addJRELibraries(IJavaProject javaProject) throws JavaModelException {		
		ArrayList<IClasspathEntry> entries = new ArrayList<IClasspathEntry>();
		entries.addAll(Arrays.asList(javaProject.getRawClasspath()));
		entries.addAll(Arrays.asList(PreferenceConstants.getDefaultJRELibrary()));
		javaProject.setRawClasspath((IClasspathEntry[])entries.toArray(new IClasspathEntry[entries.size()]), null);
	}

	private void addSourceFolders(IJavaProject javaProject) throws JavaModelException, CoreException {
		ArrayList<IClasspathEntry> entries = new ArrayList<IClasspathEntry>();
		entries.addAll(Arrays.asList(javaProject.getRawClasspath()));
		addSourceFolder(javaProject, entries, "src/main/java");
		addSourceFolder(javaProject, entries, "src/main/jpdl");
		javaProject.setRawClasspath((IClasspathEntry[])entries.toArray(new IClasspathEntry[entries.size()]), null);
	}

	private void addSourceFolder(IJavaProject javaProject, ArrayList<IClasspathEntry> entries, String path) throws CoreException {
		IFolder folder = javaProject.getProject().getFolder(path);
		createFolder(folder);
		IPackageFragmentRoot root = javaProject.getPackageFragmentRoot(folder);
		entries.add(JavaCore.newSourceEntry(root.getPath()));
	}
	
	private void createFolder(IFolder folder) throws CoreException {
		IContainer parent = folder.getParent();
		if (parent != null && !parent.exists() && parent instanceof IFolder) {
			createFolder((IFolder)parent);
		}
		folder.create(true, true, null);
	}
	
	private JbpmInstallation getJbpmInstallation() {
		return PreferencesManager.INSTANCE.getJbpmInstallation(getCoreJbpmName());
	}
	
	private void createJbpmLibraryContainer(IJavaProject javaProject) throws JavaModelException {
		JavaCore.setClasspathContainer(
				new Path("JBPM/" + getJbpmInstallation().name),
				new IJavaProject[] { javaProject },
				new IClasspathContainer[] { new JbpmClasspathContainer(javaProject, getJbpmInstallation()) },
				null);		
	}
	
	private String getCoreJbpmName() {
		return coreJbpmPage.getCoreJbpmName();
	}
	
	private void addJbpmLibraries(IJavaProject javaProject) throws JavaModelException {
		createJbpmLibraryContainer(javaProject);
		ArrayList<IClasspathEntry> entries = new ArrayList<IClasspathEntry>();
		entries.addAll(Arrays.asList(javaProject.getRawClasspath()));
		entries.add(JavaCore.newContainerEntry(new Path("JBPM/" + getJbpmInstallation().name)));
		javaProject.setRawClasspath((IClasspathEntry[])entries.toArray(new IClasspathEntry[entries.size()]), null);
	}
	
	private void createOutputLocation(IJavaProject javaProject) throws JavaModelException, CoreException {
		IFolder binFolder = javaProject.getProject().getFolder("bin");
		createFolder(binFolder);
		IPath outputLocation = binFolder.getFullPath();
		javaProject.setOutputLocation(outputLocation, null);
	}
	
	private void addJavaBuilder(IJavaProject javaProject) throws CoreException {
		IProjectDescription desc = javaProject.getProject().getDescription();
		ICommand command = desc.newCommand();
		command.setBuilderName(JavaCore.BUILDER_ID);
		desc.setBuildSpec(new ICommand[] { command });
		javaProject.getProject().setDescription(desc, null);
	}
	
	private void createJavaProject() { 
		try {
			newProject = createNewProject();
			newProject.setPersistentProperty(new QualifiedName("", "jbpmName"), getCoreJbpmName());
			IJavaProject javaProject = JavaCore.create(newProject);
			createOutputLocation(javaProject);
			addJavaBuilder(javaProject);
			setClasspath(javaProject);
			createInitialContent(javaProject);
			addNature(newProject);
			newProject.build(IncrementalProjectBuilder.FULL_BUILD, null);
		} catch (JavaModelException e) {
			ErrorDialog.openError(getShell(), "Problem creating java project", null, e.getStatus());			
		} catch (CoreException e) {
			ErrorDialog.openError(getShell(), "Problem creating java project", null, e.getStatus());						
		} catch (IOException e) {
			ErrorDialog.openError(getShell(), "Problem creating java project", null, null);									
		}
	}
	
	/**
	 * Add Template Nature on a project
	 * 
	 * @param project
	 *            to have Watchpoint Template nature added 
	 */
	private void addNature(IProject project) {
		try {
			IProjectDescription description = project.getDescription();
			String[] natures = description.getNatureIds();

			// Add the nature
			String[] newNatures = new String[natures.length + 1];
			System.arraycopy(natures, 0, newNatures, 0, natures.length);
			newNatures[natures.length] = WatchpointTemplateNature.NATURE_ID;
			description.setNatureIds(newNatures);
			project.setDescription(description, null);
		} catch (CoreException e) {
			ErrorDialog.openError(getShell(), "Problem creating java project", null, null);
		}
	}
	
	private void createInitialContent(IJavaProject javaProject) throws CoreException, JavaModelException, IOException {
		if (coreJbpmPage.checkbox.getSelection()) {
			createMessageActionHandler(javaProject);
			createSimpleProcessDefinition(javaProject);
		}
		createResourceGroupFolder(javaProject);
	}
	
	private void createResourceGroupFolder(IJavaProject javaProject) throws CoreException {
		IFolder folder = javaProject.getProject().getFolder("groups");
		if (!folder.exists()) {
			folder.create(true, true, null);
		}
	}
	
	private void createSimpleProcessDefinition(IJavaProject javaProject) throws CoreException, JavaModelException, IOException {
		JbpmInstallation jbpmInstallation = PreferencesManager.INSTANCE.getJbpmInstallation(getCoreJbpmName());
		if (jbpmInstallation == null) return;
		IFolder folder = javaProject.getProject().getFolder("src/main/jpdl/simple");
		if (!folder.exists()) {
			folder.create(true, true, null);
		}
		copyJbpmResource(this.getClass().getResourceAsStream(WorkflowTemplateDTO.TEMPLATE_DEFINITION_FILE_NAME), "simple." + WorkflowTemplateDTO.TEMPLATE_DEFINITION_FILE_EXTENSION_NAME, folder);
		copyJbpmResource(this.getClass().getResourceAsStream(WorkflowTemplateDTO.GPD_FILE_NAME), WorkflowTemplateDTO.GPD_FILE_NAME, folder);
		copyJbpmResource(this.getClass().getResourceAsStream(WorkflowTemplateDTO.TEMPLATE_IMAGE_FILE_NAME), WorkflowTemplateDTO.TEMPLATE_IMAGE_FILE_NAME, folder);
	}

	private void createMessageActionHandler(IJavaProject javaProject) throws JavaModelException, IOException {
		String resourceName = "org/jbpm/gd/jpdl/resource/MessageActionHandler.java.template";
		IFolder folder = javaProject.getProject().getFolder("src/main/java");
		IPackageFragmentRoot root = javaProject.getPackageFragmentRoot(folder);
		IPackageFragment pack = root.createPackageFragment("com.sample.action", true, null);
		InputStream stream = getClass().getClassLoader().getResourceAsStream(resourceName);
		byte[] content = readStream(stream);
		pack.createCompilationUnit("MessageActionHandler.java", new String(content), true, null);
	}
	
	private void copyJbpmResource(InputStream in, String fileName, IFolder destination) throws CoreException {
		IFile file = destination.getFile(fileName);
		file.create(in, true, null);			
	}

	private void setClasspath(IJavaProject javaProject) throws JavaModelException, CoreException {
		javaProject.setRawClasspath(new IClasspathEntry[0], null);
		addSourceFolders(javaProject);
		addJRELibraries(javaProject);
		addJbpmLibraries(javaProject);
		// Hack to overcome the problems of the classpath container not being created in the classpath.
		javaProject.getRawClasspath();
	}

	private IProjectDescription createProjectDescription(
			IProject newProjectHandle) {
		IPath newPath = null;
		if (!mainPage.useDefaults())
			newPath = mainPage.getLocationPath();
		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		IProjectDescription description = workspace
				.newProjectDescription(newProjectHandle.getName());
		description.setLocation(newPath);
		addJavaNature(description);
		return description;
	}
	
	private void addJavaNature(IProjectDescription description) {
		ArrayList<String> natures = new ArrayList<String>();
		natures.addAll(Arrays.asList(description.getNatureIds()));
		natures.add(JavaCore.NATURE_ID);
		description.setNatureIds((String[])natures.toArray(new String[natures.size()]));
	}

	private void runProjectCreationOperation(WorkspaceModifyOperation op,
			IProject newProjectHandle) {
		try {
			getContainer().run(false, true, op);
		} catch (InterruptedException e) {
			Logger.logError("InterruptedException while creating project", e);
		} catch (InvocationTargetException e) {
			Throwable t = e.getTargetException();
			if (t instanceof CoreException) {
				handleCoreException(newProjectHandle, (CoreException) t);
			} else {
				handleOtherProblem(t);
			}
		}
	}

	private void handleOtherProblem(Throwable t) {
		MessageDialog.openError(getShell(), "Creation Problems",
				"Internal error: " + t.getMessage());
	}

	private void handleCoreException(final IProject newProjectHandle,
			CoreException e) {
		if (e.getStatus().getCode() == IResourceStatus.CASE_VARIANT_EXISTS) {
			MessageDialog
					.openError(
							getShell(),
							"Creation Problems",
							"The underlying file system is case insensitive. There is an existing project which conflicts with '"
									+ newProjectHandle.getName() + "'.");
		} else {
			ErrorDialog.openError(getShell(), "Creation Problems", null, e
					.getStatus());
		}
	}

	void createProject(IProjectDescription description, IProject projectHandle,
			IProgressMonitor monitor) throws CoreException,
			OperationCanceledException {
		try {
			monitor.beginTask("", 2000);
			projectHandle.create(description, new SubProgressMonitor(monitor,
					1000));
			if (monitor.isCanceled()) {
				throw new OperationCanceledException();
			}
			projectHandle.open(IResource.BACKGROUND_REFRESH,
					new SubProgressMonitor(monitor, 1000));
		} finally {
			monitor.done();
		}
	}

	public IProject getNewProject() {
		return newProject;
	}

	public boolean performFinish() {
		if (!isRuntimeAvailable()) {
			String name = configureRuntimePage.nameText.getText();
			String location = configureRuntimePage.locationText.getText();
			String version = configureRuntimePage.versionText.getText();
			PreferencesManager.INSTANCE.initializeDefaultJbpmInstallation(name, location, version);
		}
		getContainer().updateButtons();
		createJavaProject();
		if (newProject == null) {
			getContainer().updateButtons();
			return false;
		}
		updatePerspective();
		selectAndReveal(newProject);
		getContainer().updateButtons();
		return true;
	}

	protected void updatePerspective() {
		try {
			IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
			if (page.findView("org.eclipse.ui.views.PropertySheet") == null) {
				page.showView("org.eclipse.ui.views.PropertySheet");
			}
		} catch (PartInitException e) {
			e.printStackTrace();
		}
	}

	protected void selectAndReveal(IResource newResource) {
		selectAndReveal(newResource, workbench.getActiveWorkbenchWindow());
	}

    private void selectAndReveal(IResource resource,
	           IWorkbenchWindow window) {
		if (!inputValid(resource, window)) return;   
		Iterator<IWorkbenchPart> itr = getParts(window.getActivePage()).iterator();
		while (itr.hasNext()) {
		    selectAndRevealTarget(
					window, 
					new StructuredSelection(resource), 
					getTarget((IWorkbenchPart)itr.next()));
		}
	}
	
	private boolean inputValid(IResource resource, IWorkbenchWindow window) {
		if (window == null || resource == null) return false;
		else if (window.getActivePage() == null) return false;
		else return true;
	}

	private void selectAndRevealTarget(IWorkbenchWindow window, final ISelection selection, ISetSelectionTarget target) {
		if (target == null) return;
		final ISetSelectionTarget finalTarget = target;
		window.getShell().getDisplay().asyncExec(new Runnable() {
		    public void run() {
		        finalTarget.selectReveal(selection);
		    }
		});
	}
	
	private ISetSelectionTarget getTarget(IWorkbenchPart part) {
        ISetSelectionTarget target = null;
        if (part instanceof ISetSelectionTarget) {
            target = (ISetSelectionTarget)part;
        }
        else {
            target = (ISetSelectionTarget)part.getAdapter(ISetSelectionTarget.class);
        }
		return target;		
	}

	private List<IWorkbenchPart> getParts(IWorkbenchPage page) {
		List<IWorkbenchPart> result = new ArrayList<IWorkbenchPart>();
		addParts(result, page.getViewReferences());
		addParts(result, page.getEditorReferences());
		return result;
	}
	
	private void addParts(List<IWorkbenchPart> parts, IWorkbenchPartReference[] refs) {
		for (int i = 0; i < refs.length; i++) {
           IWorkbenchPart part = refs[i].getPart(false);
           if (part != null) {
               parts.add(part);
           }
	    }		
	}
	
	private byte[] readStream(InputStream in) throws IOException {
		byte[] contents = null;
		int fileSize = 0;
		byte[] buffer = new byte[1024];
		int bytesRead = in.read(buffer);
		while (bytesRead != -1) {
			byte[] newContents = new byte[fileSize + bytesRead];
			if (fileSize > 0) {
				System.arraycopy(contents, 0, newContents, 0, fileSize);
			}
			System.arraycopy(buffer, 0, newContents, fileSize, bytesRead);
			contents = newContents;
			fileSize += bytesRead;
			bytesRead = in.read(buffer);
		}
		return contents;
	}
	
}