package com.tandbergtv.watchpoint.studio.util;

import java.util.Map.Entry;

import org.jbpm.gd.common.model.AbstractNamedElement;
import org.jbpm.gd.common.model.GenericElement;
import org.jbpm.gd.common.model.SemanticElement;
import org.jbpm.gd.jpdl.model.AbstractAsyncableTimerNode;
import org.jbpm.gd.jpdl.model.AbstractNode;
import org.jbpm.gd.jpdl.model.AbstractTimerNode;
import org.jbpm.gd.jpdl.model.Action;
import org.jbpm.gd.jpdl.model.CancelTimer;
import org.jbpm.gd.jpdl.model.Condition;
import org.jbpm.gd.jpdl.model.Controller;
import org.jbpm.gd.jpdl.model.CreateTimer;
import org.jbpm.gd.jpdl.model.Decision;
import org.jbpm.gd.jpdl.model.Delegation;
import org.jbpm.gd.jpdl.model.Description;
import org.jbpm.gd.jpdl.model.Event;
import org.jbpm.gd.jpdl.model.ExceptionHandler;
import org.jbpm.gd.jpdl.model.Fork;
import org.jbpm.gd.jpdl.model.Handler;
import org.jbpm.gd.jpdl.model.Join;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.ProcessState;
import org.jbpm.gd.jpdl.model.Script;
import org.jbpm.gd.jpdl.model.SubProcess;
import org.jbpm.gd.jpdl.model.Subject;
import org.jbpm.gd.jpdl.model.Task;
import org.jbpm.gd.jpdl.model.Text;
import org.jbpm.gd.jpdl.model.Timer;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.ui.model.AbstractTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.AssignNode;
import com.tandbergtv.watchpoint.studio.ui.model.AutomaticTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.LoopNode;
import com.tandbergtv.watchpoint.studio.ui.model.MailNode;
import com.tandbergtv.watchpoint.studio.ui.model.ManualTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.NodeGroup;
import com.tandbergtv.watchpoint.studio.ui.model.ResourceGroup;
import com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants;
import com.tandbergtv.watchpoint.studio.ui.model.WPEndState;
import com.tandbergtv.watchpoint.studio.ui.model.WPTransition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.util.NameValuePair;

/**
 * Utility class to clone semantic elements.<br> 
 * This is class is used mostly by the copy/paste feature for the template editor
 * @author Patrik
 *	
 */
public final class SemanticElementCloneUtil {
	public static SemanticElement cloneSemanticElement(SemanticElement element){
		if(element instanceof AutomaticTaskNode){
			AutomaticTaskNode atn = (AutomaticTaskNode)element;
			// Clone automatic task node object
			return atn.cloneAutomaticTaskNode();
		} else if(element instanceof NodeDefinition){
			NodeDefinition nd = (NodeDefinition)element;
			if(nd.getNodeType().equals(NodeDefinitionType.MessageNode)){
				// Clone node definition object
				return nd.cloneNodeDefinition();
			} else if( nd.getNodeType().equals(NodeDefinitionType.SuperState)){
				// Clone superstate object
				return nd.cloneNodeDefinition();
			}
		} else if(element instanceof Fork){
			Fork fork = (Fork) element;
			// Clone fork object
			return cloneFork(fork);
		} else if(element instanceof Join){
			Join join = (Join) element;
			// Clone join object
			return cloneJoin(join);
		} else if(element instanceof Decision){
			Decision decision = (Decision) element;
			// Clone decision object
			return cloneDecision(decision);
		} else if(element instanceof ManualTaskNode){
			ManualTaskNode manualTaskNode = (ManualTaskNode) element;
			// Clone manual task node object
			return manualTaskNode.cloneManualTaskNode();
		} else if(element instanceof WPEndState){
			WPEndState endState = (WPEndState) element;
			// Clone manual task node object
			return cloneWPEndState(endState);
		} else if(element instanceof MailNode){
			MailNode mailNode = (MailNode) element;
			// Clone mail node object
			return cloneMailNode(mailNode);
		} else if(element instanceof ProcessState){
			ProcessState processState = (ProcessState) element;
			return cloneProcessState(processState);
		} else if(element instanceof LoopNode){
			LoopNode loopNode = (LoopNode)element;
			return cloneLoopNode(loopNode);
		} else if (element instanceof AssignNode) {
		    AssignNode assignNode = (AssignNode)element;
		    return cloneAssignNode(assignNode);
		}

		return null;
	}
	
    private static void copyAbstractTaskNode(AbstractTaskNode source, AbstractTaskNode copy){
		/* Action, CancelTimer, CreateTimer and Script are set to the same class attribute */
		if(source.getAction() != null){
			copy.setAction( cloneAction(source.getAction()) );
		}
		if(source.getCancelTimer() != null){
			copy.setCancelTimer( cloneCancelTimer(source.getCancelTimer()) );
		}
		if(source.getCreateTimer() != null){
			copy.setCreateTimer( cloneCreateTimer(source.getCreateTimer()) );
		}
		if (source.getScript() != null) {
			copy.setScript( cloneScript(source.getScript()) );
		}
		
		copy.setAsync(source.getAsync());
		copy.setCreateTasks(source.getCreateTasks());
		copy.setDueDate(source.getDueDate());
		copy.setEndTasks(source.getEndTasks());
		copy.setResourceGroup( cloneResourceGroup(source.getResourceGroup()) );
		copy.setSignal(source.getSignal());
		
		for (Object task : source.getTasks()) {
			copy.addTask( cloneTask((Task) task) );
		}
	}
	
	private static void copyDelegation (Delegation source, Delegation copy){
		copy.setClassName(source.getClassName());
		copy.setConfigInfo(source.getConfigInfo());
		copy.setConfigType(source.getConfigType());
		
		for (Object genericElement : source.getGenericElements()) {
			copy.addGenericElement( cloneGenericElement((GenericElement)genericElement) );
		}
	}
	
	private static void copyAbstractTimerNode (AbstractTimerNode source, AbstractTimerNode copy){
		for (Object timer : source.getTimers()) {
			copy.addTimer( cloneTimer((Timer) timer) );
		}
	}
	
	private static void copyAbstractAsyncableTimerNode (AbstractAsyncableTimerNode source, AbstractAsyncableTimerNode copy){
		copy.setAsync(source.getAsync());
	}
	
	private static void copyAbstractNamedElement(AbstractNamedElement source, AbstractNamedElement copy){
		copy.setName(source.getName());
	}
	
	private static void copyAbstractNode(AbstractNode source, AbstractNode copy){
		for (Object event : source.getEvents()) {
			copy.addEvent( cloneEvent((Event)event) );
		}
		for (Object exceptionHandler : source.getExceptionHandlers()) {
			copy.addExceptionHandler( cloneExceptionHandler((ExceptionHandler) exceptionHandler) );
		}
		for (Object transition : source.getTransitions()) {
			copy.addTransition( ((WPTransition)transition).cloneTransition() );
		}
		copy.setDescription( cloneDescription(source.getDescription()) );
	}
	
	public static LoopNode cloneLoopNode(LoopNode source){
		LoopNode clone = null;
		if(source != null){
			clone = (LoopNode)source.getFactory().createById(source.getElementId());
			
			copyAbstractNamedElement(source, clone);
			copyAbstractNode(source, clone);
			
			clone.setExpression(source.getExpression());
			clone.setIndex(source.getIndex());
			clone.setInitialValue(source.getInitialValue());
			clone.setListName(source.getListName());
			clone.setLoopType(source.getLoopType());
			
			for (NodeElement element : source.getNodeElements()) {
				clone.addNodeElement((NodeElement) cloneSemanticElement(element));
			}
			
		}
		return clone;
	}
	
	public static ProcessState cloneProcessState(ProcessState source){
		ProcessState clone = null;
		if(source != null){
			clone = (ProcessState)source.getFactory().createById(source.getElementId());
			
			copyAbstractNamedElement(source, clone);
			copyAbstractAsyncableTimerNode(source, clone);
			copyAbstractNode(source, clone);
			copyAbstractTimerNode(source, clone);
			
			clone.setSubProcess( cloneSubProcess(source.getSubProcess()) );
			
			for (Object variable : source.getVariables()) {
				clone.addVariable( ((WPVariable)variable).cloneVariable() );
			}
		}
		return clone;
	}
	
	public static SubProcess cloneSubProcess(SubProcess source){
		SubProcess clone = null;
		if(source != null){
			clone = (SubProcess)source.getFactory().createById(source.getElementId());
			
			copyAbstractNamedElement(source, clone);
			
			clone.setVersion(source.getVersion());
		}
		return clone;
	}
	
	public static MailNode cloneMailNode(MailNode source){
		MailNode clone = null;
		if(source != null){
			clone = (MailNode)source.getFactory().createById(source.getElementId());
			
			copyAbstractNamedElement(source, clone);
			copyAbstractAsyncableTimerNode(source, clone);
			copyAbstractNode(source, clone);
			copyAbstractTimerNode(source, clone);
			
			clone.setActors(source.getActors());
			clone.setAttachment(source.getAttachment());
			clone.setSubject( cloneSubject(source.getSubject()) );
			clone.setTemplate(source.getTemplate());
			clone.setText( cloneText(source.getText()) );
			clone.setTo(source.getTo());
		}
		return clone;
	}
	
	public static Subject cloneSubject(Subject source){
		Subject clone = null;
		if(source != null){
			clone = (Subject)source.getFactory().createById(source.getElementId());
			
			clone.setSubject(source.getSubject());
		}
		return clone;
	}
	
	public static Text cloneText(Text source){
		Text clone = null;
		if(source != null){
			clone = (Text)source.getFactory().createById(source.getElementId());
			
			clone.setText(source.getText());
		}
		return clone;
	}
	
	public static Fork cloneFork(Fork source){
		Fork clone = null;
		if(source != null){
			clone = (Fork)source.getFactory().createById(source.getElementId());
			
			copyAbstractNamedElement(source, clone);
			copyAbstractNode(source, clone);
			copyAbstractTimerNode(source, clone);
			
			clone.setAsync(source.getAsync());
			
			for (Object script : source.getScripts()) {
				clone.addScript( cloneScript((Script) script) );
			}
		}
		return clone;
	}
	
	public static Join cloneJoin(Join source){
		Join clone = null;
		if(source != null){
			clone = (Join)source.getFactory().createById(source.getElementId());
			
			copyAbstractNamedElement(source, clone);
			copyAbstractNode(source, clone);
			copyAbstractTimerNode(source, clone);
			
			clone.setAsync(source.getAsync());
		}
		return clone;
	}
	
	public static Decision cloneDecision(Decision source){
		Decision clone = null;
		if(source != null){
			clone = (Decision)source.getFactory().createById(source.getElementId());
			
			copyAbstractNamedElement(source, clone);
			copyAbstractNode(source, clone);
			
			clone.setAsync(source.getAsync());
			clone.setExpression(source.getExpression());
			clone.setHandler( cloneHandler(source.getHandler()) );
			
		}
		return clone;
	}
	
	public static NodeGroup cloneNodeGroup(NodeGroup source){
		NodeGroup clone = null;
		if(source != null){
			clone = (NodeGroup)source.getFactory().createById(source.getElementId());
			
			copyAbstractNamedElement(source, clone);
			copyAbstractNode(source, clone);
			copyAbstractTimerNode(source, clone);
			
			clone.setAsync(source.getAsync());
			clone.setDueDate(source.getDueDate());
			clone.setResourceGroup(source.getResourceGroup());
			
			for (NodeElement element : source.getNodeElements()) {
				clone.addNodeElement((NodeElement) cloneSemanticElement(element));
			}
		}
		return clone;
	}
	
	public static ManualTaskNode cloneManualTaskNode(ManualTaskNode source){
		ManualTaskNode clone = null;
		if(source != null){
			clone = (ManualTaskNode)source.getFactory().createById(source.getElementId());
			
			copyAbstractNamedElement(source, clone);
			copyAbstractNode(source, clone);
			copyAbstractTimerNode(source, clone);
			copyAbstractTaskNode(source, clone);
			
		}
		return clone;
	}
	
	public static AutomaticTaskNode cloneAutomaticTaskNode(AutomaticTaskNode source){
		AutomaticTaskNode clone = null;
		if(source != null){
			clone = (AutomaticTaskNode)source.getFactory().createById(source.getElementId());
			
			copyAbstractNamedElement(source, clone);
			copyAbstractNode(source, clone);
			copyAbstractTimerNode(source, clone);
			copyAbstractTaskNode(source, clone);
			
			if(source.getRollbackVariables() != null && !source.getRollbackVariables().isEmpty()){
				for (Object variable : source.getRollbackVariables()) {
					clone.getRollbackVariables().add( ((WPVariable)variable).cloneVariable() );
				}
			}
			
			if(source.getUid() != null && !source.getUid().isEmpty()){
				clone.setUid(source.getUid());
			}
			
			for (Entry<String, String> entry : source.getCompositeKeys().entrySet()) {
				clone.setCompositeKey(entry.getKey(), entry.getValue());
			}
			
			for (Object variable : source.getVariables()) {
				clone.addVariable( ((WPVariable)variable).cloneVariable() );
			}
		}
		return clone;
	}
	
	public static NodeDefinition cloneNodeDefinition(NodeDefinition source){
		NodeDefinition clone = null;
		if(source != null){
			clone = (NodeDefinition)source.getFactory().createById(source.getElementId());
			
			copyAbstractNode(source, clone);
			
			clone.setDefinitionName(source.getDefinitionName());
			clone.setDueDate(source.getDueDate());
			clone.setId(source.getId());
			clone.setIncoming(source.isIncoming());
			clone.setName(source.getName());
			
			if(source.getNode() instanceof AutomaticTaskNode){
				AutomaticTaskNode atn = (AutomaticTaskNode) source.getNode();
				clone.setNode(atn.cloneAutomaticTaskNode());
			}
			if(source.getNode() instanceof NodeGroup){
				NodeGroup ng = (NodeGroup) source.getNode();
				clone.setNode(ng.cloneNodeGroup());
			}
			
			clone.setNodeType(source.getNodeType());
			
			if(source.getResourceGroup() != null){
				clone.setResourceGroup( cloneResourceGroup(source.getResourceGroup()) );
			}
			clone.setResourceType(source.getResourceType());
			clone.setUid(source.getUid());
			
			if(source.getRollbackVariables() != null && !source.getRollbackVariables().isEmpty()){
				for (Object variable : source.getRollbackVariables()) {
					clone.getRollbackVariables().add( ((WPVariable)variable).cloneVariable() );
				}
			}
			
			for (Entry<String, String> entry : source.getCompositeKeys().entrySet()) {
				clone.setCompositeKey(entry.getKey(), entry.getValue());
			}
			for (Object nameValuePair : source.getConstants()) {
				NameValuePair constant = (NameValuePair) nameValuePair;
				clone.setConstantValue(constant.getName(), constant.getValue());
			}
			for (Object variable : source.getVariables()) {
				clone.addVariable( ((WPVariable)variable).cloneVariable() );
			}
		}
		return clone;
	}
	
	public static WPTransition cloneWPTransition(WPTransition source){
		WPTransition clone = null;
		if(source != null){
			clone = (WPTransition) source.getFactory().createById(source.getElementId());
			
			if(source.getCondition() != null){
				clone.setCondition( cloneCondition(source.getCondition()) );
			}
			clone.setDescription( cloneDescription(source.getDescription()) );
			clone.setName(source.getName());
			clone.setSource(source.getSource());
			clone.setTo(source.getTo());
			clone.setType(source.getType());
			
			for (Object actionElement : source.getActionElements()) {
				if(actionElement instanceof Action){
					clone.addActionElement( cloneAction((Action)actionElement) );
				}
			}
			
			for (Object exceptionHandler : source.getExceptionHandlers()) {
				clone.addExceptionHandler( cloneExceptionHandler((ExceptionHandler) exceptionHandler) );
			}
		}
		return clone;
	}
	
	public static ResourceGroup cloneResourceGroup(ResourceGroup source){
		ResourceGroup clone = null;
		if(source != null){
			clone = (ResourceGroup)source.getFactory().createById(source.getElementId());
			
			copyAbstractNamedElement(source, clone);
		}
		return clone;
	}
	
	
	public static Action cloneAction(Action source){
		Action clone = null;
		if(source != null){
			clone = (Action)source.getFactory().createById(source.getElementId());
			
			copyDelegation(source, clone);
			
			clone.setAcceptPropagatedEvents(source.getAcceptPropagatedEvents());
			clone.setAsync(source.getAsync());
			clone.setExpression(source.getExpression());
			clone.setName(source.getName());
			clone.setRefName(source.getRefName());
		}
		return clone;
	}
	
	public static CancelTimer cloneCancelTimer(CancelTimer source){
		CancelTimer clone = null;
		if(source != null){
			clone = (CancelTimer)source.getFactory().createById(source.getElementId());
			clone.setName(source.getName());
		}
		return clone;
	}
	
	public static CreateTimer cloneCreateTimer(CreateTimer source){
		CreateTimer clone = null;
		if(source != null){
			clone = (CreateTimer)source.getFactory().createById(source.getElementId());
			
			clone = (CreateTimer)cloneTimer(clone);
		}
		return clone;
	}
	
	public static Script cloneScript(Script source){
		Script clone = null;
		if(source != null){
			clone = (Script)source.getFactory().createById(source.getElementId());
			
			clone.setAcceptPropagatedEvents(source.getAcceptPropagatedEvents());
			clone.setName(source.getName());
			clone.setScript(source.getScript());
		}
		return clone;
	}
	
	public static Description cloneDescription(Description source){
		Description clone = null;
		if(source != null){
			clone = (Description)source.getFactory().createById(SemanticElementConstants.DESCRIPTION_SEID);
			clone.setDescription(source.getDescription());
		}
		return clone;
	}
	
	public static GenericElement cloneGenericElement(GenericElement source){
		GenericElement clone = null;
		if(source != null){
			clone = (GenericElement)source.getFactory().createById(source.getElementId());
			
			clone.setName(source.getName());
			clone.setValue(source.getValue());
			
			for (Object genericElement : source.getGenericElements()) {
				clone.addGenericElement( cloneGenericElement((GenericElement)genericElement) );
			}
		}
		return clone;
	}
	
	public static Timer cloneTimer(Timer source){
		Timer clone = null;
		if(source != null){
			clone = (Timer) source.getFactory().createById(source.getElementId());
			
			copyAbstractNamedElement(source, clone);
			
			if(source.getAction() != null){
				clone.setAction( cloneAction( source.getAction()) );
			}
			if (source.getScript() != null) {
				clone.setScript( cloneScript(source.getScript()) );
			}
			
			clone.setDueDate(source.getDueDate());
			clone.setRepeat(source.getRepeat());
			clone.setTransition(source.getTransition());
		}
		return clone;
	}
	
	public static ExceptionHandler cloneExceptionHandler(ExceptionHandler source){
		ExceptionHandler clone = null;
		if(source != null){
			clone = (ExceptionHandler) source.getFactory().createById(source.getElementId());
			
			clone.setExceptionClass(source.getExceptionClass());
						for (Object actionElement : source.getActionElements()) {
				if(actionElement instanceof Action){
					clone.addActionElement( cloneAction((Action)actionElement) );
				}
			}
		}
		return clone;
	}
	
	public static Condition cloneCondition(Condition source){
		Condition clone = null;
		if(source != null){
			clone = (Condition) source.getFactory().createById(source.getElementId());
			
			clone.setExpression(source.getExpression());
			clone.setScript(source.getScript());
		}
		return clone;
	}
	
	public static Task cloneTask(Task source){
		Task clone = null;
		if(source != null){
			clone = (Task) source.getFactory().createById(source.getElementId());
			
			clone.setAssignment(source.getAssignment());
			clone.setBlocking(source.getBlocking());
			
			if(source.getController() != null){
				clone.setController( cloneController(source.getController()) );
			}
			
			clone.setDescription( cloneDescription(source.getDescription()) );
			clone.setDueDate(source.getDueDate());
			clone.setName(source.getName());
			clone.setNotify(source.getNotify());
			clone.setPriority(source.getPriority());
			clone.setReminder(source.getReminder());
			clone.setSignalling(source.getSignalling());
			clone.setSwimlane(source.getSwimlane());
			
			for ( Object entry : source.getProperties().entrySet() ) {
				Entry<String, String> mapEntry = (Entry<String, String>)entry;
				clone.setProperty(mapEntry.getKey(), mapEntry.getValue());
			}
			
			for (Object timer : source.getTimers()) {
				clone.addTimer( cloneTimer((Timer) timer) );
			}
		}
		return clone;
	}
	
	public static Controller cloneController(Controller source){
		Controller clone = null;
		if(source != null){
			clone = (Controller) source.getFactory().createById(source.getElementId());
			
			copyDelegation(source, clone);
			
			for (Object variable : source.getVariables()) {
				clone.addVariable( ((WPVariable)variable).cloneVariable() );
			}
		}
		return clone;
	}
	
	public static Event cloneEvent(Event source){
		Event clone = null;
		if(source != null){
			clone = (Event) source.getFactory().createById(source.getElementId());
			
			clone.setType(source.getType());
			for (Object actionElement : source.getActionElements()) {
				if(actionElement instanceof Action){
					clone.addActionElement( cloneAction((Action)actionElement) );
				}
			}
		}
		return clone;
	}
	
	public static Handler cloneHandler(Handler source){
		Handler clone = null;
		if(source != null){
			clone = (Handler) source.getFactory().createById(source.getElementId());
			
			copyDelegation(source, clone);
		}
		return clone;
	}
	
	public static WPEndState cloneWPEndState(WPEndState source){
		WPEndState clone = null;
		if(source != null){
			clone = (WPEndState) source.getFactory().createById(source.getElementId());
			
			copyAbstractNamedElement(source, clone);
			copyAbstractNode(source, clone);
			
		}
		return clone;
	}

    private static SemanticElement cloneAssignNode(AssignNode source) {
        AssignNode clone = null;
        if(source != null){
            clone = (AssignNode)source.getFactory().createById(source.getElementId());

            copyAbstractNamedElement(source, clone);
            copyAbstractNode(source, clone);
            copyAbstractTimerNode(source, clone);
            copyAbstractTaskNode(source, clone);
        }
        return clone;
    }

}
