package org.jbpm.gd.common.xml;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.jbpm.gd.common.model.SemanticElement;
import org.jbpm.gd.common.model.SemanticElementFactory;
import org.jbpm.gd.common.registry.RegistryRegistry;
import org.jbpm.gd.common.registry.XmlAdapterRegistry;
import org.jbpm.gd.jpdl.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

/**
 * 		Class intended to speed up the UI Model parsing based on the XML DOM model, instead of the eclipse SSE API.		
 * 		Although its not recommended, this class had to be present in this package due to the method org.jbpm.gd.common.xml.XmlAdapter.setFactory(XmlAdapterFactory) which is
 * package-protected.
 * 
 * @author <a href="mailto:francisco.bento.silva.neto@ericsson.com">efrasio - Francisco Bento da Silva Neto</a>
 *
 */
public class SimpleDOMAdapterFactory extends XmlAdapterFactory {

	private XmlAdapterRegistry xmlAdapterRegistry;
	private Document document;
	
	public SimpleDOMAdapterFactory(Document document, SemanticElementFactory elementFactory) {
		super(document, elementFactory);
		xmlAdapterRegistry = RegistryRegistry.getXmlAdapterRegistry(elementFactory.getEditorId());
		this.document = document;
	}

	@Override
	public XmlAdapter adapt(Node node) {
		return (XmlAdapter) createAdapter(node);
	}
	
	/**
	 * 		Analog to org.jbpm.gd.common.xml.XmlAdapterFactory.createAdapter(INodeNotifier) but based on the DOM model instead of SSE model.
	 * 
	 * @see org.jbpm.gd.common.xml.XmlAdapterFactory.createAdapter(INodeNotifier)
	 * @param target
	 * @return
	 */
	protected XmlAdapter createAdapter(Node target) {
		XmlAdapter result = null;
		try {
			IConfigurationElement element = xmlAdapterRegistry.getConfigurationElementByXmlNode((Node)target);
			if (element != null) {
				result = (XmlAdapter)element.createExecutableExtension("adapterClass");
			} else if (((Node)target).getNodeType() != Node.TEXT_NODE){
				result = new GenericElementXmlAdapter();
			}
			if (result != null) {
				result.setNode((Node)target);
				result.setFactory(this);
			}
		} catch (CoreException e) {
			throw new RuntimeException("Creation of executable extension failed", e);
		}
		return result;
	}
	
	/**
	 * 		Analog to org.jbpm.gd.common.xml.XmlAdapterFactory.calculateElementName(IConfigurationElement, SemanticElement) but based on the DOM model instead of SSE model.
	 * 
	 * @see org.jbpm.gd.common.xml.XmlAdapterFactory.calculateElementName(IConfigurationElement, SemanticElement)
	 * @param target
	 * @return
	 */
	private String calculateElementName(IConfigurationElement configurationElement, SemanticElement semanticElement) {
		String elementName = configurationElement.getAttribute("xmlElement");
		if (elementName == null) {
			String nameProviderClass = configurationElement.getAttribute("nameProvider");
			if (nameProviderClass != null) {
				try {
					XmlAdapterNameProvider nameProvider = (XmlAdapterNameProvider)configurationElement.createExecutableExtension("nameProvider");
					if (nameProvider != null) {
						elementName = nameProvider.getName(semanticElement);
					}
				}
				catch (CoreException e) {
					Logger.logError("Problem creating nameProvider for " + semanticElement.getElementId(), e);
				}
			}
		}
		return elementName;
	}
	
	@Override
	public XmlAdapter createAdapterFromModel(SemanticElement semanticElement) {
		IConfigurationElement configurationElement = 
			xmlAdapterRegistry.getConfigurationElementBySemanticElementId(semanticElement.getElementId());
		String elementName = calculateElementName(configurationElement, semanticElement);
		if (elementName != null) {
			return createAdapter(configurationElement, elementName);
		} else {
			return null;
		}
	}
	
	/**
	 * 		Analog to org.jbpm.gd.common.xml.XmlAdapterFactory.createAdapter(IConfigurationElement, String) but based on the DOM model instead of SSE model.
	 * 
	 * @see org.jbpm.gd.common.xml.XmlAdapterFactory.createAdapter(IConfigurationElement, String)
	 * @param target
	 * @return
	 */
	private XmlAdapter createAdapter(IConfigurationElement configurationElement, String elementName) {
		XmlAdapter result = null;
		try {
			result = (XmlAdapter) configurationElement.createExecutableExtension("adapterClass");

			Node element = document.createElement(elementName);
			result.setFactory(this);
			result = (XmlAdapter)adapt(element);
			result.setNode((Node)element);
		} catch (CoreException e) {
		    Logger.logError("Unable to create XML Adapter for " + elementName, e);
		}
		return result;
	}
	
	
}