package com.tandbergtv.watchpoint.studio.builder;

import static com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType.MessageNode;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.JPFConstants.ATTR_LIBRARY_PATH;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.JPFConstants.ELEM_LIBRARY;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants.PLUGIN_NODE_FOLDER;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants.RESOURCE_TYPE_BUILD_PROPERTIES_FILENAME;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants.RESOURCE_TYPE_PLUGIN_FILE_NAME;
import static com.tandbergtv.watchpoint.studio.ui.util.Utility.RESOURCE_TYPE_EDITOR_ID;
import static com.tandbergtv.watchpoint.studio.ui.wizard.export.ResourceTypeExportConstants.BIN_INCLUDES_KEY;
import static org.eclipse.ui.IWorkbenchPage.MATCH_ID;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.statushandlers.StatusManager;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.external.fs.NodeDefinitionParser;
import com.tandbergtv.watchpoint.studio.external.wpexport.impl.JPFConstants;
import com.tandbergtv.watchpoint.studio.external.wpexport.impl.JPFExportUtil;
import com.tandbergtv.watchpoint.studio.external.wpexport.impl.ResourceTypeDocumentExporter;
import com.tandbergtv.watchpoint.studio.service.INodeDefinitionService;
import com.tandbergtv.watchpoint.studio.service.IResourceGroupService;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.ResourceTypeEditor;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.ResourceTypeEditorInput;
import com.tandbergtv.watchpoint.studio.ui.model.DataCollectionManager;
import com.tandbergtv.watchpoint.studio.util.ResourceTypePluginFileGenerator;
import com.tandbergtv.watchpoint.studio.util.ValidationMessageUtil;
import com.tandbergtv.watchpoint.studio.util.XMLDocumentUtility;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;

public class WatchpointResourceTypeVisitor implements IResourceDeltaVisitor, IResourceVisitor {

	// The Logger
	private static final Logger logger = Logger.getLogger(WatchpointResourceTypeVisitor.class);

	public static final String MARKER_TYPE = "org.jbpm.gd.jpdl.resourceTypeProblem";

	public static final String JAR_EXTENSION = ".jar";

	/**
	 * This method is invoked when a resource changes.
	 * 
	 * @see org.eclipse.core.resources.IResourceDeltaVisitor#visit(org.eclipse.core.resources.IResourceDelta)
	 */
	@Override
    public boolean visit(IResourceDelta delta) throws CoreException {
		IResource resource = delta.getResource();

		// Verify if its not the "classes" or "bin" folder where the java compiler creates the compiled files.
		if (isClassesFolder(resource)) {
            return false;
        }

		if (isResourceTypePluginXML(resource)) {
			switch (delta.getKind()) {
			case IResourceDelta.ADDED:
				update(resource);
				break;
			case IResourceDelta.CHANGED:
				update(resource);
				break;
			case IResourceDelta.REMOVED:
				delete(resource);
				break;
			}
		} else if (isNode(resource)) {
			switch (delta.getKind()) {
			case IResourceDelta.ADDED:
				nodeAdded(resource);
				break;
			case IResourceDelta.CHANGED:
				nodeUpdated(resource);
				break;
			case IResourceDelta.REMOVED:
				nodeDeleted(resource);
				break;
			}
		} else if (isBuildProperties(resource)) {
			// Updates plugin.xml jars
			updatePluginJars(resource);
		} else {
			// Is not resource type, call Resource Group
			new ResourceGroupVisitor().visit(delta);
		}

		return true;
	}

	/**
	 * Invoked when a FULL BUILD is executed.
	 * 
	 * @see org.eclipse.core.resources.IResourceVisitor#visit(org.eclipse.core.resources.IResource)
	 */
	@Override
    public boolean visit(IResource resource) {
		// Verify if its not the "classes" or "bin" folder where the java compiler put the compiled files.
		if (isClassesFolder(resource)) {
            return false;
        }

		if (isResourceTypePluginXML(resource)) {
            update(resource);
        } else {
            new ResourceGroupVisitor().visit(resource);
        }

		//continue visiting children.
		return true;
	}

	/**
	 * Updates the resource type represented by the resource. Conditions:
	 * - general properties changed
	 * - adaptor properties changed
	 * - management properties changed
	 * - message added, updated (description), deleted
	 * 
	 * @param resource the plugin.xml
	 */
	protected void update(IResource resource) {
		IFile file = (IFile) resource;

		try {
			clean(getResourceType(file.getProject()), file);
			org.jbpm.gd.jpdl.Logger.logInfo("Finished building resource type " + file.getProject().getName());
		} catch (Exception e) {
			logger.error("Error building resource type " + file.getProject().getName(), e);
			handleResourceTypeBuildError(file, e.getMessage());
		}
	}
	
	protected void clean(ResourceType rt, IFile file) {
		logger.debug("Start " + file.getProject().getName());

		IResourceTypeService service = getResourceTypeService();
		ResourceType previousResourceType = service.getResourceTypeByPath(rt.getPath());

		if (previousResourceType == null) {
			createNewResourceType(rt);
		} else {
			cleanResourceTypeMessagesAndNodes(previousResourceType);
			rt.setId(previousResourceType.getId());

			if (!rt.getMessages().isEmpty()) {
                createOrUpdateMessages(rt);
            }
			
			if (!rt.getNodes().isEmpty()) {
                createOrUpdateNodeDefinitions(rt);
            }
			
			updateRuntimeTagInPluginFile(file.getProject());
			service.updateResourceTypeInDataBase(rt);
		}

		validateResourceType(file, rt);
		validateDependencies(rt);

        ResourceTypeDocumentExporter.export(rt);

        refreshEditor(rt);
	}

    private void updateRuntimeTagInPluginFile(IProject project){
		IFile propertiesFile = project.getFile(RESOURCE_TYPE_BUILD_PROPERTIES_FILENAME);
		updatePluginJars(propertiesFile);
	}
	
	/**
	 * Called when a node's XML file is created. We have to handle superstate creation only. 
	 * Regular node create is handled by the resource type update instead (because plugin.xml
	 * changes).
	 * 
	 * @param resource
	 */
	private void nodeAdded(IResource resource) {
		IFile file = (IFile) resource;
		ResourceType rt = getResourceType(file.getProject().getName());
		
		try {
			NodeDefinitionDTO dto = 
				new NodeDefinitionParser(null, rt.getMessages()).parse(file.getLocation().toFile());
			
			if (dto.getType() == MessageNode) {
                return;
            }

			/* Fix database relationships for the calls to work */
			dto.setResourceType(rt);
			getNodeDefinitionService().createNodeDefinition(dto);
			rt.addNode(dto);
			getResourceTypeService().updateResourceTypeInDataBase(rt);

			validateResourceType(file, rt);
			validateDependencies(rt);
		} catch (Exception e) {
			logger.error("Error building resource type " + file.getProject().getName(), e);
			handleResourceTypeBuildError(file, e.getMessage());
		}
	}
	
	private void nodeUpdated(IResource resource) throws CoreException {
		IFile file = (IFile) resource;
		ResourceType rt = getResourceType(file.getProject().getName());
		
		try {
			/* IFile contents are stale - not sure why....have to read the node from filesystem */
			NodeDefinitionDTO dto = getNodeDefinition(file);

			/* Fix database relationships for the calls to work */
			dto.setId(rt.getNode(dto.getName()).getId());
			dto.setResourceType(rt);

			getNodeDefinitionService().deleteNodeDefinition(dto.getId(), true);
			rt.removeNode(dto);

			getNodeDefinitionService().createNodeDefinition(dto);
			rt.addNode(dto);
			getResourceTypeService().updateResourceTypeInDataBase(rt);

			validateResourceType(file, rt);
			validateDependencies(rt);
		} catch (Exception e) {
			logger.error("Error building resource type " + file.getProject().getName(), e);
			handleResourceTypeBuildError(file, e.getMessage());
		}
	}
	
	/**
	 * Called when a node's XML file is deleted. We have to handle superstate deletion only. 
	 * Regular node deletion is handled by the resource type update instead (because plugin.xml
	 * changes on message deletion, see ).
	 * 
	 * @param resource
	 */
	private void nodeDeleted(IResource resource) {
		IFile file = (IFile) resource;
		IProject project = file.getProject();
		ResourceType rt = getResourceType(project.getName());
		
		try {
			String name = file.getLocation().removeFileExtension().lastSegment();
			NodeDefinitionDTO dto = rt.getNode(name);

			if (dto == null || dto.getType() == MessageNode) {
                return;
            }
			
			dto.setResourceType(rt);
			getNodeDefinitionService().deleteNodeDefinition(dto.getId(), true);
			
			rt.removeNode(dto);
			getResourceTypeService().updateResourceTypeInDataBase(rt);

			validateResourceType(project.getFile(RESOURCE_TYPE_PLUGIN_FILE_NAME), rt);
			validateDependencies(rt);
		} catch (Exception e) {
			logger.error("Error building resource type " + project.getName(), e);
			handleResourceTypeBuildError(project.getFile(RESOURCE_TYPE_PLUGIN_FILE_NAME), e.getMessage());
		}
	}
	
	private ResourceType getResourceType(IProject project) {
		return getResourceTypeService().getResourceTypeFromFileSystem(project.getName());
	}
	
	private ResourceType getResourceType(String name) {
		return getResourceTypeService().getResourceTypeByName(name);
	}
	
	private NodeDefinitionDTO getNodeDefinition(IFile file) throws CoreException {
		ResourceType rt = getResourceType(file.getProject());
		String name = file.getLocation().removeFileExtension().lastSegment();
		
		return rt.getNode(name);
	}

	/**
	 * Handle a resource type error during the build. <br>
	 * Adds a marker to the plugin.xml file in the resource type project.
	 * @param file The given resource type plugin.xml
	 */
	private void handleResourceTypeBuildError(IFile file, String reason) {
		try {
			IMarker marker = file.createMarker(MARKER_TYPE);
			String msg = "Error building Resource Type: " + reason;
			String resourceTypeName = file.getProject().getName();
			marker.setAttribute(IMarker.MESSAGE, msg + resourceTypeName);
			marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
			marker.setAttribute(IMarker.LINE_NUMBER, 1);
		} catch (CoreException e) {
			logger.warn("Error adding marker", e);
		}
	}

	/*
	 * 		Re-Validate all ResourceType dependent entities.
	 * 
	 * @param resourceType
	 */
	private void validateDependencies(ResourceType resourceType) {
		/* Groups must be validated first, since dependency is:
		 * Resource Type <- Resource Group <- Template */
		
		/* Validate the Groups */
		IResourceGroupService groupService = getResourceGroupService();
		ResourceGroupVisitor groupVisitor = new ResourceGroupVisitor();
		List<ResourceGroup> referencingGroups = groupService.getResourceGroupsByResourceType(resourceType.getSystemId());
		HashSet<ResourceGroup> groups = new HashSet<ResourceGroup>(referencingGroups);

		for (ResourceGroup referencingGroup : groups) {
			IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(referencingGroup.getPath()));
			groupVisitor.validateResourceGroup(file, referencingGroup);
		}
		
		/* Validate the Templates. */
		IWorkflowTemplateService templateService = getTemplateService();
		WatchpointTemplateVisitor templateVisitor = new WatchpointTemplateVisitor();
		List<WorkflowTemplateDTO> referencingTemplates = templateService.getTemplatesByResourceTypeUsage(resourceType.getSystemId());
		Set<WorkflowTemplateDTO> set = new HashSet<WorkflowTemplateDTO>(referencingTemplates);
		
		for (WorkflowTemplateDTO referencingTemplate : set) {
			IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(referencingTemplate.getPath()));
			templateVisitor.validateTemplate(file, referencingTemplate);
		}
	}

	private void refreshEditor(ResourceType parsed) {
		final ResourceType resourceType = parsed;
		PlatformUI.getWorkbench().getDisplay().asyncExec(new Runnable() {
			@Override
            public void run() {
				IWorkbenchWindow dwindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
				// may be null during eclipse shutdown...
				if (dwindow == null) {
                    return;
                }

				IWorkbenchPage page = dwindow.getActivePage();
				
				if (page == null) {
                    return;
                }
				
				IEditorReference[] editors = page.findEditors(null, RESOURCE_TYPE_EDITOR_ID, MATCH_ID);
				
				if (editors.length > 0) {
					for (IEditorReference editor2 : editors) {
						ResourceTypeEditor editor = (ResourceTypeEditor) editor2.getEditor(true).getSite().getPart();
						ResourceTypeEditorInput editorInput = (ResourceTypeEditorInput) editor.getEditorInput();

						if (editorInput.getResourceType().getPath().equals(resourceType.getPath())) {
							editor.updateTitleImage();
							editor.refreshEditor();
						}
					}
				}
			}
		});
	}

	private void cleanResourceTypeMessagesAndNodes(ResourceType rt) {
		INodeDefinitionService nodeService = getNodeDefinitionService();
		IResourceTypeService rtService = getResourceTypeService();
		
		for (NodeDefinitionDTO nodeDTO : rt.getNodes()) {
            nodeService.deleteNodeDefinition(nodeDTO.getId(), true);
        }
		
		for (Message message : rt.getMessages()) {
            rtService.deleteMessage(message.getId());
        }

		rt.getNodes().clear();

		rtService.updateResourceTypeInDataBase(rt);
	}

	private void createNewResourceType(ResourceType resourceType) {
		IResourceTypeService rtService = getResourceTypeService();
		ResourceType tempCopy = new ResourceType();
		
		tempCopy.getMessages().addAll(resourceType.getMessages());
		tempCopy.getNodes().addAll(resourceType.getNodes());
		resourceType.getMessages().clear();
		resourceType.getNodes().clear();

		//Creates the resource type
		resourceType = rtService.createResourceTypeInDataBase(resourceType);

		resourceType.getMessages().addAll(tempCopy.getMessages());
		resourceType.getNodes().addAll(tempCopy.getNodes());

		if (!resourceType.getMessages().isEmpty()) {
            createOrUpdateMessages(resourceType);
        }

		if (!resourceType.getNodes().isEmpty()) {
            createOrUpdateNodeDefinitions(resourceType);
        }

		rtService.updateResourceTypeInDataBase(resourceType);
	}

	private void createOrUpdateMessages(ResourceType resourceType) {
		IResourceTypeService rtService = getResourceTypeService();
		
		for (Message message : resourceType.getMessages()) {
            rtService.createMessage(message);
        }
	}

	private void createOrUpdateNodeDefinitions(ResourceType resourceType) {
		INodeDefinitionService service = getNodeDefinitionService();
		
		for (NodeDefinitionDTO nodeDTO : resourceType.getNodes()) {
            service.createNodeDefinition(nodeDTO);
        }
	}

	/**
	 * 		Validates the resource type, and apply the markers when needed.
	 * @param file
	 * @param resourceType
	 */
	private void validateResourceType(IFile file, ResourceType resourceType) {
		IResourceTypeService service = getResourceTypeService();
		List<ValidationMessage> messages = service.validateResourceType(resourceType.getId());
		
		try {
			file.getProject().deleteMarkers(MARKER_TYPE, true, IResource.DEPTH_INFINITE);
		} catch (CoreException e) {
			logger.warn("Error deleting markers", e);
		}
		
		boolean hasErrors = false;
		boolean hasWarnings = false;
		
		for (ValidationMessage message : messages) {
			try {
				IMarker marker = file.createMarker(MARKER_TYPE);
                String msg = ValidationMessageUtil.getMessageFromValidationMessage(message);
				
				marker.setAttribute(IMarker.MESSAGE, msg);

				// default severity
				int severity = IMarker.SEVERITY_ERROR;
				switch (message.getType()) {
				case Error:
					hasErrors = true;
					break;

				case Warning:
					hasWarnings = true;
					severity = IMarker.SEVERITY_WARNING;
					break;

				case Notification:
					severity = IMarker.SEVERITY_INFO;
					break;
				}
				
				marker.setAttribute(IMarker.SEVERITY, severity);
				marker.setAttribute(IMarker.LINE_NUMBER, 1);
                message.copyAttributesTo(marker);
			} catch (CoreException e) {
				logger.warn("Error adding marker", e);
			}
		}
		
		resourceType.setHasErrors(hasErrors);
		resourceType.setHasWarnings(hasWarnings);
		// Update the ResourceType hasErrors and hasWarnings properties.
		service.updateResourceTypeInDataBase(resourceType);
	}
	
	/**
	 * 		Deletes the resource type.
	 * 
	 * @param resource
	 */
    public void delete(IResource resource) {
        IResourceTypeService rtService = getResourceTypeService();
        List<ResourceType> resourceTypes = rtService.getResourceTypeByProject(resource.getProject().getName());

        for (ResourceType resourceType : resourceTypes) {
            delete(resourceType);
        }
        // remove resource type from cache
        DataCollectionManager.getInstance().removeAll(resource.getProject().getName());
        WatchpointResourceTypeBuilder.refreshViews();
    }

	private void delete(ResourceType resourceType) {
		IResourceTypeService rtService = getResourceTypeService();
		
		cleanResourceTypeMessagesAndNodes(resourceType);
		rtService.deleteResourceType(resourceType.getId());
		validateDependencies(resourceType);
	}

	/**
	 * 		Check if its a valid resource type file.
	 * @param resource
	 * @return
	 */
	private boolean isResourceTypePluginXML(IResource resource) {
		boolean isPluginXML = false;
		
		resource.getProjectRelativePath();
		
		if (resource instanceof IFile && resource.getName().equals(RESOURCE_TYPE_PLUGIN_FILE_NAME)) {
            isPluginXML = true;
        }

		String pluginFilePath = "/" + resource.getProject().getName() + "/" + RESOURCE_TYPE_PLUGIN_FILE_NAME;
		
		if (!resource.getFullPath().toString().equals(pluginFilePath)) {
            isPluginXML = false;
        }

		return isPluginXML;
	}

	private boolean isBuildProperties(IResource resource) {
		boolean isBuildProperties = false;
		
		resource.getProjectRelativePath();
		
		if (resource instanceof IFile && resource.getName().equals(RESOURCE_TYPE_BUILD_PROPERTIES_FILENAME)) {
            isBuildProperties = true;
        }
		
		String buildPropertiesPath = "/" + resource.getProject().getName() + "/" + RESOURCE_TYPE_BUILD_PROPERTIES_FILENAME;
		
		if (!resource.getFullPath().toString().equals(buildPropertiesPath)) {
            isBuildProperties = false;
        }

		return isBuildProperties;
	}

	/**
	 * Node = PROJECT/resources/node/NAME.xml
	 * 
	 * @param resource
	 * @return
	 */
	private boolean isNode(IResource resource) {
		if (!(resource instanceof IFile)) {
            return false;
        }

		IFile file = (IFile) resource;
		IContainer parent = file.getParent();

		return ("xml".equals(file.getFileExtension()) && parent instanceof IFolder && parent.getName().equals(PLUGIN_NODE_FOLDER));
	}

	/**
	 * Method called to update the plugin.xml file with the jars selected in the build.properties
	 * file located in the resource type project root.
	 * @param resource The build.properties file
	 */
	private void updatePluginJars(IResource resource) {
		IFile file = (IFile) resource;
		IFile pluginFile = file.getProject().getFile(RESOURCE_TYPE_PLUGIN_FILE_NAME);
		
		if (pluginFile != null) {
			Properties prop = new Properties();
			
			try {
				prop.load(file.getContents());
				String includes = prop.getProperty(BIN_INCLUDES_KEY);
				String[] exportFiles = includes.split(",");

				List<String> jarFiles = new ArrayList<String>();
				for (String exportFile : exportFiles) {
					if (exportFile.endsWith(JAR_EXTENSION)) {
						jarFiles.add(exportFile);
					}
				}
				
				insertJarRecords(jarFiles, pluginFile);
			} catch (Exception e) {
				handleException("Error while updating plugin.xml with jar libraries.", e);
			}
		}
	}

	private void insertJarRecords(List<String> jarFiles, IFile pluginFile) {
		try {
			Document pluginDoc = XMLDocumentUtility.loadFile(pluginFile.getContents(), false);

			/* Create or find the Runtime Element node */
			Element pluginElement = pluginDoc.getDocumentElement();

			/* Collect all the current libraries */
			List<String> currentLibrariesPath = new ArrayList<String>();
			NodeList nodes = pluginElement.getElementsByTagName(JPFConstants.ELEM_RUNTIME);
			
			if (nodes != null && !(nodes.getLength() == 0)) {
				Element element = (Element) nodes.item(0);
				NodeList runtimeNodes = element.getChildNodes();
				
				if (runtimeNodes != null) {
					for (int i = 0; i < runtimeNodes.getLength(); i++) {
						Node childNode = runtimeNodes.item(i);
						
						if (childNode instanceof Element && ELEM_LIBRARY.equals(childNode.getNodeName())) {
							Element childElement = (Element) childNode;
							String path = childElement.getAttribute(ATTR_LIBRARY_PATH);
							
							currentLibrariesPath.add(path); // FIXME This is useless

							if (path.endsWith(JAR_EXTENSION)) {
                                element.removeChild(childNode);
                            }
						}
					}

					for (int i = 0; i < runtimeNodes.getLength(); i++) {
						Node childNode = runtimeNodes.item(i);
						
						if (childNode.getNodeType() == Node.TEXT_NODE) {
                            childNode.getParentNode().removeChild(childNode);
                        }
					}
				}
			}

			JPFExportUtil.addRuntimeLibraries(pluginDoc, jarFiles, null);

			ByteArrayOutputStream plugin = 
				ResourceTypePluginFileGenerator.createWritablePluginDocument(pluginDoc, new File(""));
			byte[] pluginContentArray = plugin.toByteArray();
			
			// Check if the file is sync with the file system.
			// If not, the setContents operation below will throw an exception
			if (!pluginFile.isSynchronized(IResource.DEPTH_ONE)) {
                pluginFile.refreshLocal(IResource.DEPTH_ONE, null);
            }

			pluginFile.setContents(new ByteArrayInputStream(pluginContentArray), IFile.KEEP_HISTORY, null);
		} catch (Exception e) {
			handleException("Error while updating plugin.xml with jar libraries.", e);
		}
	}

	/**
	 * 		Check if its the java compiled classes folder - usually project/bin or project/classes . 
	 * 
	 * @param resource
	 * @return
	 */
	private boolean isClassesFolder(IResource resource) {
		boolean isClassesFolder = false;

		IJavaProject project = JavaCore.create(resource.getProject());
		try {
			IPath outputFolder = project.getOutputLocation();
			if (outputFolder.isPrefixOf(resource.getFullPath())) {
				isClassesFolder = true;
			}
		} catch (JavaModelException e) {
			// ignore as its just for optimization
		}
		return isClassesFolder;
	}

	private void handleException(String errorMessage, Exception e) {
		Status status = new Status(IStatus.ERROR, "WatchPoint Studio", IStatus.ERROR, errorMessage, e);
		StatusManager.getManager().handle(status, StatusManager.SHOW);
	}
	
	private INodeDefinitionService getNodeDefinitionService() {
		return ServiceFactory.createFactory().createNodeDefinitionService();
	}
	
	private IResourceTypeService getResourceTypeService() {
		return ServiceFactory.createFactory().createResourceTypeService();
	}
	
	private IResourceGroupService getResourceGroupService() {
		return ServiceFactory.createFactory().createResourceGroupService();
	}
	
	private IWorkflowTemplateService getTemplateService() {
		return ServiceFactory.createFactory().createWorkflowTemplateService();
	}
}
