/*
 * Created on Jun 25, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.dataaccess.jpa;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.persistence.Query;

import com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;

/**
 * The JPA Data Access Implementation for the NodeDefinitionDTO entity.
 * 
 * @author Vijay Silva
 */
public class NodeDefinitionDTOPDAO extends PersistenceDAO<NodeDefinitionDTO, Long> implements
		NodeDefinitionDTODAI {
	/**
	 * Constructor
	 * 
	 * @param persistenceContext
	 *            The Persistence Context to use for data operations.
	 */
	public NodeDefinitionDTOPDAO(JPAPersistenceContext persistenceContext) {
		super(NodeDefinitionDTO.class, persistenceContext);
	}

	/**
	 * Selects a subset of the Node Definition data.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.jpa.PersistenceDAO#findAll()
	 */
	@Override
	public List<NodeDefinitionDTO> findAll() {
		String queryName = "NodeDefinitionDTO.All";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		return this.buildNodeDefinitions(query.getResultList());
	}

	/**
	 * Selects a subset of the Node Definition data.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI#findByResourceType(long)
	 */
	public List<NodeDefinitionDTO> findByResourceType(long resourceTypeId) {
		String queryName = "NodeDefinitionDTO.ByResourceType";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("resourceTypeId", resourceTypeId);
		return this.buildNodeDefinitions(query.getResultList());
	}

	/**
	 * Finds all node definitions which are of type Single Node and have the resource type passed.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI#findSingleNodeDefinitionsByResourceType(long)
	 */
	public List<NodeDefinitionDTO> findSingleNodeDefinitionsByResourceType(long resourceTypeId) {
		String queryName = "NodeDefinitionDTO.SingleNodeDefinitionsByResourceType";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("resourceTypeId", resourceTypeId);
		return this.buildNodeDefinitions(query.getResultList());
	}
	
	/**
	 * Finds all node definitions which are of type Single Node and have the resource type passed.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI#findSingleNodeDefinitionsByResourceType(long)
	 */
	@SuppressWarnings("unchecked")
	public List<NodeDefinitionDTO> findSingleNodeDefinitionsByResourceTypeAndMessageType(long resourceTypeId, boolean incoming) {
		String queryName = "NodeDefinitionDTO.SingleNodeDefinitionsByResourceTypeAndMessageType";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("resourceTypeId", resourceTypeId);
		query.setParameter("incoming", incoming);
		query.setParameter("type", NodeDefinitionType.MessageNode);
		
		return query.getResultList();
	}

	/**
	 * Finds all node definitions which are parents of the node definition whose id is passed.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI#findByParentUsage(long)
	 */
	public List<NodeDefinitionDTO> findByParentUsage(long nodeDefinitionId) {
		String queryName = "NodeDefinitionDTO.ByParentUsage";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("nodeDefinitionId", nodeDefinitionId);
//		return this.buildNodeDefinitions(query.getResultList());
		return query.getResultList();
	}

	@Override
	public NodeDefinitionDTO findByPath(String path) {
		String queryName = "NodeDefinitionDTO.ByPath";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("path", path);

		NodeDefinitionDTO result = null;
		try {
			result = (NodeDefinitionDTO) query.getSingleResult();
		} catch (javax.persistence.NoResultException e) {
			// No NodeDefinitionDTO found - return null
		}

		return result;
	}

	/**
	 * Finds all super state node definitions that are mapped to exactly the same given set of messages.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI#findSuperStateNodeDefinitionsByMessages(java.util.Set)
	 */
	public List<NodeDefinitionDTO> findSuperStateNodeDefinitionsByMessages(Set<Message> messages) {
		if (messages == null || messages.isEmpty())
			return new ArrayList<NodeDefinitionDTO>();

		// Build the query string to get all node definitions that map at least to the given set of messages
		StringBuilder queryBuf = new StringBuilder();
		
		queryBuf.append("SELECT nd.id, message.id ");
		queryBuf.append(this.getFromClause()).append(" nd, ");
		queryBuf.append("IN (nd.messages) message ");
		queryBuf.append("WHERE nd.id in (");
		
		boolean first = true;
		for(Message message : messages) {
			if(first) {
				first = false;
			}
			else {
				queryBuf.append(" and nd.id in (");
			}
			queryBuf.append("SELECT nd.id ");
			queryBuf.append(this.getFromClause()).append(" nd, ");
			queryBuf.append("IN (nd.messages) message ");
			queryBuf.append("WHERE nd.type = 1 AND message.id = ").append(message.getId());
		}
		
		for(int i=1; i<messages.size(); ++i)
			queryBuf.append(")");

		queryBuf.append(")");
		
		// Create and execute the query
		Query query = this.getEntityManager().createQuery(queryBuf.toString());
		Map<Long, Set<Long>> ndMessagesMap = buildNodeDefMessageMapping(query.getResultList());
		
		// Get the node definitions that map only to the given set of messages
		List<NodeDefinitionDTO> nodeDefinitions = new ArrayList<NodeDefinitionDTO>();
		for(Map.Entry<Long, Set<Long>> ndMessagesMapEntry : ndMessagesMap.entrySet()) {
			Long nodeDefID = ndMessagesMapEntry.getKey();
			Set<Long> mappedMessages = ndMessagesMapEntry.getValue();
			if(isSameSetOfMessages(mappedMessages, messages)) {
				NodeDefinitionDTO nodeDefinition = new NodeDefinitionDTO();
				nodeDefinition.setId(nodeDefID);
				nodeDefinitions.add(nodeDefinition);
			}
		}
		return nodeDefinitions;
	}


	/**
	 * Finds all single-node node definitions that are mapped to the same message.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI#findSingleNodeDefinitionsByMessage(long)
	 */
	public List<NodeDefinitionDTO> findSingleNodeDefinitionsByMessage(long id) {
		String queryName = "NodeDefinitionDTO.SingleNodeDefinitionsByMessage";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("id", id);
		return this.buildNodeDefinitions(query.getResultList());
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI#getCountByParentUsage(long)
	 */
	public int getCountByParentUsage(long nodeDefinitionId) {
		String queryName = "NodeDefinitionDTO.CountByParentUsage";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("nodeDefinitionId", nodeDefinitionId);
		Long count = (Long) query.getSingleResult();
		return count.intValue();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI#getCountByResourceType(long)
	 */
	public int getCountByResourceType(long resourceTypeId) {
		String queryName = "NodeDefinitionDTO.CountByResourceType";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("resourceTypeId", resourceTypeId);
		Long count = (Long) query.getSingleResult();
		return count.intValue();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI#getCountByMessage(long)
	 */
	public int getCountByMessage(long messageId) {
		String queryName = "NodeDefinitionDTO.CountByMessage";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("messageId", messageId);
		Long count = (Long) query.getSingleResult();
		return count.intValue();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI#findByKeys(java.util.Collection)
	 */
	public List<NodeDefinitionDTO> findByKeys(Collection<Long> keys) {
		if (keys == null || keys.size() == 0)
			return new ArrayList<NodeDefinitionDTO>();

		// Build the Query String
		StringBuilder queryBuf = new StringBuilder();
		queryBuf.append("SELECT nd.id, nd.name ");
		queryBuf.append(this.getFromClause()).append(" nd ");
		queryBuf.append("WHERE nd.id IN ( ");

		boolean first = true;
		for (Long key : keys) {
			if (!first)
				queryBuf.append(", ");
			else
				first = false;

			queryBuf.append(key.toString());
		}

		queryBuf.append(" ) ");

		// Create and execute the query
		Query query = this.getEntityManager().createQuery(queryBuf.toString());
		return this.buildNodeDefinitions(query.getResultList());
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI#getCountByName(java.lang.String)
	 */
	public int getCountByName(String name) {
		String queryName = "NodeDefinitionDTO.CountByName";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("name", name);
		Long count = (Long) query.getSingleResult();
		return count.intValue();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.NodeDefinitionDTODAI#deleteNodeDefinitionReferences(long)
	 */
	public void deleteNodeDefinitionReferences(long nodeDefinitionId) {
		String sqlQuery = "DELETE from TB_NODEDEFINITIONPARENTMAP WHERE NODEDEFINITIONID= :nodeDefinitionId";
		Query query = getEntityManager().createNativeQuery(sqlQuery);

		query.setParameter("nodeDefinitionId", nodeDefinitionId);
		query.executeUpdate();
	}

	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/*
	 * Create NodeDefinitionDTO entities from the array of column values.
	 */
	private List<NodeDefinitionDTO> buildNodeDefinitions(List<?> data) {
		List<NodeDefinitionDTO> nodeDefinitions = new ArrayList<NodeDefinitionDTO>();

		if (data != null) {
			for (Object dataObject : data) {
				Object[] columns = (Object[]) dataObject;
				NodeDefinitionDTO nodeDefinition = new NodeDefinitionDTO();
				nodeDefinition.setResourceType(new ResourceType());
				for (int i = 0; i < columns.length; i++) {
					switch (i) {
						case 0:
							nodeDefinition.setId((Long) columns[0]);
							break;
						case 1:
							nodeDefinition.setName((String) columns[1]);
							break;
						case 2:
							nodeDefinition.setType((NodeDefinitionType) columns[2]);
							break;
						case 3:
							nodeDefinition.getResourceType().setId((Long) columns[3]);
							break;
						case 4:
							nodeDefinition.getResourceType().setName((String) columns[4]);
							break;
						case 5:
							nodeDefinition.getResourceType().setOutOfTheBox((Boolean) columns[5]);
							break;
					}
				}

				nodeDefinitions.add(nodeDefinition);
			}
		}

		return nodeDefinitions;
	}
	
	/* Builds a map between node definition id and list of message ids
	 * from the given list of rows with columns node definition id & message id.
	 */
	private Map<Long, Set<Long>> buildNodeDefMessageMapping(List<?> data) {
		Map<Long, Set<Long>> ndMessagesMap = new HashMap<Long, Set<Long>>();

		if (data != null) {
			for (Object dataObject : data) {
				Object[] columns = (Object[]) dataObject;
				Long nodeDefID = (Long) columns[0];
				Long messageID = (Long) columns[1];
				Set<Long> messages = ndMessagesMap.get(nodeDefID);
				if(messages == null) {
					messages = new HashSet<Long>();
					ndMessagesMap.put(nodeDefID, messages);
				}
				messages.add(messageID);
			}
		}

		return ndMessagesMap;
	}

	/* Checks if the set of messages are same (with respect to IDs) as the set of message ids;
	 * No more, no less.
	 */
	private boolean isSameSetOfMessages(Set<Long> messageIDs, Set<Message> messages) {
		if(messageIDs.size() == messages.size()) {
			for(Message message : messages) {
				if(!messageIDs.contains(message.getId()))
					return false;
			}
			return true;
		}

		return false;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<NodeDefinitionDTO> findByMessageUID(String messageUID,
			NodeDefinitionType type) {
		String queryName = "NodeDefinitionDTO.ByMessageUID";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("messageUID", messageUID);
		query.setParameter("type", type);
		return query.getResultList();	
	}
}
