/*
 * Created on Jul 5, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.service;

import java.util.List;
import java.util.Map;

import com.tandbergtv.watchpoint.studio.dto.ProtectionKey;
import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExportFailureException;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;

/**
 * Service interface defining operations on ResourceGroup entities.
 * 
 * @author Vijay Silva
 */
public interface IResourceGroupService extends IService
{
	int FIRST_VERSION = 1;
	
	/**
	 * Gets all the available Resource Groups.
	 * 
	 * @return The list of all resource groups
	 */
	List<ResourceGroup> getAllResourceGroups();
	
	/**
	 * Gets the ResourceGroup by path
	 * 
	 * @param path
	 * @return
	 */
	ResourceGroup getResourceGroupByPath(String path);
	
	/**
	 * Validates the ResourceGroup
	 * 
	 * @param group The ResourceGroup to validate
	 * @return
	 */
	List<ValidationMessage> validateResourceGroup(ResourceGroup group);
	
	/**
	 * Get all the Resource Groups for a Resource Type
	 * 
	 * @param resourceTypeSystemId
	 *            The Resource Type system Id
	 * 
	 * @return The list of matching Resource Groups
	 */
	List<ResourceGroup> getResourceGroupsByResourceType(String resourceTypeSystemId);

	/**
	 * Get all the Resource Groups for the Human Resource Type
	 * 
	 * @return The list of matching Resource Groups
	 */
	List<ResourceGroup> getResourceGroupsByHumanResourceType();
	
	/**
	 * Get all the Resource Groups for the referred template
	 * 
	 * @return The list of matching Resource Groups
	 */
	List<ResourceGroup> getResourceGroupsByTemplateUsage(String templatePath);

	/**
	 * Get a Resource Group given the resource group id.
	 * 
	 * @param resourceGroupId
	 *            The Resource Group Id
	 * 
	 * @return The matching Resource Group
	 */
	ResourceGroup getResourceGroup(long resourceGroupId);
	
	/**
	 * Get default ResourceGroup for a Resource Type given.
	 * 
	 * @param resourceType
	 *            The Resource Type
	 * 
	 * @return The matching Resource Group or null case it can't be defined.
	 */
	ResourceGroup getDefaultResourceGroupForResourceType(ResourceType resourceType);

	/**
	 * Get all the ResourceGroups with the given name.
	 * 
	 * @param name
	 *            The Resource Group name
	 * 
	 * @return The list of resource groups with a matching name
	 */
	List<ResourceGroup> getResourceGroupByName(String name);
	
	/**
	 * Get all the ResourceGroups for the given project.
	 * 
	 * @param name
	 *            The project name.
	 * 
	 * @return The list of resource groups for the project
	 */
	List<ResourceGroup> getResourceGroupsByProject(String project);

	/**
	 * Get a Protection Key of a Resource Group given the name of the resource group.
	 * 
	 * @param name
	 *            The Resource Group name
	 * 
	 * @return The Protection Key of the resource group with a matching name
	 */
	ProtectionKey getProtectionKeyByResourceGroupName(String name);
	
	/**
	 * Gets functional types used by all Resource Groups that are
	 * of the given Resource Type
	 *  
	 * @param resourceTypeSystemId
	 * 			The Resource Type system Id
	 * 
	 * @return The List of Functional Types for the given Resource Type
	 */
	List<String> getAllFunctionalTypesForResourceType(String resourceTypeSystemId);
	
	/**
	 * Checks if a Resource Group with the given name exists.
	 * 
	 * @param name
	 * 			The Resource Group Name
	 * @return true if a Resource Group with the given name exist; false otherwise
	 */
	boolean isNameExists(String name);
	
	/**
	 * Creates the given ResourceGroup to the database, or updates if it already exists.
	 * 
	 * @param group The Resource Group to save
	 * @return the saved Resource Group
	 */
	ResourceGroup saveResourceGroup(ResourceGroup group);
	
	/**
	 * Deletes the resource group 
	 * 
	 * @param group The Resource Group path
	 * 
	 */
	void deleteResourceGroup(String resourceGroupPath);
	
	/**
	 * Exports the resource group from database to file system.
	 * 
	 * @param group The Resource Group that has to be exported
	 * @param exportInputs Attributes necessary for exporting
	 * @return The Resource Group obtained from the database
	 * @throws ExportFailureException
	 */
	ResourceGroup exportResourceGroup(ResourceGroup group, Map<String, Object> exportInputs)
																throws ExportFailureException;
	
	/**
	 * Gets all the existing protection keys.
	 * 
	 * @return List of Protection Keys
	 */
	List<ProtectionKey> getAllProtectionKeys();
	
}
