/*
 * Created on Jul 6, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.service;

import java.util.List;

import com.tandbergtv.watchpoint.studio.dto.ConnectionType;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;

/**
 * Service interface defining operations on ResourceType entity and its child entities.
 * 
 * @author Vijay Silva
 */
public interface IResourceTypeService extends IService
{
	/**
	 * Gets the list of all currently available Resource Types in the system.
	 * 
	 * @return The list of available Resource Types.
	 */
	List<ResourceType> getResourceTypeList();

	/**
	 * Creates a new Resource Type in the system. Ensures that the Resource Type name specified is
	 * unique at the point of creating the Resource Type. All messages contained in the Resource
	 * Type are ignored, and not created / updated in the persistence layer.
	 * 
	 * @param resourceType
	 *            The new Resource Type to create
	 * 
	 * @return The newly created Resource Type.
	 */
	ResourceType createResourceTypeInDataBase(ResourceType resourceType);
	
	/**
	 * Updates an existing Resource Type. Messages contained in the input Resource Type that
	 * currently exist in the persistence layer are updated. All other messages are ignored.
	 * 
	 * @param resourceType
	 *            The Resource Type to update.
	 * 
	 * @return The updated Resource Type.
	 */
	public ResourceType updateResourceTypeInDataBase(ResourceType resourceType);
	
	/**
	 * Updates and existing Resource Type in the file system.
	 * @param resourceType The Resource Type to update
	 * @return The updated Resource Type
	 */
	public ResourceType updateResourceTypeInFileSystem(ResourceType resourceType);
	
	/**
	 * Generates a message UID.<br>
	 * This method is called if the user choose to auto generate the message UID
	 * @param resourceType The Resource Type for which the message is being created
	 * @param messageForGeneration The message to be set with the generated UID
	 */
	public void generateMessageUID(ResourceType resourceType, Message messageForGeneration);
	
	/**
	 * Returns a Resource Type on the file system given the Resource Type project name.
	 * @param projectName The Resource Type project name
	 * @return The Resouce Type object parsed from the files in file system
	 */
	public ResourceType getResourceTypeFromFileSystem(String projectName);

	/**
	 * Delete an existing Resource Type and all its contained messages.
	 * 
	 * @param resourceTypeId
	 *            The Id of the resource type to delete.
	 */
	void deleteResourceType(long resourceTypeId);

	/**
	 * Get a Resource Type with a matching Id.
	 * 
	 * @param resourceTypeId
	 *            The Resource Type Id.
	 * 
	 * @return The matching Resource Type.
	 */
	ResourceType getResourceType(long resourceTypeId);
	
	/**
	 * Get a Resource Type by its path. 
	 * 
	 * @param path
	 *            The Resource Type path.
	 * 
	 * @return The Resource Type.
	 */
	ResourceType getResourceTypeByPath(String path);
	
	/**
	 * Get the Resource Types for a specified project.
	 * 
	 * @return The List of Resource Types for that project.
	 */
	List<ResourceType> getResourceTypeByProject(String projectName);

	/**
	 * Get the Resource Type with a matching System Id.
	 * 
	 * @param systemId
	 *            The System Id
	 * 
	 * @return The resource type with a matching system Id.
	 */
	ResourceType getResourceTypeBySystemId(String systemId);

	/**
	 * Get the Resource Type with a matching name.
	 * 
	 * @param name
	 *            The name of the Resource Type
	 * 
	 * @return The resource type with a matching name.
	 */
	ResourceType getResourceTypeByName(String name);

	/**
	 * Validates a Resource Type, checking if it is ready for export.
	 * 
	 * @param resourceTypeId
	 *            The Id of the resource type being validated.
	 * 
	 * @return The list of validation messages from validation.
	 */
	List<ValidationMessage> validateResourceType(long resourceTypeId);

	/**
	 * Check to see if the specified name is not already used by any existing Resource Type.
	 * 
	 * @param name
	 *            The name to check for uniqueness
	 * 
	 * @return true if the name is unique, false if the name is not unique.
	 */
	boolean isResourceTypeNameUnique(String name);

	/**
	 * Check to see if the system Id is not already used by an existing Resource Type.
	 * 
	 * @param systemId
	 *            The System Id to check for uniqueness
	 * 
	 * @return true if the System Id is unique, false if not unique.
	 */
	boolean isResourceTypeSystemIdUnique(String systemId);

	/**
	 * Gets messages for the Resource Type specified by the resourceTypeId
	 *  that are not used by single-node Node Definitions.
	 *   
	 * @param resourceTypeId
	 * 			the Id of the Resource Type 
	 * @return List of Messages
	 */
	List<Message> getUnusedResourceTypeMessages(long resourceTypeId);

	/**
	 * Gets messages for the Resource Type specified by the resourceTypeId
	 * that are not used by single-node Node Definitions except the node definition whose id is passed.
	 *   
	 * @param resourceTypeId
	 * 			the Id of the Resource Type
	 * @param nodeDefinitionId
	 * 			the id of the node definition whose message is not to be excluded
	 * @return List of Messages
	 */
	List<Message> getUnusedResourceTypeMessages(long resourceTypeId, long nodeDefinitionId);
	
	/**
	 * Finds the Message based on the given UID.
	 * 
	 * @param messageUID The Message UID
	 * @return	The Message
	 */
	Message findMessageByUID(String messageUID);
	
	/**
	 * Gets the list of all available Connection Types
	 * 
	 * @return The list of ConnectionTypes
	 */
	List<ConnectionType> getAllConnectionTypes();

	/**
	 * Creates a new Message for a Resource Type. The parent Resource Type in the Message must
	 * contain the resource type id at the very least.
	 * 
	 * @param message
	 *            The Message to create
	 * 
	 * @return The created Message
	 */
	Message createMessage(Message message);

	/**
	 * Deletes an existing Message for a Resource Type.
	 * 
	 * @param messageId
	 *            The Message Id of the message to delete
	 */
	void deleteMessage(long messageId);
}
