package com.tandbergtv.watchpoint.studio.ui;

import static com.tandbergtv.watchpoint.studio.builder.WatchpointTemplateNature.NATURE_ID;
import static org.eclipse.core.resources.IResourceChangeEvent.PRE_DELETE;
import static org.eclipse.core.resources.IResourceDelta.CHANGED;
import static org.eclipse.core.resources.IResourceDelta.OPEN;
import static org.eclipse.core.resources.IncrementalProjectBuilder.FULL_BUILD;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.navigator.CommonViewer;
import org.eclipse.ui.navigator.resources.ProjectExplorer;
import org.jbpm.gd.jpdl.Plugin;
import org.osgi.framework.BundleContext;

import com.tandbergtv.watchpoint.studio.builder.ResourceGroupVisitor;
import com.tandbergtv.watchpoint.studio.builder.WatchpointResourceTypeBuilder;
import com.tandbergtv.watchpoint.studio.builder.WatchpointResourceTypeNature;
import com.tandbergtv.watchpoint.studio.builder.WatchpointResourceTypeVisitor;
import com.tandbergtv.watchpoint.studio.builder.WatchpointTemplateBuilder;
import com.tandbergtv.watchpoint.studio.builder.WatchpointTemplateVisitor;
import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.service.IResourceGroupService;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.view.resourcetype.actions.ProjectExplorerDoubleClickListener;
import com.tandbergtv.watchpoint.studio.util.RequireInitializeValuesEntities;
import com.tandbergtv.watchpoint.studio.util.ResourceTypeRequiredDataInitializer;

public class WatchPointStudioPlugin extends Plugin {
	
	// The Logger
	private static final Logger logger = Logger.getLogger(WatchPointStudioPlugin.class);

	public WatchPointStudioPlugin() {
		super();
	}

	/* (non-Javadoc)
	 * @see org.jbpm.gd.jpdl.JpdlPlugin#start(org.osgi.framework.BundleContext)
	 */
	@Override
	public void start(BundleContext context) throws Exception {
		/*Register listener for workbench lifecycle events*/
		super.start(context);
		
		// Run the builders asynchronously
		final String jobNamePrefix = "Building Watchpoint Studio Resources";
		Job job = new Job(jobNamePrefix) {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				setName(jobNamePrefix + ": Initializing data");
				initializeRequiredData();
				setName(jobNamePrefix + ": Resource Types");
				initializeResourceTypeBuilder();
				setName(jobNamePrefix + ": Templates");
				initializeTemplateBuilder();
				setName(jobNamePrefix + ": Checking cache consistency");
				checkDatabaseSynch();
				
				addDeleteListener();
				
				addDoubleClickProjectExplorerListener();

				return Status.OK_STATUS;
			}
		};
		
		job.schedule();
	}
	
	private void addDoubleClickProjectExplorerListener() {
		PlatformUI.getWorkbench().getDisplay().asyncExec(new Runnable() {
			public void run() {
				IWorkbenchWindow dwindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
				final IWorkbenchPage page = dwindow.getActivePage();

				if (page == null)
					return;

				IViewPart part = page.findView(ProjectExplorer.VIEW_ID);

				if (part != null) {
					StructuredViewer viewer = (StructuredViewer) part.getAdapter(CommonViewer.class);

					viewer.addDoubleClickListener(new ProjectExplorerDoubleClickListener());
				}
			}
		});
	}
	
	private void initializeRequiredData() {
		ResourceTypeRequiredDataInitializer initializer = new ResourceTypeRequiredDataInitializer();
		initializer.initializeData(RequireInitializeValuesEntities.CONNECTION_TYPE);
	}
	
	/**
	 *	Initialize the resource type builder. Constructs the Resource Types database
	 */
	private void initializeResourceTypeBuilder() {
		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		IProject projects[] = workspace.getRoot().getProjects();
		
		// Build all projects
		for (IProject project : projects) {
			try {
				if (!project.isAccessible() || !project.hasNature(WatchpointResourceTypeNature.NATURE_ID))
					continue;

				IResourceTypeService service = 
					ServiceFactory.createFactory().createResourceTypeService();
				
				// Project haven't been loaded. Build-it.
				if (service.getResourceTypeByProject(project.getName()).isEmpty() || hasUnloadedResourceGroups(project))
					project.build(FULL_BUILD, WatchpointResourceTypeBuilder.BUILDER_ID, null, null);
			} catch (CoreException e) {
				// Error 
				logger.error("Error building project " + project, e);
			}
		}
	}
	
	/**
	 * 		Verifies if ResourceGroups that hasn't been loaded in the database for a given project.
	 * 
	 * @param project
	 * @return
	 */
	private boolean hasUnloadedResourceGroups(IProject project) {
		boolean hasUnloaded = false;
		
		IFolder groupsFolder = project.getFolder("groups");
		
		if (!groupsFolder.exists())
			return hasUnloaded;
		
		try {
			IResourceGroupService service = ServiceFactory.createFactory().createResourceGroupService();
			for (IResource resource : groupsFolder.members()) {
				if (resource instanceof IFolder) {
					IFolder groupFolder = (IFolder) resource;
					IFile groupFile = groupFolder.getFile(Utility.PLUGIN_XML);
					if (groupFile.exists()) {
						hasUnloaded = service.getResourceGroupByPath(groupFile.getFullPath().toPortableString()) == null;
						if (hasUnloaded) {
							break;
						}
					}
				}
			}
		} catch (CoreException e) {
			logger.error("Error reading folder: " + groupsFolder, e);
		}
		
		return hasUnloaded;
	}

	
	/**
	 * 	Housekeeping function to check if there are any existing entities in the database that no longer
	 * exists in the filesystem. If entity exists in the database and doesn't in the filesystem, then it will 
	 * be deleted from the database.  
	 */
	private void checkDatabaseSynch() {
		IWorkflowTemplateService templateService = ServiceFactory.createFactory().createWorkflowTemplateService();
		for (WorkflowTemplateDTO template : templateService.getTemplateList()) {
			IWorkspace workspace = ResourcesPlugin.getWorkspace();
			IFile file = workspace.getRoot().getFile(new Path(template.getPath()));
			if (!file.exists()) {
				new WatchpointTemplateVisitor().deleteTemplate(file);
			}
		}
		IResourceGroupService resourceGroupService = ServiceFactory.createFactory().createResourceGroupService();
		for (ResourceGroup resourceGroup : resourceGroupService.getAllResourceGroups()) {
			IWorkspace workspace = ResourcesPlugin.getWorkspace();
			IFile file = workspace.getRoot().getFile(new Path(resourceGroup.getPath()));
			if (!file.exists()) {
				new ResourceGroupVisitor().deleteResourceGroup(file);
			}
		}
		IResourceTypeService resourceTypeService = ServiceFactory.createFactory().createResourceTypeService();
		for (ResourceType resourceType : resourceTypeService.getResourceTypeList()) {
			IWorkspace workspace = ResourcesPlugin.getWorkspace();
			IFile file = workspace.getRoot().getFile(new Path(resourceType.getPath()));
			if (!file.exists()) {
				new WatchpointResourceTypeVisitor().delete(file);
			}
		}
	}

	/**
	 * 		Initialize the Template Builder. Constructs the Templates database.
	 */
	private void initializeTemplateBuilder() {
		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		IProject projects[] = workspace.getRoot().getProjects();
		
		// Build all projects
		for (IProject project : projects) {
			try {
				if (!project.isAccessible() || !project.hasNature(NATURE_ID))
					continue;
				
				IWorkflowTemplateService service = 
					ServiceFactory.createFactory().createWorkflowTemplateService();
				
				// Project haven't been loaded. Build-it.
				if (service.getTemplatesByProject(project.getName()).isEmpty() || hasUnloadedResourceGroups(project))
					project.build(FULL_BUILD, WatchpointTemplateBuilder.BUILDER_ID, null, null);
			} catch (CoreException e) {
				// Error 
				logger.error("Error building project " + project, e);
			}
		}
	}
	
	/**
	 * 	Adds a listener to detect project deletion, as builders doesnt detect this event.
	 * 
	 */
	private void addDeleteListener() {
		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		// As Project deletion is not detected by the Builder, add a Listener just to detect this case
		IResourceChangeListener listener = new IResourceChangeListener() {
			public void resourceChanged(IResourceChangeEvent event) {
				if (event.getDelta() != null && event.getDelta().getKind() == CHANGED) {
					for (IResourceDelta resourceDelta : event.getDelta().getAffectedChildren()) {
						boolean isOpenAction = (resourceDelta.getFlags() & OPEN) == OPEN;
						
						if (resourceDelta.getResource() instanceof IProject && isOpenAction) {
							final IProject project = (IProject) resourceDelta.getResource();
							
							if (project.isOpen())
								// Project was closed and is being opened, call the builders
								runBuilders(project);
							else
								// Project was open and is being closed, clean it
								cleanProjectResources(project);
						}
					}
				}
				
				if (event.getResource() instanceof IProject) {
					final IProject project = (IProject) event.getResource();
					
					if (event.getType() == PRE_DELETE) {
						cleanProjectResources(project);
					}
				}
			}
		};
		
		workspace.addResourceChangeListener(listener);
	}
	
	void runBuilders(final IProject project) {
		Job job = new Job("Building Watchpoint Studio Resources") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				try {
					project.build(FULL_BUILD, WatchpointTemplateBuilder.BUILDER_ID, null, null);
					project.build(FULL_BUILD, WatchpointResourceTypeBuilder.BUILDER_ID, null, null);
				} catch (CoreException e) {
					logger.error("Error building project ", e);
				}
				return Status.OK_STATUS;
			}

		};
		
		job.schedule();
	}
	
	/**
	 * Cleans the project's resources generated by the builder 
	 * 
	 * @param project
	 */
	void cleanProjectResources(final IProject project) {
		Job job = new Job("Cleaning Watchpoint Studio Resources") {
			@Override
			protected IStatus run(IProgressMonitor monitor) {
				// Deletes Templates first, because no entity depends on Templates 
				new WatchpointTemplateVisitor().delete(project);
				// Deletes ResourceGroups first, because ResourceGroups depends on Resource Types
				new ResourceGroupVisitor().delete(project);
				new WatchpointResourceTypeVisitor().delete(project);
				return Status.OK_STATUS;
			}
		};
		
		job.schedule();
	}

	/* (non-Javadoc)
	 * @see org.jbpm.gd.jpdl.JpdlPlugin#stop(org.osgi.framework.BundleContext)
	 */
	@Override
	public void stop(BundleContext context) throws Exception {
		super.stop(context);
	}
}
