/*
 * Created on Sep 18, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.editor.resourcetype;
import static com.tandbergtv.watchpoint.studio.builder.WatchpointResourceTypeVisitor.MARKER_TYPE;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbenchPartConstants;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.forms.AbstractFormPart;
import org.eclipse.ui.forms.IFormPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.editor.IFormPage;
import org.eclipse.ui.part.FileEditorInput;

import com.tandbergtv.watchpoint.studio.dto.AdaptorType;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceException;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.service.ServiceValidationException;
import com.tandbergtv.watchpoint.studio.ui.UIException;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpage.MessagesPage;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpage.ResourceTypeFormPage;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * The Resource Type Editor, which has multiple pages allowing editing of the different properties
 * of a Resource Type.
 * 
 * @author Vijay Silva
 */
public class ResourceTypeEditor extends FormEditor
{
	// The Logger
	private static final Logger logger = Logger.getLogger(ResourceTypeEditor.class);

	/* Flag to indicate if this editor is dirty or not */
	private boolean editorModified = false;
	
	/* Array holding the resource type pages extension */
	IConfigurationElement[] configurationElements = Platform.getExtensionRegistry()
	.getConfigurationElementsFor("org.jbpm.gd.jpdl.resourceTypePages");

	/**
	 * Creates the multi-page ResourceType Editor.
	 */
	public ResourceTypeEditor()
	{
		super();
	}

	// ========================================================================
	// ===================== OVERRIDDEN METHODS
	// ========================================================================

	/**
	 * Creates the pages for the editor.
	 * 
	 * @see org.eclipse.ui.forms.editor.FormEditor#addPages()
	 */
	@Override
	protected void addPages()
	{
		this.updateTitle();

		for (int i = 0; i < configurationElements.length; i++) {
			try {
				// Get the class name attribute
				Class<?> clazz = Class.forName(configurationElements[i].getAttribute("class"));
				// Create a class instance
				Object classObject = clazz.getConstructor(this.getClass()).newInstance(this);
				IFormPage page = (IFormPage)classObject;
				// Add the class loaded to this editor
				this.addPage(page);
			} catch (Exception e) {
				String msg = "Failed to initialize Resource Type Editor.";
				logger.error(msg, e);
				throw new UIException(msg, e);
			} 
		}
			
		IEditorInput input = this.getEditorInput();
		if (input != null)
		{
			Message message = (Message) input.getAdapter(Message.class);
			if (message != null)
				this.setActivePage(MessagesPage.PAGE_ID);
		}
	}

	/**
	 * Saves the multi-page editor's document.
	 */
	@Override
	public void doSave(IProgressMonitor monitor)
	{
		try
		{
			/* Build the Resource Type to save */
			for (int i = 0; i < this.getPageCount(); i++)
			{
				Object page = this.pages.get(i);
				if (page instanceof IFormPage)
				{
					IFormPage formPage = (IFormPage) page;
					formPage.doSave(monitor);
				}
			}

			ResourceTypeEditorInput input = (ResourceTypeEditorInput) this.getEditorInput();
			ResourceType resourceType = (ResourceType) input.getAdapter(ResourceType.class);
			this.prepareResourceTypeForSave(resourceType);

			ServiceFactory serviceFactory = ServiceFactory.createFactory();
			IResourceTypeService service = serviceFactory.createResourceTypeService();
			resourceType = service.updateResourceTypeInFileSystem(resourceType);
			input.updatePersistedResourceType(resourceType);

			this.editorSaved();
		}
		catch (ServiceValidationException sve)
		{
			monitor.setCanceled(true);

			Shell shell = this.getSite().getShell();
			String message = "The Resource Type is invalid, please fix validation errors before saving.";
			MessageDialog.openError(shell, this.getTitle(), message);
		}
		catch (ServiceException se)
		{
			monitor.setCanceled(true);
			Utility.reportError("Failed to save the Resource Type.", se);
		}
	}
	
	public void refreshEditor(){
		ResourceTypeEditorInput currentInput = (ResourceTypeEditorInput) this.getEditorInput();
		ResourceType old = currentInput.getResourceType();
		ResourceType parsed = getResourceTypeFromFileSystem(old.getProjectName(), old.getPath()); 
		
		ResourceTypeEditorInput input = new ResourceTypeEditorInput(parsed, null, true);
		this.setInput(input);
		this.firePropertyChange( IWorkbenchPartConstants.PROP_INPUT );
		
		for (int i = 0; i < this.getPageCount(); i++)
		{
			Object page = this.pages.get(i);
			if (page instanceof IFormPage)
			{
				ResourceTypeFormPage formPage = (ResourceTypeFormPage)page;
				formPage.setEditorInput(input);
				if(formPage.getManagedForm() != null){
					formPage.getManagedForm().setInput(input);
					IFormPart[] parts = formPage.getManagedForm().getParts();
					for (int j = 0; j < parts.length; j++) {
						parts[j].refresh();
					}
				}
			}
		}
	}
	
	private ResourceType getResourceTypeFromFileSystem(String projectName, String path){
		IResourceTypeService resourceTypeService = ServiceFactory.createFactory().createResourceTypeService();
		ResourceType parsed = resourceTypeService.getResourceTypeFromFileSystem(projectName);
		parsed.setPath(path);
		parsed.setProjectName(projectName);
		return  parsed;
	}

	public ResourceType getResourceType() {
		/* Build the Resource Type to save */
		for (int i = 0; i < this.getPageCount(); i++)
		{
			Object page = this.pages.get(i);
			if (page instanceof IFormPage)
			{
				IFormPage formPage = (IFormPage) page;
				formPage.doSave(new NullProgressMonitor());
			}
		}

		ResourceTypeEditorInput input = (ResourceTypeEditorInput) this.getEditorInput();
		return (ResourceType) input.getAdapter(ResourceType.class);
	}
	
	/**
	 * @see org.eclipse.ui.part.EditorPart#doSaveAs()
	 */
	@Override
	public void doSaveAs()
	{
		return;
	}

	/**
	 * The Editor requires input which is an instance of <code>ResourceTypeEditorInput</code>.
	 * 
	 * @see org.eclipse.ui.forms.editor.FormEditor#init(org.eclipse.ui.IEditorSite,
	 *      org.eclipse.ui.IEditorInput)
	 */
	@Override
	public void init(IEditorSite site, IEditorInput editorInput) throws PartInitException
	{
		if (!(editorInput instanceof ResourceTypeEditorInput || editorInput instanceof FileEditorInput))
		{
			String editorClassName = editorInput.getClass().getName();
			String expectedClassName = ResourceTypeEditorInput.class.getName();
			String msg = "Invalid Input, expecting input of type: " + expectedClassName
					+ ", but received input of type: " + editorClassName;
			throw new PartInitException(msg);
		}else{
			if(editorInput instanceof FileEditorInput){
				IFile file = ((FileEditorInput) editorInput).getFile();
				IResourceTypeService resourceTypeService = ServiceFactory.createFactory().createResourceTypeService();
				ResourceType parsed = resourceTypeService.getResourceTypeFromFileSystem(file.getProject().getName());
				parsed.setPath(file.getFullPath().toPortableString());
				parsed.setProjectName(file.getProject().getName());
				ResourceTypeEditorInput input = new ResourceTypeEditorInput(parsed, null, true);
				setInput(input);
			}else{
				if(editorInput instanceof ResourceTypeEditorInput){
					setInput(editorInput);
				}
			}
		}
		super.init(site, getEditorInput());
		Utility.setWindow(site.getWorkbenchWindow());
	}

	/**
	 * Cannot 'Save As' for a Resource Type
	 * 
	 * @see org.eclipse.ui.part.EditorPart#isSaveAsAllowed()
	 */
	public boolean isSaveAsAllowed()
	{
		return false;
	}

	/**
	 * @see org.eclipse.ui.forms.editor.FormEditor#isDirty()
	 */
	@Override
	public boolean isDirty()
	{
		return (this.editorModified || super.isDirty());
	}

	/**
	 * @see org.eclipse.ui.forms.editor.FormEditor#editorDirtyStateChanged()
	 */
	@Override
	public void editorDirtyStateChanged()
	{
		/* The Editor is now dirty or clean. If Dirty, update the editorModified flag */
		if ((!this.editorModified) && this.isDirty())
			this.editorModified = true;

		super.editorDirtyStateChanged();
	}

	/*
	 * Update the ResourceType properties removing any unwanted properties
	 */
	private void prepareResourceTypeForSave(ResourceType resourceType)
	{
		/* Fix Adaptor Properties based on the Adaptor Type */
		AdaptorType adaptorType = resourceType.getAdaptorType();
		if (adaptorType == null)
		{
			adaptorType = AdaptorType.NONE;
			resourceType.setAdaptorType(adaptorType);
		}

		/* Remove Default Adaptor Properties if not required */
		if (adaptorType != AdaptorType.DEFAULT)
		{
			resourceType.setMessageIdentificationClassName(null);
		}

		/* Remove Custom Adaptor Properties if not required */
		if (adaptorType != AdaptorType.CUSTOM)
		{
			resourceType.setAdaptorClassName(null);
			resourceType.setAdaptorScope(null);
			resourceType.setAdaptorParameters(null);
		}

		/* Fix the Initialization Properties */
		if (!resourceType.isInitializationRequired())
		{
			resourceType.setInitializationStrategyClass(null);
			resourceType.setInitializationParameters(null);
		}

		/* Fix the Heart Beat Properties */
		if (!resourceType.isHeartbeatRequired())
		{
			resourceType.setHeartbeatStrategyClass(null);
			resourceType.setHeartbeatConnectionType(null);
			resourceType.setHeartbeatParameters(null);
		}
	}

	/*
	 * The Editor has been successfully saved.
	 */
	private void editorSaved()
	{
		this.editorModified = false;
		this.editorDirtyStateChanged();

		/* Mark all the Form Parts in the Editor as Stale */
		for (int i = 0; i < this.getPageCount(); i++)
		{
			Object page = this.pages.get(i);
			if (page instanceof IFormPage)
			{
				IFormPage formPage = (IFormPage) page;
				IManagedForm managedForm = formPage.getManagedForm();
				if (managedForm != null)
				{
					for (IFormPart formPart : managedForm.getParts())
					{
						if (formPart instanceof AbstractFormPart)
						{
							((AbstractFormPart) formPart).markStale();
						}
					}
				}
			}
		}

		this.updateTitle();
	}

	/*
	 * Update the Title of the Editor
	 */
	private void updateTitle()
	{
		IEditorInput input = this.getEditorInput();
		this.setPartName(input.getName());
		this.setTitleToolTip(input.getToolTipText());
	}
	
	public void updateTitleImage(){
		IEditorInput input = this.getEditorInput();
		ResourceTypeEditorInput resourceTypeInput = (ResourceTypeEditorInput)input;
		if(hasErrors(resourceTypeInput.getResourceType().getPath())){
			ImageDescriptor descriptor = Utility.getImageDescriptor("resource_type_error.png");
			this.setTitleImage(new Image(Display.getCurrent(), descriptor.getImageData()));
		}else{
			ImageDescriptor descriptor = Utility.getImageDescriptor("resource_type.png");
			this.setTitleImage(new Image(Display.getCurrent(), descriptor.getImageData()));
		}
	}
	
	private boolean hasErrors(String resourceTypeName){
		IFile pluginFile = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(resourceTypeName));
		try {
			IMarker[] errors = pluginFile.findMarkers(MARKER_TYPE, true, IResource.DEPTH_ZERO);
			if(errors.length > 0){
				return true;
			}
		} catch (CoreException e) {
			String msg = "Error fetching validation messages";
			logger.error(msg, e);
		}
		return false;
	}
	
}
