/*
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpage;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CheckboxCellEditor;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormPage;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.part.FileEditorInput;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.ResourceTypeMappingEditor;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.MessageInput;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.util.MessageXMLGenerator;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;
import com.tandbergtv.watchpoint.studio.util.XMLDocumentUtility;

/**
 * The XPath mapping page used to mapping response variables
 * 
 * @author Patrik Araujo
 */
public class XPathMappingPage extends FormPage
{
	private static final String PAGE_TITLE = "XPath Mapping";
	public static final String PAGE_ID = "XPath.Mapping";
	
	private static final String SELECT_COLUMN = "Select";
	private static final String VARIABLE_NAME_COLUMN = "Variable Name";
	private static final String XPATH_COLUMN = "XPath";
	
	private static final String SOURCE_X_PATH_TAG_NAME = "SourceXPath";
	private static final String DESTINATION_X_PATH_TAG_NAME = "DestinationXPath";
	
	private static final String MAPPING_TAG = "Mapping";
	private static final String XPATH_CONFIGURATION_TAG = "xPathConfiguration";
	
	TableViewer messagesTableViewer;
	Table messagesTable;
	TableColumn selectColumn, variableColumn, xpathColumn;
	Label message;
	Button add, delete;
	Composite buttonComposite;
	
	private IResourceTypeService rtService;
	String[] variables ;
	String[] columnNames = {SELECT_COLUMN, VARIABLE_NAME_COLUMN, XPATH_COLUMN};
	private ArrayList<TableMappingObject> inputElements;
	boolean pageModified = false;
	
	
	/**
	 * @param editor
	 *            The Parent Editor
	 */
	public XPathMappingPage(ResourceTypeMappingEditor editor)
	{
		super(editor, PAGE_ID, PAGE_TITLE);
	}
	
	@Override
	public void setFocus() {
		super.setFocus();
		initMessageVariables();
		initCellEditors();
		if(variables.length == 0){
			message.setVisible(true);
			add.setEnabled(false);
		}else{
			message.setVisible(false);
			add.setEnabled(true);
		}
	}
	
	private void initMessageVariables(){
		this.rtService = ServiceFactory.createFactory().createResourceTypeService();
		IProject project = ((FileEditorInput) getEditorInput()).getFile().getProject();
		ResourceType rt = this.rtService.getResourceTypeByName(project.getName());
		String messageUid = getMessageUid();
		NodeDefinition messageNodeDefinition = null;
		Set<NodeDefinitionDTO> nodes = rt.getNodes();
		for (NodeDefinitionDTO nodeDefinitionDTO : nodes) {
			try {
				NodeDefinition nodeDef = SemanticElementUtil.createNodeDefinition(nodeDefinitionDTO.getXml());
				nodeDef.setId(nodeDefinitionDTO.getId());
				if(nodeDef.getUid().equals(messageUid)){
					messageNodeDefinition = nodeDef;
					break;
				}
			} catch (Exception e) {
			}
		}
		List<WPVariable> messageVariables = messageNodeDefinition.getVariables();
		ArrayList<String> variablesNames = new ArrayList<String>();
		for (WPVariable wpVariable : messageVariables) {
			variablesNames.add(wpVariable.getMappedName());
		}
		
		this.variables = variablesNames.toArray(new String[0]);
	}
	
	@Override
	public void init(IEditorSite site, IEditorInput input) {
		super.init(site, input);
		initMessageVariables();
	}
	
	@Override
	protected void createFormContent(IManagedForm managedForm) {
		super.createFormContent(managedForm);
		createPageControls();
		createControlsLayout();
		initCellEditors();
		populateData();
	}
	
	private void createPageControls()
	{
		IManagedForm managedForm = this.getManagedForm();
		ScrolledForm form = managedForm.getForm();
		form.setText("Message Mapping");
		
		FormToolkit toolkit = managedForm.getToolkit();
		
		int style = SWT.SINGLE | SWT.FULL_SELECTION | SWT.H_SCROLL | SWT.V_SCROLL;
		this.messagesTable = toolkit.createTable(form.getBody(), style);
		toolkit.paintBordersFor(this.messagesTable);
		
		this.selectColumn = new TableColumn(messagesTable, SWT.LEFT, 0);
		this.selectColumn.setText("");
		this.selectColumn.setToolTipText("Select");
		this.selectColumn.setResizable(true);
		this.selectColumn.setMoveable(false);
		this.selectColumn.setWidth(25);
		
		this.variableColumn = new TableColumn(messagesTable, SWT.LEFT, 1);
		this.variableColumn.setText(columnNames[1]);
		this.variableColumn.setToolTipText("Variable Name");
		this.variableColumn.setResizable(true);
		this.variableColumn.setMoveable(false);
		this.variableColumn.setWidth(150);

		this.xpathColumn = new TableColumn(messagesTable, SWT.LEFT, 2);
		this.xpathColumn.setText(columnNames[2]);
		this.xpathColumn.setToolTipText("Xpath");
		this.xpathColumn.setResizable(true);
		this.xpathColumn.setMoveable(false);
		this.xpathColumn.setWidth(642);
		
		
		this.buttonComposite = toolkit.createComposite(form.getBody());
		//Add Button
		this.add = toolkit.createButton(buttonComposite, "Add", SWT.PUSH | SWT.CENTER);
	    add.addSelectionListener(new SelectionAdapter() {
	        // Add a mapping to table and refresh the view
		    public void widgetSelected(SelectionEvent e) {
		    	addNewMapping();
	        }
	    });
	    
	    //Delete Button
	    this.delete = toolkit.createButton(buttonComposite, "Delete", SWT.PUSH | SWT.CENTER);
	    this.delete.setEnabled(false);
	    delete.addSelectionListener(new SelectionAdapter() {
	        // Delete a mapping from table and refresh the view
		    public void widgetSelected(SelectionEvent e) {
		    	deleteSelectedMappings();
	        }
	    });
	    
	    message = toolkit.createLabel(form.getBody(), "There's no variable to map in this message!");
	    message.setVisible(false);

		this.messagesTable.setHeaderVisible(true);
		this.messagesTable.setTopIndex(0);
		this.messagesTable.setLinesVisible(true);
		
		this.messagesTableViewer = new TableViewer(messagesTable);
		this.messagesTableViewer.setLabelProvider(new MessageLabelProvider());
		this.messagesTableViewer.setContentProvider(new MessageContentProvider());
		this.messagesTableViewer.setComparator(new ViewerComparator());

		/* Create the Page Contents */
	}
	
	private void initCellEditors(){
		// Create the cell editors
		CellEditor[] editors = new CellEditor[3];
		
		// Column 1 : Select (Checkbox)
		editors[0] = new CheckboxCellEditor(messagesTable);
		// Column 1 : Variable (Combo Box)
		editors[1] = new ComboBoxCellEditor(messagesTable, variables, SWT.READ_ONLY);
		// Column 2 : XPath (Free text)
		TextCellEditor textEditor = new TextCellEditor(messagesTable);
        ((Text) textEditor.getControl()).setTextLimit(60);
        editors[2] = textEditor;
        
        // Assign the cell editors to the viewer 
        messagesTableViewer.setCellEditors(editors);
        messagesTableViewer.setCellModifier(new XpathTableModifier());
        messagesTableViewer.setColumnProperties(columnNames);
	}
	
	private void createControlsLayout(){
		IManagedForm managedForm = this.getManagedForm();
		ScrolledForm form = managedForm.getForm();
		/* Set the Layout for the Section Client in the Form */
		form.getBody().setLayout(this.createGridLayout(1, false));

		/* Set the Layout data for the widgets */
		GridData gridData = new GridData(SWT.FILL, SWT.FILL, false, false);
		
		gridData.minimumHeight = 150;
		gridData.heightHint = 300;
		gridData.widthHint = 800;
		this.messagesTable.setLayoutData(gridData);
		
		this.buttonComposite.setLayoutData(new GridData(SWT.BEGINNING, SWT.BEGINNING, false, false));

		RowLayout rowLayout = new RowLayout(SWT.HORIZONTAL);
		rowLayout.fill = true;
		rowLayout.pack = false;
		rowLayout.marginTop = rowLayout.marginBottom = 0;
		rowLayout.marginLeft = rowLayout.marginRight = 0;
		rowLayout.spacing = 5;
		this.buttonComposite.setLayout(rowLayout);
		
		add.setLayoutData(new RowData());
		delete.setLayoutData(new RowData());
	}
	
	protected GridLayout createGridLayout(int columnCount, boolean makeColumnsEqualWidth)
	{
		GridLayout layout = new GridLayout(columnCount, makeColumnsEqualWidth);
		layout.marginLeft = layout.marginRight = 2;
		layout.marginTop = layout.marginBottom = 2;
		return layout;
	}
	
	private void populateData(){
		Object[] inputElementsArray = getInputElements(((ResourceTypeMappingEditor)this.getEditor()).getResponseInput());
		this.inputElements = new ArrayList<TableMappingObject>();
		for (int i = 0; i < inputElementsArray.length; i++) {
			inputElements.add((TableMappingObject)inputElementsArray[i]);
		}
		this.messagesTableViewer.setInput(inputElements);
	}
	
	public Object[] getInputElements(Object inputElement)
	{
		Object[] elements = new Object[0];
		if (inputElement instanceof IFile){
			IFile input = (IFile) inputElement;
			try {
				InputStream fileContent = input.getContents();
				Document xpath = XMLDocumentUtility.loadFile(fileContent, false);
				NodeList mappings = xpath.getElementsByTagName(MAPPING_TAG);
				if(mappings.getLength() > 0){
					ArrayList<TableMappingObject> mappingsList = new ArrayList<TableMappingObject>();
					for(int i = 0 ; i < mappings.getLength() ; ++i){
						Node mapping = mappings.item(i);
						TableMappingObject mo = new TableMappingObject(mapping);
						mappingsList.add(mo);
					}
					elements = mappingsList.toArray();
				}
			} catch (Exception e) {
				Utility.reportError("Error while reading file content", e);
			}
		} 
		return elements;
	}
	
	private ArrayList<TableMappingObject> getInputElementsList(){
		ArrayList<TableMappingObject> inputElements = new ArrayList<TableMappingObject>();
		TableItem[] items = messagesTableViewer.getTable().getItems();
		for (int i = 0; i < items.length; i++) {
			TableMappingObject mo = (TableMappingObject) items[i].getData();
			inputElements.add(mo);
		}
		return inputElements;
	}
	
	private void deleteSelectedMappings(){
		ArrayList<TableMappingObject> inputElements = getInputElementsList();
		int elementsDeleted = 0;
		for (TableMappingObject mo : inputElements) {
			if(mo.isSelected()){
				messagesTableViewer.remove(mo);
				elementsDeleted++;
			}
		}
		if(elementsDeleted > 0){
			pageModified = true;
			getEditor().editorDirtyStateChanged();
		}
		checkIfThereIsSelectedItems();
	}
	
	private void addNewMapping(){
		TableMappingObject mo = new TableMappingObject();
        mo.setSelected(false);
        mo.setVariableName(variables[0]);
        mo.setXpath("");
        messagesTableViewer.add(mo);
        pageModified = true;
		getEditor().editorDirtyStateChanged();
	}
	
	@Override
	public void doSave(IProgressMonitor monitor) {
		super.doSave(monitor);
		ArrayList<TableMappingObject> inputElements = getInputElementsList();
		try {
			saveResponseMappingFile(inputElements);
			saveWPCLMessageTemplateFile(inputElements);
			this.pageModified = false;
		} catch (TransformerException e) {
			Utility.reportError("Error while saving file changes.", e);
		} catch (IOException e) {
			Utility.reportError("Error while saving file changes.", e);
		} catch (CoreException e) {
			Utility.reportError("Error while saving file changes.", e);
		} catch (ParserConfigurationException e) {
			Utility.reportError("Error while saving file changes.", e);
		}
	}
	
	private void saveWPCLMessageTemplateFile(ArrayList<TableMappingObject> inputElements) throws TransformerException, IOException, CoreException, ParserConfigurationException{
		ArrayList<String> variablesNames = new ArrayList<String>();
		for (TableMappingObject tableMappingObject : inputElements) {
			variablesNames.add(tableMappingObject.getVariableName());
		}
		String messageUid = getMessageUid();
		
		ByteArrayOutputStream fileContent = MessageXMLGenerator.createWPCLMessageXMLFile(messageUid, variablesNames);
		byte[] fileContentArray = fileContent.toByteArray();
		IFile wpclMessageTemplateFile = getWPCLMessageTemplateFile();
		wpclMessageTemplateFile.setContents(new ByteArrayInputStream(fileContentArray) , IFile.KEEP_HISTORY, null);
	}
	
	private void saveResponseMappingFile(ArrayList<TableMappingObject> inputElements) throws TransformerException, IOException, CoreException{
		Document xmlDocument = createDocToSave(inputElements);
		ByteArrayOutputStream fileContent = XMLDocumentUtility.createByteArray(xmlDocument);
		byte[] fileContentArray = fileContent.toByteArray();
		IFile responseFile = (IFile) ((ResourceTypeMappingEditor)this.getEditor()).getResponseInput();
		responseFile.setContents(new ByteArrayInputStream(fileContentArray) , IFile.KEEP_HISTORY, null);
	}
	
	private Document createDocToSave(ArrayList<TableMappingObject> input){
		if( input != null ){
			try {
				DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
				Document doc = factory.newDocumentBuilder().newDocument();
				doc.setXmlVersion("1.0");
				Element rootNode = doc.createElement(XPATH_CONFIGURATION_TAG);
				doc.appendChild(rootNode);
				// Create new Mapping nodes
				for (TableMappingObject mo : input) {
					Element mapping = createMappingNode(mo , doc);
					rootNode.appendChild(mapping);
				}
				return doc;
			} catch (ParserConfigurationException e) {
				Utility.reportError("Error creating document to save.", e);
			}
		}
		return null;
	}
	
	private Element createMappingNode(TableMappingObject mo, Document doc) throws ParserConfigurationException{
			Element mapping = doc.createElement("Mapping");
			Element sourceXPath = doc.createElement("SourceXPath");
			Element destinationXPath = doc.createElement("DestinationXPath");
			org.w3c.dom.Text sourceXPathText = doc.createTextNode(mo.xpath);
			org.w3c.dom.Text destinationXPathText = doc.createTextNode("//Parameter[@Name='" + mo.getVariableName() + "']/Value");
			
			mapping.appendChild(sourceXPath);
			mapping.appendChild(destinationXPath);
			
			sourceXPath.appendChild(sourceXPathText);
			destinationXPath.appendChild(destinationXPathText);
			
			return mapping;
	}
	
	public IFile getWPCLMessageTemplateFile(){
		IFile responseFile = null;
		IEditorInput editorInput = getEditorInput();
		if(editorInput instanceof FileEditorInput){
			IPath filePath = ((FileEditorInput) editorInput).getFile().getFullPath();
			
			IProject project = ((FileEditorInput) editorInput).getFile().getProject();
			
			String messageFolderPath = filePath.removeLastSegments(2).removeFirstSegments(1).toString();
			
			if(project.exists()){
				IFolder messageFolder = project.getFolder(messageFolderPath);
				responseFile = messageFolder.getFile("template/incoming.xml");
				if(responseFile.exists()){
					return responseFile;
				}
			}
		}
		return responseFile;
	}
	
	@Override
	public boolean isDirty() {
		return (this.pageModified || super.isDirty());
	}
	
	private String getMessageUid(){
		IPath filePath = ((FileEditorInput) getEditorInput()).getFile().getFullPath();
		String messageUid = filePath.removeLastSegments(2).removeFirstSegments(3).toString();
		return messageUid;
	}
	
	public void checkIfThereIsSelectedItems(){
		boolean thereIsItemsSelected = false;
		for (int i = 0; i < messagesTable.getItems().length; i++) {
			TableMappingObject itemObject = ( (TableMappingObject) messagesTable.getItems()[i].getData() );
			if(itemObject.isSelected()){
				thereIsItemsSelected = true;
			}
		}
		if(thereIsItemsSelected){
			delete.setEnabled(true);
		}else{
			delete.setEnabled(false);
		}
	}
	
	// ========================================================================
	// ===================== INTERNAL CLASSES FOR TABLE VIEWER
	// ========================================================================

	/*
	 * Cell modifier
	 */
	private class XpathTableModifier implements ICellModifier{

		@Override
		public boolean canModify(Object arg0, String arg1) {
			return true;
		}

		@Override
		public Object getValue(Object element, String property) {
			// Find the index of the column
			int columnIndex = Arrays.asList(columnNames).indexOf(property);

			Object result = null;
			TableMappingObject mo = (TableMappingObject) element;

			switch (columnIndex) {
			case 0: // SELECT COLUMN
				result = mo.isSelected();
				break;
			case 1: // VARIABLE NAME COLUMN
				String stringValue = mo.getVariableName();
				String[] choices = variables;
				int i = choices.length - 1;
				while (!stringValue.equals(choices[i]) && i > 0)
					--i;
				result = new Integer(i);
				break;
			case 2: // XPATH COLUMN
				result = mo.getXpath();
				break;
			}

			return result;
		}

		@Override
		public void modify(Object element, String property, Object value) {
			// Find the index of the column
			int columnIndex = Arrays.asList(columnNames).indexOf(property);
			TableItem item = (TableItem)element;
			TableMappingObject mo = (TableMappingObject) item.getData();
			boolean itemChanged = false;
			switch (columnIndex) {
			case 0: // SELECT COLUMN
				boolean selected = (Boolean) value;
				boolean oldSelectedValue = mo.isSelected();
				if(selected != oldSelectedValue){
					itemChanged = true;
				}
				mo.setSelected(selected);
				checkIfThereIsSelectedItems();
				break;
			case 1: // VARIABLE NAME COLUMN
				String variableName = variables[Integer.parseInt(value.toString())];
				String oldVariableNameValue = mo.getVariableName();
				if(!variableName.equals(oldVariableNameValue)){
					itemChanged = true;
				}
				mo.setVariableName(variableName);
				break;
			case 2: // XPATH COLUMN
				String xpath = ((String)value).trim();
				String oldXPathValue = mo.getXpath();
				if(!xpath.equals(oldXPathValue)){
					itemChanged = true;
				}
				mo.setXpath(xpath);
				break;
			}
			
			messagesTableViewer.update(mo, null);
			if(itemChanged)
				pageModified = true;
			getEditor().editorDirtyStateChanged();
		}
		
	}
	
	private class TableMappingObject{
		private boolean selected;
		private String variableName;
		private String xpath;
		
		public void setSelected(boolean selected) {
			this.selected = selected;
		}

		public boolean isSelected() {
			return selected;
		}

		public String getVariableName() {
			return variableName;
		}

		public void setVariableName(String variableName) {
			this.variableName = variableName;
		}

		public String getXpath() {
			return xpath;
		}

		public void setXpath(String xpath) {
			this.xpath = xpath;
		}
		
		public TableMappingObject(){
			
		}

		public TableMappingObject(Node node) {
			selected = false;
			variableName = getNodeVariableName(node);
			xpath = getNodeXPath(node);
		}
		
		private String getMappingChild(Node node, String childElement){
			NodeList children = node.getChildNodes();
			String elementValue = "";
			for(int j=0; j < children.getLength(); j++) {
				Node curr = children.item(j);
				if(curr.getNodeName().equalsIgnoreCase(childElement)) {
					elementValue = curr.getTextContent().trim();
				} 
			}
			return elementValue;
		}
		
		private String getNodeXPath(Node node){
			return getMappingChild(node, SOURCE_X_PATH_TAG_NAME);
		}
		
		private String getNodeVariableName(Node node){
			String variableName = getMappingChild(node, DESTINATION_X_PATH_TAG_NAME);
			return variableName.split("'")[1];
		}
	}
	
	/*
	 * The Label Provider for the Table
	 */
	private static class MessageLabelProvider extends LabelProvider implements ITableLabelProvider
	{	
		// Names of images used to represent checkboxes
		public static final String CHECKED_IMAGE 	= "checked";
		public static final String UNCHECKED_IMAGE  = "unchecked";
		// For the checkbox images
		public static ImageRegistry imageRegistry = new ImageRegistry();
		
		private Image getImage(boolean isSelected) {
			String key = isSelected ? CHECKED_IMAGE : UNCHECKED_IMAGE;
			return  imageRegistry.get(key);
		}
		
		static {
			ImageDescriptor checkedImage = Utility.getImageDescriptor("checked.gif");
			imageRegistry.put( CHECKED_IMAGE, checkedImage );
			ImageDescriptor unCheckedImage = Utility.getImageDescriptor("unchecked.gif");
			imageRegistry.put( UNCHECKED_IMAGE, unCheckedImage );	
		}
		
		public Image getColumnImage(Object element, int columnIndex)
		{	
			Image image = null;
			// SELECTED_COLUMN?
			if (columnIndex == 0){
				image = getImage(((TableMappingObject) element).isSelected());
			}
			return image;
		}

		public String getColumnText(Object element, int columnIndex)
		{
			if (!(element instanceof TableMappingObject))
				return null;

			TableMappingObject input = (TableMappingObject) element;
			String label = "";
			switch (columnIndex)
			{
			case 0:
				break;
			
			case 1:
				label = input.getVariableName();
				break;

			case 2:
				label = input.getXpath();
				break;
			}

			if (label != null)
				label = label.trim();

			return label;
		}
		
		@Override
		public String getText(Object element)
		{
			if (!(element instanceof MessageInput))
				return null;

			MessageInput input = (MessageInput) element;
			Message message = input.getMessage();
			if (message == null)
				return null;
			
			return message.getName();
		}
	}

	/*
	 * Content Provider for the Messages
	 */
	private class MessageContentProvider implements IStructuredContentProvider
	{
		public Object[] getElements(Object inputElement){
			Object[] elements = null;
			if(inputElement instanceof ArrayList){
				elements = ((ArrayList<?>)inputElement).toArray();
			}
			return elements;
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput){ 
		}

		public void dispose(){
		}
	}

}
