package com.tandbergtv.watchpoint.studio.ui.preferences;

import static com.tandbergtv.watchpoint.studio.ui.preferences.PreferenceConstants.DEPLOYMENT_KEY;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.jface.preference.FieldEditorPreferencePage;
import org.eclipse.jface.preference.StringFieldEditor;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;

import com.tandbergtv.watchpoint.studio.ui.UI;
import com.tandbergtv.watchpoint.studio.ui.WatchPointStudioPlugin;
import com.tandbergtv.watchpoint.studio.ui.util.ValidationMessages;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;

/**
 * This class represents a preference page that
 * is contributed to the Preferences dialog. By 
 * subclassing <samp>FieldEditorPreferencePage</samp>, we
 * can use the field support built into JFace that allows
 * us to create a page that is small and knows how to 
 * save, restore and apply itself.
 * <p>
 * This page is used to modify preferences only. They
 * are stored in the preference store that belongs to
 * the main plug-in class. That way, preferences can
 * be accessed directly via the preference store.
 */
public class DeploymentKeyPreferencePage extends FieldEditorPreferencePage implements IWorkbenchPreferencePage {

	// Any digit
	private static final Pattern DEPLOYMENT_KEY_PATTERN = Pattern.compile("\\d*");
	
	private StringFieldEditor deploymentKey;
	
	public DeploymentKeyPreferencePage() {
		super(GRID);
		setPreferenceStore(WatchPointStudioPlugin.getDefault().getPreferenceStore());
		setDescription(UI.PreferencePage_Title);
	}

	/**
	 * Creates the field editors. Field editors are abstractions of
	 * the common GUI blocks needed to manipulate various types
	 * of preferences. Each field editor knows how to save and
	 * restore itself.
	 */
	public void createFieldEditors() {
		deploymentKey = new StringFieldEditor(DEPLOYMENT_KEY, UI.PreferencePage_DeploymentKey, getFieldEditorParent());
		deploymentKey.setTextLimit(3);
		deploymentKey.setValidateStrategy(StringFieldEditor.VALIDATE_ON_KEY_STROKE);

		addField(deploymentKey);
	}

	/**
	 * 		Validates the properties. Called on each keystroke.
	 */
	@Override
	public void propertyChange(PropertyChangeEvent event) {
		String errorMessage = null;
		boolean isDeploymentKeyValid = true;
		Matcher m = DEPLOYMENT_KEY_PATTERN.matcher(deploymentKey.getStringValue());
		if (!m.matches()) {
			isDeploymentKeyValid = false;
				try {
					errorMessage = ValidationMessages.getInstance().getProperty(ValidationMessageCode.INVALID_DEPLOYMENT_KEY.getCode());
				} catch (Exception e) {}
		}
		setErrorMessage(errorMessage);
		deploymentKey.setErrorMessage(errorMessage);
		setValid(isDeploymentKeyValid);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.IWorkbenchPreferencePage#init(org.eclipse.ui.IWorkbench)
	 */
	public void init(IWorkbench workbench) {
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.preference.FieldEditorPreferencePage#checkState()
	 */
	@Override
	protected void checkState() {
		super.checkState();
		
		if (!isValid())
			return;
		
		setErrorMessage(null);
		setValid(true);
	}
}
