package com.tandbergtv.watchpoint.studio.ui.view.resourcetype.actions;

import static com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants.DESCRIPTION_SEID;
import static com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants.VARIABLE_SEID;
import static com.tandbergtv.watchpoint.studio.ui.util.ServiceErrors.getInstance;
import static org.eclipse.jface.dialogs.MessageDialog.openError;
import static org.eclipse.swt.widgets.Display.getCurrent;

import java.lang.reflect.InvocationTargetException;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.window.Window;
import org.jbpm.gd.common.xml.XmlAdapter;
import org.jbpm.gd.jpdl.model.Description;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.service.INodeDefinitionService;
import com.tandbergtv.watchpoint.studio.service.ServiceException;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.view.ResourceTypeMessageDialog;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;

public class AddParameterAction extends AbstractNodeDefinitionAction {
	
	private NodeDefinitionDTO dto;
	private ResourceTypeMessageDialog dialog;
	
	@Override
	public void run(IAction action) {
		if (dto == null)
			return;
		
		NodeDefinition node = null;
		try {
			node = SemanticElementUtil.createNodeDefinition(dto.getXml());
			node.setId(dto.getId());
        } catch (Exception e) {
			handleException("Error creating message node from xml: " + dto.getName(), e);
			return;
		}
		
		/* get the references for the node if the node is being used by templates or superstates*/
		this.nodeReferences = checkNodeInUse(node);
		
		dialog = buildDialog(node);
		
		if (dialog.open() != Window.OK)
			return;
		
		if (!checkDialogParameters(dialog))
			return;
		
		addParameter( node );
		
		if(dialog.getUpdateReferences()){
			try {
				new TemplateNodesRefactoring().refactorMessageNodes(dialog.getAffectedMessages(), dto, dto.getName());
			} catch (InvocationTargetException e) {
				handleException("Error refactoring the message nodes", e);
			} catch (InterruptedException e) {
				handleException("Error refactoring the message nodes", e);
			}
		}
	}
	
	protected ResourceTypeMessageDialog buildDialog(NodeDefinition node) {
		ResourceTypeMessageDialog dialog = 
			new ResourceTypeMessageDialog(getCurrent().getActiveShell());
		
		dialog.initReferences( buildReferencesMap(dto) );
		dialog.setMessageName(dto.getName());
		
		dialog.setAddMode(true);
		dialog.setBlockOnOpen(true);
		
		return dialog;
	}
	
	protected boolean checkDialogParameters(ResourceTypeMessageDialog dialog) {
		String name = dialog.getName();
		
		if (name == null || name.length() == 0) {
			openError(getCurrent().getActiveShell(), "Error", "Variable name can not be empty.");
			return false;
		}
		
		if (name.contains(":")) {
			openError(getCurrent().getActiveShell(), "Error", "Variable name can not contain ':' character.");
			return false;
		}
		
		String constant = dialog.getConstant();
		
		if (dialog.getConsButtonSelection() && (constant == null || constant.length() == 0)) {
			openError(getCurrent().getActiveShell(), "Error", "Constant value can not be empty.");
			return false;
		}
		
		return true;
	}
	
	@Override
	protected void updateNodeInSuperstate(NodeDefinition nodeSuperstate) {
		addParemeterToNode(nodeSuperstate);
	}
	
	protected void addParemeterToNode( NodeDefinition node ){
		String name = dialog.getName();
		String varDesc = dialog.getDescription();
		String datatype = dialog.getDatatype();
		boolean compositeKey = dialog.isComposite();
		String constant = dialog.getConstant();
		
		if (compositeKey) {
			node.setCompositeKey(name, name);
		} else if (dialog.getConsButtonSelection()) {
			node.setConstantValue(name, constant);
		} else {
			WPVariable variable = (WPVariable) node.getFactory().createById(VARIABLE_SEID);
			
			variable.setMappedName(name);
			variable.setType(datatype);
			
			if (dialog.isOutgoing()) {
				variable.setAccess("read");
			} else {
				variable.setAccess("write");
			}

			((IMessageContainer) node.getNode()).addVariable(variable);
		}
		
		Description description = node.getDescription();
		String desc = (description != null) ? description.getDescription() : null;
		
		if (description == null || desc.trim().length() == 0) {
			description = (Description) node.getFactory().createById(DESCRIPTION_SEID);
			
			if (varDesc != null) {
				description.setDescription("@param " + name + " : " + varDesc);
				node.setDescription(description);
			}
		} else {
			description.setDescription(parseDescription(desc, name, varDesc));
		}
	}
	
	protected void addParameter( NodeDefinition node ) {
		
		
		try {
			XmlAdapter adapter = Utility.getAdapterFromString(dto.getXml(), node.getFactory());
			adapter.initialize(node);
			
			addParemeterToNode(node);
			
			/* Check if the node definition is being used in a superstate. If true, updates the
			   superstate */
			for (IWatchPointDTO referenceDTO : this.nodeReferences) {
				if(referenceDTO instanceof NodeDefinitionDTO){
					NodeDefinitionDTO nodeDTO = (NodeDefinitionDTO) referenceDTO;
					if(nodeDTO.getType().equals(NodeDefinitionType.SuperState)){
						updateSuperstateNodeDefinition(nodeDTO, node, dto.getName());
					}
				}
			}
			
			String xml = Utility.getXMLFromDocument(adapter.getNode().getOwnerDocument());
			dto.setXml(xml);
			
			INodeDefinitionService service = ServiceFactory.createFactory().createNodeDefinitionService();
			
			service.updateNodeDefinitionInFileSystem(dto);
		} catch (ServiceException e) {
			handleException(getInstance().getProperty(e.getServiceErrorCode().getCode()), e);
		}
	}

	@Override
	public void selectionChanged(IAction action, ISelection selection) {
		if (!selection.isEmpty()) {
			Object element = ((TreeSelection) selection).getFirstElement();
			if (element instanceof NodeDefinitionDTO) {
				this.dto = (NodeDefinitionDTO) element;
			}
		}
	}

}
