package com.tandbergtv.watchpoint.studio.ui.view.resourcetype.actions;

import static com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants.TEMPLATE_SEID;
import static com.tandbergtv.watchpoint.studio.ui.util.Utility.TEMPLATE_EDITOR_ID;
import static org.eclipse.ui.IWorkbenchPage.MATCH_ID;

import java.io.ByteArrayInputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.progress.IProgressService;
import org.jbpm.gd.common.model.SemanticElementFactory;
import org.jbpm.gd.common.xml.XmlAdapter;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;
import org.jbpm.gd.jpdl.model.Transition;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.TemplateMessageDTO;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.ui.editor.WatchPointTemplateEditor;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.NodeGroup;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.sync.resource.UpdateMessageCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.template.UpdateTemplateSuperStateCommand;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * The class responsible for update the template nodes given a node definition
 * @author Patrik
 *
 */
public class TemplateNodesRefactoring {
	protected WatchPointTemplateEditor editor = null;
	protected XmlAdapter adapter;
	protected String currentNodeDefinitionName;
	protected NodeDefinitionType nodeTypeToRefactor;
	
	/**
	 * Refactor the templates by removing the given message nodes
	 * @param messages The list with the message nodes to be deleted
	 * @param currentNodeDefinitionName The node definition name
	 * @throws Exception
	 */
	public void deleteMessageNode(List<IWatchPointDTO> messages, String currentNodeDefinitionName) throws Exception{
		final List<IWatchPointDTO> affectedMessages = messages;
		this.currentNodeDefinitionName = currentNodeDefinitionName;
		
		IWorkbench wb = PlatformUI.getWorkbench();
		IProgressService service = wb.getProgressService();
		service.busyCursorWhile( new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) throws InvocationTargetException,
					InterruptedException {
				monitor.beginTask("Removing message ", affectedMessages.size() + 1);
				try {
					deleteMessageNodeWithMonitor(affectedMessages, monitor);
				} catch (Exception e) {
					throw new InterruptedException("Error refactoring templates");
				}
				monitor.done();
			}
		}) ;
	}
	
	private void deleteMessageNodeWithMonitor(List<IWatchPointDTO> messages,  IProgressMonitor monitor) throws Exception{
		/* Group all the messages for the same template. This allow to call the template builder 
		   only once and apply all the messages changes in one shot */
		Map<WorkflowTemplateDTO, List<TemplateMessageDTO>> templateMessagesMap = groupMessageNodes(messages);
		
		for (WorkflowTemplateDTO templateDTO : templateMessagesMap.keySet()){
			WorkflowTemplate template = getTargetTemplate(templateDTO);
			
			deleteTemplateMessageNode(template, templateMessagesMap.get(templateDTO), monitor);
				
			saveTargetTemplate(templateDTO);
		}
	}
	
	private void deleteTemplateMessageNode(WorkflowTemplate template, List<TemplateMessageDTO> messagesToDelete, IProgressMonitor monitor){
		for (IWatchPointDTO messageDTO : messagesToDelete) {
			TemplateMessageDTO message = (TemplateMessageDTO) messageDTO;
			monitor.setTaskName("Removing message : " + message.getTemplateNodeName());
			
			if(message.getNodeType().equals(NodeDefinitionType.SuperState)){
				removeMessageInSuperstate(template, message);
			} else {
				removeMessageInTemplate(template, message);
			}
			
			monitor.worked(1);
		}
	}
	
	private void removeMessageInTemplate(WorkflowTemplate template, TemplateMessageDTO message) {
		final NodeDefinition templateNode = (NodeDefinition)template.getNodeElementByName(message.getTemplateNodeName());
		final WorkflowTemplate thisTemplate = template;
		PlatformUI.getWorkbench().getDisplay().syncExec(new Runnable() {
			public void run() {
				removeNodeAndIncomingTransitions(thisTemplate, templateNode);
			}
		});
	}

	private void removeMessageInSuperstate(WorkflowTemplate template, TemplateMessageDTO message) {
		NodeDefinition superstate = (NodeDefinition)template.getNodeElementByName(message.getTemplateNodeName());
		NodeDefinition superStateNode = getMessageNodeInSuperstate(superstate, currentNodeDefinitionName);
		if(superstate != superStateNode){
			removeNodeAndIncomingTransitions(superstate, superStateNode);
		} else {
			removeNodeAndIncomingTransitions(template, superstate);
		}
	}
	
	private void removeNodeAndIncomingTransitions(NodeElementContainer container, NodeDefinition node){
		container.removeNodeElement(node);
		removeIncomingTransitions(container, node);
	}
	
	private void removeIncomingTransitions(NodeElementContainer container, NodeDefinition targetNode){
		for (NodeElement nodeElement : container.getNodeElements()) {
			Transition transitionToTargetNode = null;
			for ( Transition transition : nodeElement.getTransitions()){
				if( targetNode.getName().equals(transition.getTo()) ){
					transitionToTargetNode = transition;
				}
			}
			if(transitionToTargetNode != null){
				nodeElement.removeTransition(transitionToTargetNode);
			}
		}
	}

	private NodeDefinition getMessageNodeInSuperstate(NodeDefinition superstate, String messageNodeName){
		NodeDefinition templateNode = null;
		
		boolean inSuperState = false;
		NodeGroup nodeGroup = (NodeGroup)superstate.getNode();
		for (NodeElement nodeElement : nodeGroup.getNodeElements()) {
			if(nodeElement instanceof NodeDefinition && ((NodeDefinition) nodeElement).getDefinitionName().equals(messageNodeName) ){
				templateNode = (NodeDefinition)nodeElement;
				inSuperState = true;
			}
		}
		/* If the message is not in the superstate the superstate is the template node */
		if(!inSuperState){
			templateNode = superstate;
		}

		return templateNode;
	}
	
	private NodeDefinition getMessageNode(WorkflowTemplate template, TemplateMessageDTO message){
		NodeDefinition templateNode = null;
		/* If the message is in a superstate get the template node in the superstate */
		if(message.getNodeType().equals(NodeDefinitionType.SuperState)){
			NodeDefinition superstate = (NodeDefinition)template.getNodeElementByName(message.getTemplateNodeName());
			templateNode = getMessageNodeInSuperstate(superstate, currentNodeDefinitionName);
		} else {
			/* If not, get the template node in the template */
			templateNode = (NodeDefinition)template.getNodeElementByName(message.getTemplateNodeName());
		}
		
		return templateNode;
	}
	
	private WorkflowTemplate getTargetTemplate(WorkflowTemplateDTO templateDTO) throws Exception{
		WorkflowTemplate targetTemplate = null;
		IFile templateFile = getTemplateFile(templateDTO);
		targetTemplate = getTemplateFromEditor(templateFile);
		if(targetTemplate == null){
			targetTemplate = getTemplateFromFile(templateFile);
		}
		
		return targetTemplate;
	}
	
	private void saveTargetTemplate(WorkflowTemplateDTO templateDTO) throws Exception{
		if(editor != null){
			PlatformUI.getWorkbench().getDisplay().syncExec(new Runnable() {
				public void run() {
					editor.doSave(null);
				}
			});
		} else {
			IFile templateFile = getTemplateFile(templateDTO);
			saveTemplate(templateFile);
		}
	}
	
	/**
	 * Refactor the given message nodes of the type SuperState with the values in the node definition DTO<br>
	 * This method applies the changes only for SuperState nodes.<br>
	 * The templates containing the SuperState nodes will be updated in order to keep the consistency
	 * with the resource type SuperState.
	 * @param messages The messages to refactor
	 * @param nodeDefinitionDTO The node definition DTO used to updated the values
	 * @param currentNodeName 
	 * @throws InvocationTargetException
	 * @throws InterruptedException
	 */
	public void refactorSuperstateNodes(List<IWatchPointDTO> messages, NodeDefinitionDTO nodeDefinitionDTO, String currentNodeDefinitionName) throws InvocationTargetException, InterruptedException{
		final List<IWatchPointDTO> affectedMessages = messages;
		final NodeDefinitionDTO nodeDTO = nodeDefinitionDTO;
		this.currentNodeDefinitionName = currentNodeDefinitionName;
		this.nodeTypeToRefactor = NodeDefinitionType.SuperState;
		
		IWorkbench wb = PlatformUI.getWorkbench();
		IProgressService service = wb.getProgressService();
		service.busyCursorWhile( new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) throws InvocationTargetException,
					InterruptedException {
				monitor.beginTask("Refactor SuperState ", affectedMessages.size() + 1);
				try {
					refactorTemplateNodesWithMonitor(monitor, affectedMessages, nodeDTO);
				} catch (Exception e) {
					throw new InterruptedException("Error refactoring templates");
				}
				monitor.done();
			}
		}) ;
	}
	
	/**
	 * Refactor the given message nodes with the values in the node definition DTO<br>
	 * The templates containing the messages nodes will be updated in order to keep the consistency
	 * with the resource type messages.
	 * @param messages The messages to refactor
	 * @param nodeDefinitionDTO The node definition DTO used to updated the values
	 * @param currentNodeName 
	 * @throws InvocationTargetException
	 * @throws InterruptedException
	 */
	public void refactorMessageNodes(List<IWatchPointDTO> messages, NodeDefinitionDTO nodeDefinitionDTO, String currentNodeDefinitionName) throws InvocationTargetException, InterruptedException{
		final List<IWatchPointDTO> affectedMessages = messages;
		final NodeDefinitionDTO nodeDTO = nodeDefinitionDTO;
		this.currentNodeDefinitionName = currentNodeDefinitionName;
		this.nodeTypeToRefactor = NodeDefinitionType.MessageNode;
		
		IWorkbench wb = PlatformUI.getWorkbench();
		IProgressService service = wb.getProgressService();
		service.busyCursorWhile( new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) throws InvocationTargetException,
					InterruptedException {
				monitor.beginTask("Refactor message node", affectedMessages.size() + 1);
				try {
					refactorTemplateNodesWithMonitor(monitor, affectedMessages, nodeDTO);
				} catch (Exception e) {
					throw new InterruptedException("Error refactoring templates");
				}
				monitor.done();
			}
		}) ;
	}
	
	
	/**
	 * Refactor the message nodes using an eclipse monitor to report the task progress
	 * @param monitor The monitor which reports the task progress
	 * @param affectedMessages The messages affected by the operation
	 * @param nodeDefinitionDTO The node definition DTO used to updated the values
	 * @throws Exception 
	 */
	private void refactorTemplateNodesWithMonitor( IProgressMonitor monitor, List<IWatchPointDTO> affectedMessages, NodeDefinitionDTO nodeDefinitionDTO) throws Exception{
		/* Group all the messages for the same template. This allow to call the template builder 
		   only once and apply all the messages changes in one shot */
		Map<WorkflowTemplateDTO, List<TemplateMessageDTO>> templateMessagesMap = groupMessageNodes(affectedMessages);
		
		for (WorkflowTemplateDTO templateDTO : templateMessagesMap.keySet()){
			/* Get the target template  */
			WorkflowTemplate template = getTargetTemplate(templateDTO);
			/* Perform the update operation */
			updateTemplateNode(template, templateMessagesMap.get(templateDTO), nodeDefinitionDTO ,monitor);
			/* Save the template */
			saveTargetTemplate(templateDTO);
		}
	}
	
	
	/**
	 * Update the nodes in a given template using an eclipse monitor to report the task
	 * progress.
	 * @param template The given template with the messages to be updated
	 * @param affectedMessages The messages to be updated
	 * @param nodeDto The node definition DTO used to updated the values 
	 * @param monitor The monitor which reports the task progress
	 */
	private void updateTemplateNode(WorkflowTemplate template, List<TemplateMessageDTO> affectedMessages, NodeDefinitionDTO nodeDto, IProgressMonitor monitor){
		for (IWatchPointDTO messageDTO : affectedMessages) {
			TemplateMessageDTO message = (TemplateMessageDTO) messageDTO;
			monitor.setTaskName("Refactoring message : " + message.getTemplateNodeName());
			
			NodeDefinition templateNode = getMessageNode(template, message);
			updateTemplateNode(templateNode, nodeDto);
			
			monitor.worked(1);
		}
	}
	
	/**
	 * Updates a given message node in a given template. <br>
	 * This method uses the UpdatedMessageCommand to perform the update operation.
	 * @param templateNode The node to be updated
	 * @param nodeDefinitionDTO The node definition DTO used to updated the values 
	 */
	protected void updateTemplateNode(NodeDefinition templateNode, NodeDefinitionDTO nodeDefinitionDTO){
		switch (nodeTypeToRefactor){
			case SuperState: 
				UpdateTemplateSuperStateCommand superstateCommand = new UpdateTemplateSuperStateCommand(templateNode, nodeDefinitionDTO);
				superstateCommand.execute();
				break;
			case MessageNode:
				UpdateMessageCommand command = new UpdateMessageCommand(templateNode, nodeDefinitionDTO);
				command.execute();
				break;
		}
	}
	
	/**
	 * Group all the message nodes that belongs to the same template in a map which the template
	 * is the key.
	 * @param affectedMessages The messages affected by the operation
	 * @return A map which the key is the template and the value is a list with the template message nodes.
	 */
	private Map<WorkflowTemplateDTO, List<TemplateMessageDTO>> groupMessageNodes(List<IWatchPointDTO> affectedMessages){
		Map<WorkflowTemplateDTO, List<TemplateMessageDTO>> templateMessagesMap = new HashMap<WorkflowTemplateDTO, List<TemplateMessageDTO>>();
		for (IWatchPointDTO messageDTO : affectedMessages) {
			TemplateMessageDTO message = (TemplateMessageDTO) messageDTO;
			WorkflowTemplateDTO template = message.getTemplate();
			
			if ( !templateMessagesMap.keySet().contains(template) ){
				templateMessagesMap.put(template, new ArrayList<TemplateMessageDTO>());
			} 
			templateMessagesMap.get(template).add(message);
		}
		return templateMessagesMap;
	}
	
	/**
	 * Get the template file in workspace given a WorkflowTemplateDTO
	 * @param workflowTemplateDTO The given WorkflowTemplateDTO
	 * @return The file in the workspace for the given WorkflowTemplateDTO
	 */
	protected IFile getTemplateFile(WorkflowTemplateDTO workflowTemplateDTO){
		String templateStringPath = workflowTemplateDTO.getPath();
		Path templatePath = new Path(templateStringPath);
		IFile templateFile = ResourcesPlugin.getWorkspace().getRoot().getFile(templatePath);
		
		return templateFile;
	}
	
	/**
	 * Get a WorkflowTemplate given a template file
	 * @param templateFile The fiven template file
	 * @return WorkflowTemplate represented by the given file
	 * @throws CoreException In case of error reading the file
	 */
	protected WorkflowTemplate getTemplateFromFile(IFile templateFile) throws CoreException {
		WorkflowTemplate template = (WorkflowTemplate) new SemanticElementFactory(TEMPLATE_EDITOR_ID).createById(TEMPLATE_SEID);

		SemanticElementFactory factory = new SemanticElementFactory(TEMPLATE_EDITOR_ID);
		adapter = Utility.getAdapterFromStream(templateFile.getName(), templateFile.getContents(), factory);
		adapter.initialize(template);
		return template;
	}
	
	/**
	 * Get a template from an open editor
	 * @param templateFile The template file of the template to be opened
	 * @return The WorkflowTemplate if the template is open in an editor or null otherwise.
	 */
	protected WorkflowTemplate getTemplateFromEditor(IFile templateFile) {
		final String filePath = templateFile.getFullPath().toOSString();
		PlatformUI.getWorkbench().getDisplay().syncExec(new Runnable() {
			public void run() {
				IWorkbenchWindow dwindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
				// may be null during eclipse shutdown...
				if (dwindow == null)
					return ;
				IWorkbenchPage page = dwindow.getActivePage();
				if (page == null)
					return ;
				IEditorReference[] editors = page.findEditors(null, TEMPLATE_EDITOR_ID, MATCH_ID);
				if (editors.length > 0) {
					for (int i = 0; i < editors.length; i++) {
						WatchPointTemplateEditor editorOpen = (WatchPointTemplateEditor) editors[i]
								.getEditor(true).getSite().getPart();
			
						FileEditorInput editorInput = (FileEditorInput) editorOpen.getEditorInput();
						if (editorInput.getFile().getFullPath().toOSString()
								.equals(filePath)) {
							editor = editorOpen;
							return;
						}
					}
				}
			}
		});
		
		if (editor != null) {
			return editor.getProcessDefinition();
		}
		
		return null;
	}
	
	/**
	 * Saves the given template file in the file system.
	 * @param templateFile The given template file to be saved
	 * @throws CoreException In case of error saving the file
	 */
	protected void saveTemplate(IFile templateFile) throws CoreException {
		String xml = Utility.getXMLFromDocument(adapter.getNode().getOwnerDocument());
		
		ByteArrayInputStream input = new ByteArrayInputStream(xml.getBytes());
		templateFile.setContents(input, true, false, null);
	}
	

}
