/*
 * Created on Sep 25, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.wizard;

import static com.tandbergtv.watchpoint.studio.ui.preferences.PreferenceConstants.MESSAGE_DISPATCHER_CLASS;
import static com.tandbergtv.watchpoint.studio.ui.preferences.PreferenceConstants.MESSAGE_HANDLER_CLASS;
import static com.tandbergtv.watchpoint.studio.ui.wizard.RTPConstants.ADAPTOR_FILE_NAME;
import static com.tandbergtv.watchpoint.studio.ui.wizard.RTPConstants.ADAPTOR_FOLDER_PATH;
import static com.tandbergtv.watchpoint.studio.ui.wizard.RTPConstants.INCOMING_MAPPING_FILE_NAME;
import static com.tandbergtv.watchpoint.studio.ui.wizard.RTPConstants.OUTGOING_MAPPING_FILE_NAME;
import static com.tandbergtv.watchpoint.studio.ui.wizard.RTPConstants.SCHEMA_FILE_NAME;
import static com.tandbergtv.watchpoint.studio.ui.wizard.RTPConstants.TIMEOUT_WRONG_FORMAT_MESSAGE;
import static com.tandbergtv.watchpoint.studio.ui.wizard.RTPConstants.WPCL_MESSAGE_TEMPLATE_FILE_NAME;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.text.Collator;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Vector;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.wst.xml.core.internal.contentmodel.CMDocument;
import org.eclipse.wst.xml.core.internal.contentmodel.CMElementDeclaration;
import org.eclipse.wst.xml.core.internal.contentmodel.CMNamedNodeMap;
import org.eclipse.wst.xml.ui.internal.wizards.NewXMLGenerator;

import com.tandbergtv.watchpoint.studio.dto.AdaptorType;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.WatchPointStudioPlugin;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.util.MessageXMLGenerator;
import com.tandbergtv.watchpoint.studio.util.ProjectPreferencesUtil;
import com.tandbergtv.watchpoint.studio.util.TimeoutValidation;

/**
 * The Message Wizard is used to create new Resource Type Messages
 * 
 * @author Raj Prakash
 */
@SuppressWarnings("restriction")
public class MessageWizard extends Wizard {
	
	private ResourceType resourceType;
	private String timeout;
	private MessageWizardPage page;
	private Message message;
	private IFile schemaFile;

	public MessageWizard(ResourceType resourceType) {
		this.resourceType = resourceType;
		this.setWindowTitle("New Resource Type Message");
		this.schemaFile = getProjectSchemaFile();
	}

	public Message getMessage() {
		return this.message;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.wizard.Wizard#performFinish()
	 */
	@Override
	public boolean performFinish() {
		try {
			if (!this.isMessageNameUnique(this.resourceType, this.page.getMessageName())) {
				this.page
						.setErrorMessage("A Message with this name already exist for this Resource Type");
				return false;
			}
			
			/* Validates the form page from the wizard. 
			 * In case of problem the wizard is not finished. */
			if(!validateFormPage()){
				return false;
			}
			
			this.message = buildMessage();

			if (this.message != null && this.schemaFile != null) {
				createInitialMapping();
			}
		} catch (CoreException e) {
			Utility.reportError("Error while creating the Message. Error creating mapping files", e);
			return false;
		} catch (Exception e) {
			Utility.reportError("Error while creating the Message.", e);
			return false;
		}

		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.wizard.Wizard#addPages()
	 */
	@Override
	public void addPages() {
		this.page = new MessageWizardPage();
		initMessagePage(this.page);

		addPage(this.page);
	}
		
	private boolean isMessageNameUnique(ResourceType rt, String messageName){
		for (Message message : rt.getMessages()) {
			if(message.getName().equals(messageName)){
				return false;
			}
		}
		return true;
	}
	
	private void initMessagePage(MessageWizardPage page){
		try {
			setComboXsdElements(page);
			initMessageUid(page);
		} catch (MalformedURLException e) {
			Utility.reportError("Error locating schema file.", e);
		}
	}
	
	private void initMessageUid(MessageWizardPage page){
		String systemId = this.resourceType.getSystemId();
		page.setRtSystemId(systemId);
	}

	private IFile getProjectSchemaFile() {
		IProject project = getResourceTypeProject();
		IFile schemaFile = null;
		String fileName = "";
		fileName = ProjectPreferencesUtil.getProjectPreference(project, SCHEMA_FILE_NAME, "");
		if (fileName != null && !fileName.isEmpty()) {
			schemaFile = project.getFile(fileName);
			if (schemaFile.exists()) {
				return schemaFile;
			}
		}
		return schemaFile;
	}

	private IProject getResourceTypeProject() {
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(this.resourceType.determineNameForProject());
		if (project.exists()) {
			return project;
		} else {
			return null;
		}
	}

	private String getSchemaUriString(IFile file) throws MalformedURLException {
		if (file != null) {
			return file.getLocationURI().toString();
		} else {
			return null;
		}
	}

	private void setComboXsdElements(MessageWizardPage page) throws MalformedURLException {
		if (schemaFile != null) {
			String schemaFileName = getSchemaUriString(schemaFile);

			String[] errorInfo = new String[2];
			CMDocument cmDocument = NewXMLGenerator.createCMDocument(schemaFileName, errorInfo);
			CMNamedNodeMap elements = cmDocument.getElements();

			Vector<String> nameNodeVector = new Vector<String>();

			for (int i = 0; i < elements.getLength(); i++) {
				CMElementDeclaration cmElementDeclaration = (CMElementDeclaration) elements.item(i);
				Object value = cmElementDeclaration.getProperty("Abstract");
				if (value != Boolean.TRUE) {
					nameNodeVector.add(cmElementDeclaration.getElementName());
				}
			}

			String[] elementsArray = nameNodeVector.toArray(new String[0]);
			if (elementsArray.length > 0) {
				Arrays.sort(elementsArray, Collator.getInstance());
			}
			page.setComboElements(elementsArray);
		}
	}

	private String getMessageType() {
		String messageType = "outgoing";
		if (buildMessage().isIncoming()) {
			messageType = "incoming";
		}
		return messageType;
	}

	private void createInitialMapping() throws CoreException, Exception {
		createFolders();
		// Create adaptor file
		createAdaptorFile();
		// Create variable mapping file
		createVariableMappingFile();
		// Create empty xpath mapping file
		createXPathMappingFile();
		// Create empty WPCL template message file
		createWPCLMessageTemplate();
	}

	private void createAdaptorFile() throws ParserConfigurationException, TransformerException,
			IOException {
		// Create the adaptor file for the message
		String messageUid = getMessageUID();
		String resourceTypeName = getResourceTypeProject().getName();
		String messageType = getMessageType();
		IPreferenceStore prefs = WatchPointStudioPlugin.getDefault().getPreferenceStore();
		String dispatcherClass = prefs.getString(MESSAGE_DISPATCHER_CLASS);
		String handlerClass = prefs.getString(MESSAGE_HANDLER_CLASS);
		ByteArrayOutputStream adaptorFile = MessageXMLGenerator.createMessageAdaptorFile(
				messageUid, resourceTypeName, messageType, handlerClass, dispatcherClass);
		String adaptorFileRelativeFolder = ADAPTOR_FOLDER_PATH + messageUid;
		createProjectFile(adaptorFileRelativeFolder, ADAPTOR_FILE_NAME, adaptorFile);
	}

	private void createWPCLMessageTemplate() throws ParserConfigurationException,
			TransformerException, IOException {
		// Create empty WPCL template message file
		String messageUid = getMessageUID();
		ByteArrayOutputStream wpclMessageTemplate = MessageXMLGenerator
				.createInitialWPCLMessage(messageUid);
		String wpclMessageTemplateRelativeFolder = ADAPTOR_FOLDER_PATH + messageUid + "/template";
		createProjectFile(wpclMessageTemplateRelativeFolder, WPCL_MESSAGE_TEMPLATE_FILE_NAME,
				wpclMessageTemplate);
	}

	private void createXPathMappingFile() throws ParserConfigurationException,
			TransformerException, IOException {
		// Create empty response mapping file
		ByteArrayOutputStream responseMapping = MessageXMLGenerator.createInitialXPathMappingFile();
		String messageUid = getMessageUID();
		String respMappingRelativeFolder = ADAPTOR_FOLDER_PATH + messageUid + "/mapping";
		createProjectFile(respMappingRelativeFolder, INCOMING_MAPPING_FILE_NAME, responseMapping);
	}

	private void createVariableMappingFile() throws MalformedURLException, Exception {
		// Create request mapping file
		String schemaFileURI = getSchemaUriString(schemaFile);
		String rootElement = page.getSelectedRequestElement();
		if(rootElement != null || this.message.isIncoming()){
			ByteArrayOutputStream requestMapping = MessageXMLGenerator.createInitialVarMappingFile(
					schemaFileURI, rootElement);
			String messageUid = getMessageUID();
			String reqMappingRelativeFolder = ADAPTOR_FOLDER_PATH + messageUid + "/template";
			createProjectFile(reqMappingRelativeFolder, OUTGOING_MAPPING_FILE_NAME, requestMapping);
		}
	}

	private void createFolders() throws CoreException {
		String messageUid = getMessageUID();
		ArrayList<String> entries = new ArrayList<String>();
		entries.add(ADAPTOR_FOLDER_PATH + messageUid + "/mapping");
		entries.add(ADAPTOR_FOLDER_PATH + messageUid + "/template");
		for (String path : entries) {
			IFolder folder = getResourceTypeProject().getFolder(path);
			createFolder(folder);
		}
	}

	private void createFolder(IFolder folder) throws CoreException {
		IContainer parent = folder.getParent();
		if (parent != null && !parent.exists() && parent instanceof IFolder) {
			createFolder((IFolder) parent);
		}
		folder.create(true, true, null);
	}

	private void createProjectFile(String relativePath, String fileName,
			ByteArrayOutputStream fileContent) {
		try {
			IProject project = getResourceTypeProject();
			if (project != null) {
				byte[] fileContentArray = fileContent.toByteArray();
				project.getFile(relativePath + "/" + fileName).create(
						new ByteArrayInputStream(fileContentArray), true, null);
			}
		} catch (CoreException ce) {
			Utility.reportError("Error while creating the Message. Error creating " + relativePath
					+ " file", ce);
		}
	}

	private Message buildMessage() {
		Message message = this.page.getMessageObject();
		message.setResourceType(this.resourceType);
		if(message.isUIDGenerationRequired()){
			IResourceTypeService resourceTypeService = ServiceFactory.createFactory().createResourceTypeService();
			resourceTypeService.generateMessageUID(this.resourceType, message);
		}
		timeout = this.page.getTimeout();
		return message;
	}
	
	private boolean validateFormPage(){
		// Check if the deployment key has been set before generate a automatic UID
		try {
			buildMessage();
		} catch (IllegalArgumentException e) {
			this.page.updateErrorMessage(e.getMessage());
			return false;
		}
		
		// Check the time out expression
		if( !TimeoutValidation.isTimeoutExpressionValid(page.getTimeout()) ){
			page.updateErrorMessage(TIMEOUT_WRONG_FORMAT_MESSAGE);
			return false;
		}
		
		// Check if there's a element selected for an outgoing message
		String rootElement = page.getSelectedRequestElement();
		if(rootElement == null && schemaFile != null && !buildMessage().isIncoming()){
			page.updateErrorMessage("A request element must be defined for outgoing messages.");
				return false;
		}
		
		//Check if there's a message with the same uid
		for (Message message : buildMessage().getResourceType().getMessages()) {
			if(buildMessage().getUid().equals(message.getUid())){
				page.updateErrorMessage("A message with this uid already exist.");
				return false;
			}
		}  
		
		// Check if the resource type adaptor type is NONE
		if(this.resourceType.getAdaptorType() == AdaptorType.NONE){
			page.updateErrorMessage("The Resource Type adaptor type cannot be 'NONE' in order to add messages");
			return false;
		}
		
		return true;
	}
	
	private String getMessageUID(){
		return this.message.getUid();
	}

	public String getTimeout() {
		return timeout;
	}
}
