/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.wizard;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;

/**
 * The wizard page for creating new superstates
 * 
 * @author Imran Naqvi
 * 
 */
public class NewSuperStateWizardPage extends WizardPage {

	private Text nameText;

	private Combo typeCombo, resourceTypeCombo;
	
	private String resourceName;
	
	protected NewSuperStateWizardPage(String resourceName) {
		super("SuperState");
		setTitle("Create SuperState");
		setDescription("Create a new SuperState");
		this.resourceName = resourceName;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		initializeDialogUnits(parent);
		Composite composite = createClientArea(parent);
		createNameText(composite);
		createTypeCombo(composite);
		createResourceTypeCombo(composite);
		setControl(composite);
		Dialog.applyDialogFont(composite);
	}

	/**
	 * @return the name of the node definition entered by the user
	 */
	public String getNodeDefinitionName() {
		return this.nameText.getText();
	}

	/**
	 * @return the node definition type selected in the combo
	 */
	public String getType() {
		return this.typeCombo.getText();
	}

	/**
	 * @return the resource type selected by the combo
	 */
	public ResourceType getResourceType() {
		return ServiceFactory.createFactory().createResourceTypeService().getResourceTypeByName(this.resourceName);
	}

	/*
	 * Sets the layout of the client area.
	 * 
	 * @param parent
	 *            the parent composite of the client area
	 * @return the composite representing the client area
	 */
	private Composite createClientArea(Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);
		FormLayout layout = new FormLayout();
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		composite.setLayout(layout);
		return composite;
	}

	/*
	 * Creates the name text box along with its label.
	 * 
	 * @param parent
	 *            the parent composite of the text
	 */
	private void createNameText(Composite parent) {
		Label label = new Label(parent, SWT.NONE);
		label.setText("Name : ");
		nameText = new Text(parent, SWT.BORDER);
		nameText.setTextLimit(50);

		FormData fd = new FormData();
		fd.left = new FormAttachment(30, 2);
		fd.right = new FormAttachment(90, 0);
		nameText.setLayoutData(fd);
		nameText.addKeyListener(new KeyListener() {
			public void keyPressed(KeyEvent e) {
			}

			public void keyReleased(KeyEvent e) {
				checkNodeDefinitionName();
				getContainer().updateButtons();
			}

			private void checkNodeDefinitionName() {
				String errorMessage = null;
				if (nameText.getText().equals(""))
					errorMessage = "SuperState name can not be empty.";
				else if (nameText.getText().startsWith(" ") || nameText.getText().endsWith(" "))
					errorMessage = "SuperState name is not valid. A SuperState name must not start or end with a blank.";
				else if (nameText.getText().contains("/"))
					errorMessage = "SuperState name is not valid. A SuperState name must not contain a '/' (forward slash)";
				if (errorMessage != null)
					setErrorMessage(errorMessage);
				else
					setErrorMessage(null);
			}
		});
	}

	/*
	 * Creates the node definition type combo box along with its label.
	 * 
	 * @param composite
	 *            the parent composite of the combo
	 */
	private void createTypeCombo(Composite composite) {
		Label label = new Label(composite, SWT.NONE);
		label.setText("Type : ");
		FormData fd = new FormData();
		fd.top = new FormAttachment(nameText, 10);
		label.setLayoutData(fd);

		typeCombo = new Combo(composite, SWT.DROP_DOWN | SWT.SINGLE | SWT.V_SCROLL | SWT.H_SCROLL
				| SWT.READ_ONLY);
		fd = new FormData();
		fd.left = new FormAttachment(30, 2);
		fd.right = new FormAttachment(90, 0);
		fd.top = new FormAttachment(nameText, 10);

		typeCombo.setLayoutData(fd);
		fillTypeCombo();
	}

	/*
	 * Creates the resource type combo box along with its label.
	 * 
	 * @param composite
	 *            the parent composite of the combo
	 */
	private void createResourceTypeCombo(Composite composite) {
		Label label = new Label(composite, SWT.NONE);
		label.setText("Resource Type : ");
		FormData fd = new FormData();
		fd.top = new FormAttachment(typeCombo, 10);
		label.setLayoutData(fd);

		resourceTypeCombo = new Combo(composite, SWT.DROP_DOWN | SWT.SINGLE | SWT.V_SCROLL
				| SWT.H_SCROLL | SWT.READ_ONLY);
		fd = new FormData();
		fd.left = new FormAttachment(30, 2);
		fd.right = new FormAttachment(90, 0);
		fd.top = new FormAttachment(typeCombo, 10);

		resourceTypeCombo.setLayoutData(fd);
		fillResourceTypeCombo();
	}

	/*
	 * Fills the node definition type combo. 
	 */
	private void fillTypeCombo() {
		this.typeCombo.add(NodeDefinitionType.SuperState.toString());
		this.typeCombo.select(0);
	}

	/*
	 * Fills the resource type combo with values from the database. 
	 */
	private void fillResourceTypeCombo() {
		this.resourceTypeCombo.add(resourceName);
		this.resourceTypeCombo.select(0);
	}
}
