package com.tandbergtv.watchpoint.studio.ui.wizard.export;

import static com.tandbergtv.watchpoint.studio.ui.wizard.export.ResourceTypeExportConstants.BIN_INCLUDES_KEY;
import static com.tandbergtv.watchpoint.studio.builder.WatchpointResourceTypeVisitor.MARKER_TYPE;
import java.io.IOException;
import java.util.Properties;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.IExportWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.statushandlers.StatusManager;

import com.tandbergtv.watchpoint.studio.ui.WatchPointStudioPlugin;

public class ResourceTypeExportWizard extends Wizard implements IExportWizard{
	private static final String STORE_SECTION = "PluginExportWizard";
	
	protected IStructuredSelection fSelection;
	protected BaseExportWizardPage fPage;

	@Override
	public void init(IWorkbench workbench, IStructuredSelection selection) {
		fSelection = selection;
		
		IDialogSettings masterSettings = WatchPointStudioPlugin.getDefault().getDialogSettings();
		setDialogSettings(getSettingsSection(masterSettings));
		
		setWindowTitle("Resource Type export");
	}
	
	public IDialogSettings getSettingsSection(IDialogSettings master) {
		String name = getSettingsSectionName();
		IDialogSettings settings = master.getSection(name);
		if (settings == null)
			settings = master.addNewSection(name);
		return settings;
	}
	
	protected String getSettingsSectionName() {
		return STORE_SECTION;
	}
	
	public void addPages() {
		fPage = createPage1();
		addPage(fPage);
	}
	
	protected BaseExportWizardPage createPage1() {
		return new ResourceTypeExportWizardPage(getSelection());
	}

	@Override
	public boolean performFinish() {
		saveSettings();
		if (!PlatformUI.getWorkbench().saveAllEditors(true))
			return false;

		if (!performPreliminaryChecks())
			return false;

		scheduleExportJob();
		return true;
	}
	
	private boolean performPreliminaryChecks(){
		Object[] selectedItems = fPage.getSelectedItems();
		for (int i = 0; i < selectedItems.length; i++) {
			IProject item = (IProject)selectedItems[i];
			
			try {
				// Check if the build.properties has the bin.includes key
				IFile buildPropertiesFile = item.getFile(ResourceTypeExportConstants.BUILD_PROPERTIES_FILE);
				Properties prop = new Properties();
				prop.load(buildPropertiesFile.getContents());
				String includes = prop.getProperty(BIN_INCLUDES_KEY);
				if(includes == null){
					fPage.setErrorMessage("The project " + item.getName() + " has no files selected to export.");
					return false;
				}
				
				// Check if the project has validation errors
				IMarker[] errors = item.getFile("plugin.xml").findMarkers(MARKER_TYPE, true, IResource.DEPTH_ZERO);
				if(errors.length > 0){
					fPage.setErrorMessage("The project " + item.getName() + " has validation errors");
					return false;
				}
			} catch (IOException e) {
				Status status = new Status(IStatus.ERROR, "WatchPoint Studio",IStatus.ERROR, "Error fetching build.properties file for " + item.getName(), e);
				StatusManager.getManager().handle(status, StatusManager.SHOW);
				return false;
			} catch (CoreException e) {
				Status status = new Status(IStatus.ERROR, "WatchPoint Studio",IStatus.ERROR, "Error fetching validation errors for " + item.getName(), e);
				StatusManager.getManager().handle(status, StatusManager.SHOW);
				return false;
			}
		}
		
		return true;
	}
	
	protected void scheduleExportJob() {
		final FeatureExportInfo info = new FeatureExportInfo();
		info.exportSource = fPage.doExportSource();
		info.destinationDirectory = fPage.getDestination();
		info.items = fPage.getSelectedItems();
		info.qualifier = fPage.getQualifier();
		info.exportGroups = fPage.getExportGroups();

		final ResourceTypeExportOperation job = new ResourceTypeExportOperation(info, "Export Resource Type");
		job.setUser(true);
		job.setRule(ResourcesPlugin.getWorkspace().getRoot());
		job.addJobChangeListener(new JobChangeAdapter() {
			public void done(IJobChangeEvent event) {
				
			}
		});
		job.schedule();
	}
	
	
	protected void saveSettings() {
		IDialogSettings settings = getDialogSettings();
		IWizardPage[] pages = getPages();
		for (int i = 0; i < pages.length; i++) {
			((BaseExportWizardPage) pages[i]).saveSettings(settings);
		}
	}
	
	public IStructuredSelection getSelection() {
		return fSelection;
	}

}
