/*
 * Created on Jul 11, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation;

import com.tandbergtv.watchpoint.studio.util.ClassLoaderUtil;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageCodeAndTypeMapper;

/**
 * Abstract Factory used to create Validator objects.
 * 
 * @author Vijay Silva
 */
public abstract class ValidatorFactory
{
	private static final String DEFAULT_FACTORY = "com.tandbergtv.watchpoint.studio.validation.impl.DefaultValidatorFactory";

	/**
	 * Constructor
	 */
	protected ValidatorFactory()
	{
	}

	/**
	 * Return the configured Factory to use for creating Validator Objects.
	 * 
	 * @return The ValidatorFactory
	 */
	public static ValidatorFactory createFactory()
	{
		return ClassLoaderUtil.createFactoryInstance(DEFAULT_FACTORY, ValidatorFactory.class);
	}

    /**
     * Return the configured Factory to use for creating Validator Objects.
     * 
     * @return The ValidatorFactory
     */
    public static ValidatorFactory createFactory(ValidationMessageCodeAndTypeMapper validationTypeMapper)
    {
        return ClassLoaderUtil.createFactoryInstance(DEFAULT_FACTORY, ValidatorFactory.class, validationTypeMapper);
    }

    /**
	 * Create a Validator that can validate an object of the given target class
	 * 
	 * @param <T>
	 *            The type of the object that can be validated by the Validator created.
	 * @param targetClass
	 *            The Class of the object to validate
	 * 
	 * @return the Validator for the specified object class.
	 */
	public abstract <T> IValidator<T> createValidator(Class<T> targetClass);

	/**
	 * Create a Validator that can validate the provided object
	 * 
	 * @param <T>
	 *            The type of the object that can be validated by the Validator created.
	 * @param target
	 *            The object that can be validated by the validator created
	 * 
	 * @return the Validator for the specified object.
	 */
	public abstract <T> IValidator<T> createValidator(T target);

	/**
	 * Create a Validation Rule object, given the class name to instantiate.
	 * 
	 * @param ruleClassName
	 *            The Class Name for the Validation Rule implementation.
	 * 
	 * @return An instance of the Validation Rule.
	 * 
	 * @throws ValidationRuleInstantiationException
	 *             Exception indicating that the Rule class could not be instantiated or that the
	 *             class is not a valid rule.
	 */
	public IValidationRule<?> createValidationRule(String ruleClassName)
			throws ValidationRuleInstantiationException
	{
		try
		{
			Class<?> clazz = Class.forName(ruleClassName);
			if (!(IValidationRule.class.isAssignableFrom(clazz)))
			{
				String msg = "The Class: " + ruleClassName + " does not implement interface: "
						+ IValidationRule.class.getName()
						+ ", cannot instantiate as Validation Rule.";
				throw new ValidationRuleInstantiationException(msg);
			}

			return (IValidationRule<?>) clazz.newInstance();
		}
		catch (ValidationRuleInstantiationException vrie)
		{
			throw vrie;
		}
		catch (Exception ex)
		{
			String msg = "Validation Rule class: " + ruleClassName + " could not be instantiated.";
			throw new ValidationRuleInstantiationException(msg, ex);
		}
	}
}
