/*
 * Created on Nov 9, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.nodeelement;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * Rule that ensures that the list of variables belonging to the Node all have unique names and
 * mapped names.
 * 
 * @param <NE>
 *            The type of the Node Element
 * 
 * @author Vijay Silva
 */
public abstract class NodeVariablesUniqueRule<NE extends NodeElement> extends ValidationRule<NE> {
	/**
	 * Validates that all variables in the Node Element have unique Mapped Names
	 * 
	 * @param target
	 *            The Node Element to validate
	 * 
	 * @return The list of validation messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	public List<ValidationMessage> validateRule(NE target)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		List<Variable> variables = this.getVariables(target);
		if (variables != null)
		{
			Set<String> names = new HashSet<String>();
			Set<String> mappedNames = new HashSet<String>();
			Set<String> duplicateNames = new HashSet<String>();
			Set<String> duplicateMappedNames = new HashSet<String>();

			for (Variable variable : variables)
			{
				/* Check for duplicate names that are not blank */
				String name = variable.getName();
				if (name != null && name.trim().length() > 0)
				{
					name = name.trim();
					if (names.contains(name))
						duplicateNames.add(name);
					else
						names.add(name);
				}

				/* Check for duplicate mapped names that are not blank */
				String mappedName = variable.getMappedName();
				if (mappedName != null && mappedName.trim().length() > 0)
				{
					mappedName = mappedName.trim();
					if (mappedNames.contains(mappedName))
						duplicateMappedNames.add(mappedName);
					else
						mappedNames.add(mappedName);
				}
			}

			this.validateDuplicateNames(target, duplicateNames, messages);
			this.validateDuplicateMappedNames(target, duplicateMappedNames, messages);
		}

		return messages;
	}

	/**
	 * Get the list of variables in the Node that needs to be validated.
	 * 
	 * @param node
	 *            The Node to validate
	 * 
	 * @return The list of variables in the node
	 */
	protected abstract List<Variable> getVariables(NE node);

	// ========================================================================
	// ===================== UNIQUE NAME VALIDATION
	// ========================================================================

	/**
	 * Check to see if the name should be validated for uniqueness. Default implementation returns
	 * true.
	 * 
	 * @return true if validation is required, false otherwise.
	 */
	protected boolean isValidatingUniqueName(NE node)
	{
		return true;
	}

	/**
	 * Generates a validation message for each duplicate mapped name.
	 * 
	 * @param target
	 *            The Node Element being validated
	 * @param duplicates
	 *            The list of duplicate mapped names
	 * @param messages
	 *            the list of validation messages to add new messages to
	 */
	protected void validateDuplicateNames(NE target, Set<String> duplicates,
			List<ValidationMessage> messages)
	{
		if (!this.isValidatingUniqueName(target))
			return;

		for (String name : duplicates)
		{
			ValidationMessageCode code = this.getDuplicateNameCode(target);
			List<String> parameters = new ArrayList<String>();
			parameters.add(name);
			ValidationMessageAdder.getInstance().addValidationMessage(messages, target, code, parameters);
		}
	}

	/**
	 * Get the ValidationMessageCode for the duplicate variable 'Mapped Name' validation error.
	 * 
	 * @param node
	 *            The Node Element being validated.
	 * 
	 * @return The validation message code: ValidationMessageCode.NODE_VARIABLES_DUPLICATE_NAMES
	 * 
	 * @see ValidationMessageCode#NODE_VARIABLES_DUPLICATE_NAMES
	 */
	protected ValidationMessageCode getDuplicateNameCode(NE node)
	{
		return ValidationMessageCode.NODE_VARIABLES_DUPLICATE_NAMES;
	}

	// ========================================================================
	// ===================== UNIQUE MAPPED NAME VALIDATION
	// ========================================================================

	/**
	 * Checks to see if the mapped name should be validated for uniqueness. Default implementation
	 * returns true.
	 * 
	 * @param node
	 *            The Node Element being validated.
	 * 
	 * @return true if validation is required, false otherwise.
	 */
	protected boolean isValidatingUniqueMappedName(NE node)
	{
		return true;
	}

	/**
	 * Generates a validation message for each duplicate mapped name.
	 * 
	 * @param target
	 *            The Node Element being validated
	 * @param duplicates
	 *            The list of duplicate mapped names
	 * @param messages
	 *            the list of validation messages to add new messages to
	 */
	protected void validateDuplicateMappedNames(NE target, Set<String> duplicates,
			List<ValidationMessage> messages)
	{
		if (!this.isValidatingUniqueMappedName(target))
			return;

		for (String name : duplicates)
		{
			ValidationMessageCode code = this.getDuplicateMappedNameCode(target);
			List<String> parameters = new ArrayList<String>();
			parameters.add(name);
			ValidationMessageAdder.getInstance().addValidationMessage(messages, target, code, parameters);
		}
	}

	/**
	 * Get the ValidationMessageCode for the duplicate variable 'Mapped Name' validation error.
	 * 
	 * @param node
	 *            The Node Element being validated.
	 * 
	 * @return The validation message code:
	 *         ValidationMessageCode.NODE_VARIABLES_DUPLICATE_MAPPED_NAMES
	 * 
	 * @see ValidationMessageCode#NODE_VARIABLES_DUPLICATE_MAPPED_NAMES
	 */
	protected ValidationMessageCode getDuplicateMappedNameCode(NE node)
	{
		return ValidationMessageCode.NODE_VARIABLES_DUPLICATE_MAPPED_NAMES;
	}
}
