/**
 * Commands.java
 * Created Aug 24, 2012
 */
package com.tandbergtv.watchpoint.studio.debugger.runtime.debug;

import static com.tandbergtv.watchpoint.studio.debugger.runtime.Services.getService;
import static com.tandbergtv.workflow.core.ProcessPriority.NORMAL;
import static org.jbpm.graph.def.Event.EVENTTYPE_TRANSITION;

import java.io.Serializable;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;
import org.jbpm.JbpmConfiguration;
import org.jbpm.graph.def.Action;
import org.jbpm.graph.def.Node;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.graph.def.Transition;
import org.jbpm.graph.exe.RuntimeAction;
import org.jbpm.instantiation.Delegation;

import com.tandbergtv.workflow.core.AutomaticTaskNode;
import com.tandbergtv.workflow.core.NodeGroup;
import com.tandbergtv.workflow.core.ProcessStatus;
import com.tandbergtv.workflow.core.WorkflowProcess;
import com.tandbergtv.workflow.core.WorkflowTemplate;
import com.tandbergtv.workflow.core.graph.IProcessFactory;
import com.tandbergtv.workflow.core.service.ServiceRegistry;
import com.tandbergtv.workflow.core.service.cache.ICacheService;
import com.tandbergtv.workflow.driver.DriverException;
import com.tandbergtv.workflow.driver.service.IProcessManagerService;

/**
 * Glue class between debugger and runtime
 * 
 * @author Sahil Verma
 */
public class Commands {
	
	private static final Logger logger = Logger.getLogger(Commands.class);
	
	/**
	 * Creates a new process
	 * 
	 * @param template the template
	 * @param parameters initial state
	 * @return
	 */
	public static WorkflowProcess create(ProcessDefinition template, Map<String, Object> parameters) {
		IProcessFactory factory = (IProcessFactory) JbpmConfiguration.Configs.getObject("jbpm.process.factory");
		WorkflowProcess process = factory.create(template, NORMAL, parameters);
		
		process.getRuntimeActions().clear();
		
		return process;
	}
	
	/**
	 * Finds the (parent) process. Because there can be more than one process due
	 * to the subprocesses, we need to find the parent in the group.
	 * 
	 * @param template
	 * @return
	 */
	public static WorkflowProcess find() {
		ICacheService<WorkflowProcess> cache = getCache();
		
		for (Serializable id : cache.getKeys()) {
			WorkflowProcess process = cache.get(id);
			
			if (process.getSuperProcessToken() == null)
				return process;
		}
		
		throw new RuntimeException();
	}
	
	public static Node findNode(WorkflowTemplate template, final String name) {
		NodeFinder visitor = new NodeFinder(name);
		
		template.depthFirstTraversal(visitor);
		
		return visitor.getTarget();
	}
	
	/**
	 * Adds a breakpoint at the specified node
	 * 
	 * @param process
	 * @param node
	 */
	public static void addBreakpoint(WorkflowProcess process, Node node) {
		if (!(node instanceof AutomaticTaskNode || node instanceof NodeGroup))
			return;
		
		Set<?> transitions = node.getArrivingTransitions();
		
		if (transitions == null || transitions.isEmpty())
			return;
		
		for (Object obj : transitions) {
			Transition t = (Transition) obj;
			Action action = new Action(new Delegation(Breakpoint.class.getName()));
			
			process.addRuntimeAction(new RuntimeAction(t, EVENTTYPE_TRANSITION, action));	
		}
		
		logger.info(process.getRootToken() + ", set breakpoint at " + node.getName());
	}
	
	/**
	 * Removes a breakpoint from the specified node
	 * 
	 * @param process
	 * @param node
	 */
	public static void removeBreakpoint(WorkflowProcess process, Node node) {
		Set<?> transitions = node.getArrivingTransitions();
		
		if (transitions == null || transitions.isEmpty())
			return;
		
		for (Object obj : transitions) {
			Transition t = (Transition) obj;
			Iterator<?> i = process.getRuntimeActions().iterator();
			
			while (i.hasNext()) {
				RuntimeAction action = (RuntimeAction) i.next();
				
				if (action.getGraphElement() == t && action.getEventType().equals(EVENTTYPE_TRANSITION))
					i.remove();
			}
		}
		
		logger.info(process.getRootToken() + ", removed breakpoint at " + node.getName());
	}
	
	/**
	 * Suspends process execution
	 * 
	 * @param process
	 */
	public static void suspend(WorkflowProcess process) {
		process.getRootToken().suspend();
	}
	
	/**
	 * Resumes or starts execution based on the status
	 * 
	 * @param process
	 * @throws DriverException
	 */
	public static void resume(WorkflowProcess process) throws DriverException {
		if (process.getRootToken().getStatus() == ProcessStatus.CREATED)
			getService(IProcessManagerService.class).start(process.getRootToken());
		else
			getService(IProcessManagerService.class).resume(process.getRootToken());
	}
	
	@SuppressWarnings("unchecked")
	private static ICacheService<WorkflowProcess> getCache() {
		return (ICacheService<WorkflowProcess>)ServiceRegistry.getDefault().lookup("Process Cache");
	}
}
