/**
 * TemplateDAO.java
 * Created Feb 17, 2010
 */
package com.tandbergtv.watchpoint.studio.external.fs.dao;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.apache.log4j.Logger;

import com.tandbergtv.watchpoint.studio.dataaccess.IPersistenceContext;
import com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.external.fs.FileSystemContext;
import com.tandbergtv.watchpoint.studio.util.FileUtil;

/**
 * DAO for templates, uses a filesystem backend
 * 
 * @author Sahil Verma
 */
public class TemplateDAO implements WorkflowTemplateDTODAI {

	private IPersistenceContext context;
	
	private static final Logger logger = Logger.getLogger(TemplateDAO.class);

	/**
	 * Creates a {@link TemplateDAO}
	 * @param context
	 */
	public TemplateDAO(FileSystemContext context) {
		super();
		this.context = context;
	}

	/**
	 * {@inheritDoc}
	 */
	public List<WorkflowTemplateDTO> findAll() {
		List<WorkflowTemplateDTO> templates = new ArrayList<WorkflowTemplateDTO>();
		String path = FileSystemContext.class.cast(context).getLocation();
		File [] files = listFiles(path);

		if (files.length == 0)
			return templates;

		for (int i = 0; i < files.length; i++) {
			File file = files[i];
			ZipFile zip = null;
			try {
				WorkflowTemplateDTO template = null;
				if (file.getName().endsWith(WorkflowTemplateDTO.DEFAULT_TEMPLATE_PACKAGE_EXTENSION)) {
					zip = new ZipFile(file);
					Enumeration<? extends ZipEntry> entries = zip.entries();
					while (entries.hasMoreElements()) {
						ZipEntry entry = entries.nextElement();
						if (entry.getName().endsWith(WorkflowTemplateDTO.TEMPLATE_DEFINITION_FILE_EXTENSION_NAME)) {
							byte[] templateBytes = FileUtil.readFile(zip.getInputStream(entry));
							template = TemplateParser.parse(new String(templateBytes));
						}
					}
				} else {
					template = TemplateParser.parse(file);
				}

				template.setPath(file.getAbsolutePath());
				templates.add(template);
			} catch (Exception e) {
				logger.warn("Ignored file " + file.getAbsolutePath(), e);
			} finally {
				if (zip != null) {
					try {
						zip.close();
					} catch (IOException e) {}
				}
					
			}
		}

		return templates;
	}

	/**
	 * {@inheritDoc}
	 */
	public List<WorkflowTemplateDTO> findByCurrentVersion() {
		Map<String, WorkflowTemplateDTO> map = new HashMap<String, WorkflowTemplateDTO>();
		List<WorkflowTemplateDTO> templates = findAll();
		
		for (WorkflowTemplateDTO template : templates) {
			String key = template.getName();
			
			if (!map.containsKey(key)) {
				map.put(key, template);
				continue;
			}
			
			WorkflowTemplateDTO other = map.get(key);
			
			if (other.getVersion() < template.getVersion())
				map.put(key, template);
		}
		
		return new ArrayList<WorkflowTemplateDTO>(map.values());
	}

	/**
	 * {@inheritDoc}
	 */
	public IPersistenceContext getPersistenceContext() {
		return this.context;
	}

	protected File[] listFiles(String location) {
		File dir = new File(location);

		if (!dir.exists() || !dir.isDirectory())
			return new File[0];

		File[] files = dir.listFiles(new FilenameFilter() {
			public boolean accept(File dir, String name) {
				return name.toLowerCase().endsWith(WorkflowTemplateDTO.TEMPLATE_DEFINITION_FILE_EXTENSION_NAME)
				    || name.toLowerCase().endsWith(WorkflowTemplateDTO.DEFAULT_TEMPLATE_PACKAGE_EXTENSION);
			}
		});

		if (files == null)
			return new File[0];

		return files;
	}

	/**
	 * {@inheritDoc}
	 */
	public void deleteNodeDefinitionReferences(long nodeDefinitionId) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public void deleteResourceGroupMap(String templateName) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public List<WorkflowTemplateDTO> findByName(String name) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public int getCountByName(String name) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public int getCountByNodeDefinitionUsage(long nodeDefinitionId) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getCountByResourceGroupUsage(String resourceGroupName) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public Integer getCurrentVersionNumberByName(String name) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public List<WorkflowTemplateDTO> findByNodeDefinitionUsage(long nodeDefinitionId, boolean activeonly) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public int getCountByNodeDefinitionUsage(long nodeDefinitionId, boolean activeonly) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public Long getTemplateIdByName(String name) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public Long getTemplateIdByNameForOldestVersion(String name) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public int setInactiveByName(String name) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public WorkflowTemplateDTO create(WorkflowTemplateDTO entity) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public void delete(Long key) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public WorkflowTemplateDTO find(Long key) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 */
	public WorkflowTemplateDTO update(WorkflowTemplateDTO entity) {
		throw new UnsupportedOperationException();
	}

	@Override
	public WorkflowTemplateDTO findByPath(String path) {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public List<WorkflowTemplateDTO> findByMessageUsage(Collection<String> messageUIDs, NodeDefinitionType type) {
		throw new UnsupportedOperationException();
	}

	@Override
	public List<WorkflowTemplateDTO> findBySuperStateUsage(String nodeDefinitionName, String systemID) {
		throw new UnsupportedOperationException();
	}

	@Override
	public List<WorkflowTemplateDTO> findByTemplateUsage(String subProcessName) {
		throw new UnsupportedOperationException();
	}

	@Override
	public List<WorkflowTemplateDTO> findByProjectName(String projectName) {
		throw new UnsupportedOperationException();
	}

	@Override
	public List<WorkflowTemplateDTO> findByResourceTypeUsage(String systemID) {
		throw new UnsupportedOperationException();
	}

	@Override
	public List<WorkflowTemplateDTO> findByResourceGroupUsage(String groupName) {
		throw new UnsupportedOperationException();
	}
}
