/*
 * Created on Aug 10, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.external.wpexport.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.jbpm.gd.jpdl.model.AbstractNode;
import org.jbpm.gd.jpdl.model.Description;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.ProcessState;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.model.AutomaticTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.LoopNode;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.NodeGroup;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.util.ModelHandler;
import com.tandbergtv.watchpoint.studio.util.FileUtil;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;
import com.tandbergtv.watchpoint.studio.util.XMLDocumentUtility;

import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import freemarker.template.TemplateExceptionHandler;

/**
 * Workflow Template document exporter that exports the template to a html document
 * 
 * @author Jerish Lin
 */
public class WorkflowTemplateDocumentExporter {
	private static final String TEMPLATE_DOC = "workflow.html";
	private static final String DOC_TEMPLATE = "workflow_template.html";
	
	private static final String N_A = "N/A";
	
	private static WorkflowTemplateDocumentExporter exporter;
	private static IResourceTypeService resourceTypeService;

	private WorkflowTemplateDocumentExporter() {
		
	}
	
	public synchronized static WorkflowTemplateDocumentExporter getInstance(){
		if (exporter == null){
			exporter = new WorkflowTemplateDocumentExporter();
			resourceTypeService = ServiceFactory.createFactory().createResourceTypeService();
		}
		return exporter;
	}
	
	@SuppressWarnings({ "unchecked", "rawtypes" })
	public void export(IFile templateJpdlFile) throws Exception {
		String xml = new String(FileUtil.readFile(templateJpdlFile.getContents()));
		WorkflowTemplate template = SemanticElementUtil.createWorkflowTemplate(xml);
		
		IFile gpdFile = templateJpdlFile.getParent().getFile(new Path(WorkflowTemplateDTO.GPD_FILE_NAME));
		String gpdXml = new String(FileUtil.readFile(gpdFile.getContents()));
		Document gpdDom = getGpdXMlDom(gpdXml);
		XPath xpath = XPathFactory.newInstance().newXPath();
		
		IFile destFile = templateJpdlFile.getParent().getFile(new Path(TEMPLATE_DOC));
		Map map = new HashMap();
		map.put("templateName", template.getName());
		map.put("templateVersion", template.getVersion());
		map.put("templateDescription", getActualDescription(template.getDescription()));
		map.put("allResourceTypes", getAllResourceType(template));
		map.put("allSubProcesses", getAllSubProcesses(template));
		map.put("requiredVariables", getAllRequiredVariable(template));
		map.put("nodeDescriptions", serializeObjectToJson(generateAllNodeDescriptions(template, gpdDom, xpath)));
		map.put("currentYear", new SimpleDateFormat("yyyy").format(new Date()));
		exportToFile(map, destFile);
	}
	
	private List<NodeElement> getAllNodesInTemplate(WorkflowTemplate template){
		List<NodeElement> nodes = new ArrayList<NodeElement>();
		for (NodeElement node : template.getNodeElements()){
			if (node instanceof LoopNode){
				LoopNode loop = (LoopNode)node;
				for (NodeElement nodeInLoop : loop.getNodeElements()){
					nodes.add(nodeInLoop);
				}
			} 
			nodes.add(node);
		}
		return nodes;
	}

	private String getActualDescription(Description description){
		if (description != null) {
			String tmpDesc = (description.getDescription() + "\n").replaceAll("@.*\n","");
			if (tmpDesc.length() == 0) {
				return tmpDesc;
			}
			return tmpDesc.substring(0, tmpDesc.length() - 1).replaceAll("^\n", "").replaceAll("\n$", "").replace("\n", "<br/>");
		}
		return "";
	}
	
	private String getResourceTypeName(String systemId){
		if (systemId != null){
			ResourceType resourceType = resourceTypeService.getResourceTypeBySystemId(systemId);
			if (resourceType != null){
				return resourceType.getName();
			}
		}
		return systemId + " - [Resource Type Not imported!!]";
	}
	
	private String getSystemIdByMessageId(String messageId){
		System.out.println(messageId);
		String firstTwo = messageId.substring(0,2);
		if (!"01".equals(firstTwo)){
			return firstTwo;
		}
		return messageId.substring(2,4);
	}
	
	private String getResourceTypeNameForNodeDefinition(NodeDefinition nodeDef){
		System.out.println(nodeDef.getName());
		String messageUid = "";
		if (nodeDef.getLabel().equals("SuperState")){
			for (NodeElement subNode : ((NodeGroup) nodeDef.getNode()).getNodeElements()){
				// Possibly Loop node under SuperState
				if (subNode instanceof LoopNode){
					LoopNode loopSubNode = (LoopNode) subNode ;
					for (NodeElement nodeInLoop : loopSubNode.getNodeElements()){
						if (nodeInLoop instanceof NodeDefinition){
							NodeDefinition subNodeDef = (NodeDefinition) nodeInLoop;
							messageUid = ((AutomaticTaskNode) subNodeDef.getNode()).getAction().getGenericElements()[0].getValue();  
							break;
						}
					}
				} 
				if (subNode instanceof NodeDefinition){
					NodeDefinition subNodeDef = (NodeDefinition) subNode;
					messageUid = ((AutomaticTaskNode) subNodeDef.getNode()).getAction().getGenericElements()[0].getValue();  
					break;
				}
			}
		} else {
			messageUid = ((AutomaticTaskNode) nodeDef.getNode()).getAction().getGenericElements()[0].getValue();
		}
		String systemId = getSystemIdByMessageId(messageUid);
		return getResourceTypeName(systemId);
	}
	
	private List<String> getAllResourceType(WorkflowTemplate template){
		List<String> resourceTypes = new ArrayList<String>();
		for (NodeElement node : getAllNodesInTemplate(template)){
			if (node instanceof NodeDefinition){
				String resourceTypeName = getResourceTypeNameForNodeDefinition((NodeDefinition)node);
				if (!resourceTypes.contains(resourceTypeName)){
					resourceTypes.add(resourceTypeName);
				}
			}
		}
		Collections.sort(resourceTypes);
		return resourceTypes;
	}

	private List<String> getAllSubProcesses(WorkflowTemplate template) {
		List<String> subProcesses = new ArrayList<String>();
		for (NodeElement node : getAllNodesInTemplate(template)){
			if (node instanceof ProcessState){
				String templateName = ((ProcessState) node).getSubProcess().getName();
				if (!subProcesses.contains(templateName)){
					subProcesses.add(templateName);
				}
			}
		}
		Collections.sort(subProcesses);
		return subProcesses;
	}
	
	private List<Map<String, String>> getAllRequiredVariable(WorkflowTemplate template) {
		List<Map<String, String>> reqVars = new ArrayList<Map<String, String>>();
		for(WPVariable var : ModelHandler.getStartStateVariables(template.getStartState())){
			if ("read,write,required".equals(var.getAccess())){
				Map<String, String> reqVar = new HashMap<String, String>();
				reqVar.put("name", var.getName());
				reqVar.put("disName", var.getMappedName());
				reqVar.put("type", var.getType());
				reqVars.add(reqVar);
			}
		}
		return reqVars;
	}
	
	private String serializeObjectToJson(Object obj) throws Exception{
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		try {
			new ObjectMapper().writeValue(out, obj);
			return out.toString();
		} finally {
			out.close();
		}
	}
	
	private Document getGpdXMlDom(String gpdXml) throws Exception{
		return XMLDocumentUtility.loadXml(gpdXml);
	}
	
	private Map<String, Object> createGpdInfo(Element gpdNode, Element parentGpdNode){
		Map<String, Object> gpdInfo = new HashMap<String, Object>();
		gpdInfo.put("x", getActualGpdVaule("x", gpdNode, parentGpdNode));
		gpdInfo.put("y", getActualGpdVaule("y", gpdNode, parentGpdNode));
		gpdInfo.put("width", Integer.parseInt(gpdNode.getAttribute("width")));
		gpdInfo.put("height", Integer.parseInt(gpdNode.getAttribute("height")));
		return gpdInfo;
	}
	
	private Integer getActualGpdVaule(String attr, Element gpdNode, Element parentGpdNode){
		return Integer.parseInt(gpdNode.getAttribute(attr)) + 
				(parentGpdNode == null ? 0 : Integer.parseInt(parentGpdNode.getAttribute(attr)));
	}
	
	private List<Element> getNodesInGpdByPath(Node domNode, XPath xpath, String path) throws Exception{
		List<Element> elements = new ArrayList<Element>();
		NodeList nodeList = (NodeList) xpath.compile(path).evaluate(domNode, XPathConstants.NODESET);
		if (nodeList != null){
			for (int i = 0; i < nodeList.getLength(); i++) {
				elements.add((Element) nodeList.item(i));
			}
		}
		return elements;
	}
	
	private Map<String, Map<String, Object>> getAllGpdInfo(Document gpdDom, XPath xpath) throws Exception{
		Map<String, Map<String, Object>> allGpdInfo = new HashMap<String, Map<String, Object>>();
		
		List<Element> nodes = getNodesInGpdByPath(gpdDom, xpath, "/root-container/node");
		for (Element node : nodes){
			allGpdInfo.put(node.getAttribute("name"), createGpdInfo(node, null));
		}
		
		List<Element> loopNodes = getNodesInGpdByPath(gpdDom, xpath, "/root-container/node-container");
		for (Element loopNode : loopNodes){
			allGpdInfo.put(loopNode.getAttribute("name"), createGpdInfo(loopNode, null));
			List<Element> subNodes = getNodesInGpdByPath(loopNode, xpath, "node");
			for (Element subNode : subNodes){
				allGpdInfo.put(subNode.getAttribute("name"), createGpdInfo(subNode, loopNode));
			}
		}
		
		// minXY[0] is the minimal x value, minXY[1] is the minimal y value.
		int[] minXY = fetchMinXY(allGpdInfo);
		//if the minimal x or y is negative, then the x, y for each node need to be adjusted.
		if (minXY[0] < 0 || minXY[1] < 0){
			for (Map<String, Object> gpdInfo : allGpdInfo.values()) {
				if (minXY[0] < 0){
					gpdInfo.put("x", (Integer)(gpdInfo.get("x")) - minXY[0]);
				}
				if (minXY[1] < 0){
					gpdInfo.put("y", (Integer)(gpdInfo.get("y")) - minXY[1]);
				}
			}
		}
		
		return allGpdInfo;
	}
	
	private int[] fetchMinXY(Map<String, Map<String, Object>> allGpdInfo) {
		int[] minXY = new int[]{0,0};
		for (Map<String, Object> gpdInfo : allGpdInfo.values()){
			Integer currentX = (Integer)(gpdInfo.get("x"));
			if (currentX < minXY[0]) {
				minXY[0] = currentX;
			}
			Integer currentY = (Integer)(gpdInfo.get("y"));
			if (currentY < minXY[1]) {
				minXY[1] = currentY;
			}
		}
		return minXY;
	}

	private void fillGpdInfoToNodeDesc(Map<String, Object> nodeDesc, Map<String, Object> gpdInfo){
		if (gpdInfo != null){ 
			nodeDesc.put("x", gpdInfo.get("x"));
			nodeDesc.put("y", gpdInfo.get("y"));
			nodeDesc.put("width", gpdInfo.get("width"));
			nodeDesc.put("height", gpdInfo.get("height"));
		}
	}
	
	private Map<String, Object> createNodeDescription(NodeElement node, Map<String, Object> gpdInfo){
		Map<String, Object> nodeDesc = new HashMap<String, Object>();
		
		nodeDesc.put("nodeName", node.getName());
		nodeDesc.put("nodeType", node.getLabel());
		fillGpdInfoToNodeDesc(nodeDesc, gpdInfo);
		if (node instanceof NodeDefinition) {
			NodeDefinition nodeDef = (NodeDefinition) node;
			nodeDesc.put("resourceType", getResourceTypeNameForNodeDefinition(nodeDef));
			nodeDesc.put("message", nodeDef.getDefinitionName());
			nodeDesc.put("resourceGroup", nodeDef.getResourceGroup().getName());
			nodeDesc.put("subProcess", N_A);
		} else {
			nodeDesc.put("resourceType", N_A);
			nodeDesc.put("message", N_A);
			nodeDesc.put("resourceGroup", N_A);
			if (node instanceof ProcessState){
				ProcessState subProcess = (ProcessState) node;
				nodeDesc.put("subProcess", subProcess.getSubProcess().getName());
			} else {
				nodeDesc.put("subProcess", N_A);
			}
		}
		nodeDesc.put("description", getActualDescription(((AbstractNode) node).getDescription()));
		return nodeDesc;
	}
	
	private List<Map<String, Object>> generateAllNodeDescriptions(WorkflowTemplate template, Document gpdDom, XPath xpath) throws Exception {
		List<Map<String, Object>> nodeDescriptions = new ArrayList<Map<String, Object>>();
		Map<String, Map<String, Object>> allGpdInfo = getAllGpdInfo(gpdDom, xpath);
		
		for (NodeElement node : getAllNodesInTemplate(template)){
			Map<String, Object> gpdInfo = allGpdInfo.get(node.getName());
			nodeDescriptions.add(createNodeDescription(node, gpdInfo));
		}
		
		return nodeDescriptions;
	}

	private void exportToFile(Object map, IFile destFile) throws IOException, CoreException{
		Configuration cfg = new Configuration(Configuration.VERSION_2_3_25);
		cfg.setDefaultEncoding("UTF-8");
		cfg.setTemplateExceptionHandler(TemplateExceptionHandler.RETHROW_HANDLER);
		cfg.setLogTemplateExceptions(false);
		cfg.setClassForTemplateLoading(WorkflowTemplateDocumentExporter.class, "/");
		Template template = cfg.getTemplate(DOC_TEMPLATE);
		
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		OutputStreamWriter writer = new OutputStreamWriter(out);
		
		try {
			template.process(map, writer);
		} catch (TemplateException e) {
			throw new IOException("Error processing freemarker template", e);
		}

		InputStream in = new ByteArrayInputStream(out.toByteArray());
		try{
	
			if (destFile.exists()) {
				destFile.setContents(in, IFile.FORCE, null);
			} else {
				destFile.create(in, IFile.FORCE, null);
			}
		} finally{
			writer.close();
			in.close();
		}
	}
	
//	public static void main(String arg[]){
//		String messageId = "@param asdfasdfasdf\n@ContentProcess\n\nasdfasdfsfas\n";
//		System.out.println(messageId);
//		System.out.println((messageId+"\n").replaceAll("@.*\n","").replaceAll("^\n", "").replaceAll("\n$", ""));
//	}
}
