/*
 * Created on Sep 25, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.dailog;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.StatusDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

/**
 * The Resource Type Parameter Dialog allows users to create / modify Resource Type Parameters
 * 
 * @author Vijay Silva
 */
public class ResourceTypeParameterDialog extends StatusDialog
{
	protected Composite formComposite;

	protected Label nameLabel, valueLabel;

	protected Text nameText, valueText;

	protected IInputValidator nameValidator, valueValidator;

	protected String pluginId;

	protected boolean isValidating = true;

	/* The initial Parameter Name shown */
	private String inputParameterName = null;

	/* The initial Parameter Value shown */
	private String inputParameterValue = null;

	/* The Parameter Name after the dialog is closed */
	private String outputParameterName = null;

	/* The Parameter Value after the dialog is closed */
	private String outputParameterValue = null;

	/* Error Message for the Name Text */
	private String nameErrorMessage = null;

	/* Error Message for the Value Text */
	private String valueErrorMessage = null;

	/**
	 * Constructor when creating a new Parameter
	 * 
	 * @param parent
	 *            The Parent Shell
	 * @param pluginId
	 *            The Plugin Id of the Part using this dialog
	 */
	public ResourceTypeParameterDialog(Shell parent, String pluginId)
	{
		this(parent, pluginId, null, null);
	}

	/**
	 * Constructor when editing a Parameter
	 * 
	 * @param parent
	 *            The Parent Shell
	 * @param pluginId
	 *            The Plugin Id of the Part using this dialog
	 * @param parameterName
	 *            The initial Parameter Name
	 * @param parameterValue
	 *            The initial Parameter Value
	 */
	public ResourceTypeParameterDialog(Shell parent, String pluginId, String parameterName, String parameterValue)
	{
		super(parent);
		this.setTitle("Resource Type Parameter Editor");

		this.pluginId = pluginId;
		this.inputParameterName = parameterName;
		this.inputParameterValue = parameterValue;
	}

	/**
	 * Set the validator to use when validating the Parameter name
	 * 
	 * @param validator
	 *            The validator for the Parameter name (null if no validation required)
	 */
	public void setParameterNameValidator(IInputValidator validator)
	{
		this.nameValidator = validator;
	}

	/**
	 * Set the validator to use when validating the Parameter value
	 * 
	 * @param validator
	 *            The validator for the Parameter value (null if no validation required)
	 */
	public void setParameterValueValidator(IInputValidator validator)
	{
		this.valueValidator = validator;
	}

	/**
	 * Gets the Parameter Name
	 * 
	 * @return The name in the Name Text widget
	 */
	public String getParameterName()
	{
		return this.outputParameterName;
	}

	/**
	 * Gets the Parameter Value
	 * 
	 * @return The name in the Name Text widget
	 */
	public String getParameterValue()
	{
		return this.outputParameterValue;
	}

	// ========================================================================
	// ===================== OVERRIDEN STATUSDIALOG METHODS
	// ========================================================================

	/**
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected Control createDialogArea(Composite parent)
	{
		Composite composite = (Composite) super.createDialogArea(parent);

		this.createDialogControls(composite);
		this.initializeDialogLayouts(composite);

		this.populateInitialData();

		return composite;
	}

	/**
	 * @see org.eclipse.jface.dialogs.Dialog#okPressed()
	 */
	@Override
	protected void okPressed()
	{
		this.validateParameterName();
		this.validateParameterData();
		IStatus currentStatus = this.getStatus();
		if (currentStatus != null && !currentStatus.matches(IStatus.ERROR))
		{
			this.outputParameterName = this.nameText.getText().trim();
			this.outputParameterValue = this.valueText.getText();
			super.okPressed();
		}
	}

	// ========================================================================
	// ===================== UI INITIALIZATION
	// ========================================================================

	/**
	 * Create the dialog controls that must be displayed in the dialog
	 */
	protected void createDialogControls(Composite parent)
	{
		this.formComposite = new Composite(parent, SWT.NONE);

		int orientation = Window.getDefaultOrientation();
		this.nameLabel = new Label(this.formComposite, orientation);
		this.nameLabel.setText("Name: ");
		this.nameText = new Text(this.formComposite, SWT.SINGLE | SWT.BORDER | orientation);
		this.nameText.setTextLimit(50);
		this.nameText.addFocusListener(new ValidatingFocusListener());
		this.nameText.addModifyListener(new ValidatingModifyListener());

		this.valueLabel = new Label(this.formComposite, orientation);
		this.valueLabel.setText("Value: ");
		this.valueText = new Text(this.formComposite, SWT.MULTI | SWT.BORDER | SWT.V_SCROLL
				| SWT.H_SCROLL | orientation);
		this.valueText.setTextLimit(256);
		this.valueText.addFocusListener(new ValidatingFocusListener());
		this.valueText.addModifyListener(new ValidatingModifyListener());
	}

	/**
	 * Initialize the layout of the Dialog Components
	 */
	protected void initializeDialogLayouts(Composite parent)
	{
		GridLayout layout = new GridLayout(2, false);
		layout.marginLeft = layout.marginRight = 2;
		layout.marginTop = layout.marginBottom = 2;
		layout.verticalSpacing = 10;
		this.formComposite.setLayout(layout);

		this.nameLabel.setLayoutData(this.createGridData());
		GridData layoutData = this.createGridData();
		layoutData.widthHint = 220 + this.valueText.getVerticalBar().getSize().x;
		this.nameText.setLayoutData(layoutData);

		this.valueLabel.setLayoutData(this.createGridData());
		layoutData = this.createGridData();
		layoutData.widthHint = 220;
		layoutData.heightHint = 60;
		this.valueText.setLayoutData(layoutData);
	}

	/**
	 * Populate the initial data into the controls
	 */
	protected void populateInitialData()
	{
		this.isValidating = false;

		String name = (this.inputParameterName != null) ? this.inputParameterName.trim() : "";
		String value = (this.inputParameterValue != null) ? this.inputParameterValue : "";
		this.nameText.setText(name);
		this.valueText.setText(value);

		this.nameErrorMessage = null;
		this.valueErrorMessage = null;
		this.updateStatus();

		this.isValidating = true;
	}

	/*
	 * Helper method to create a basic Grid Data element
	 */
	private GridData createGridData()
	{
		return new GridData(SWT.BEGINNING, SWT.BEGINNING, false, false);
	}

	// ========================================================================
	// ===================== DATA VALIDATION
	// ========================================================================

	/* Validate the data displayed */
	private void validateParameterData()
	{
		if (!this.isValidating)
			return;

		String name = this.nameText.getText().trim();
		this.nameErrorMessage = this.validateData(this.nameValidator, name);
		this.valueErrorMessage = this.validateData(this.valueValidator, this.valueText.getText());
		this.updateStatus();
	}

	/*
	 * Validate the Parameter Name and update the name error message
	 */
	private void validateParameterName()
	{
		if (!this.isValidating)
			return;

		String name = this.nameText.getText().trim();
		this.nameErrorMessage = this.validateData(this.nameValidator, name);
		this.updateStatus();
	}

	/*
	 * Validate the Parameter Value and update the value error message
	 */
	private void validateParameterValue()
	{
		if (!this.isValidating)
			return;

		String value = this.valueText.getText();
		this.valueErrorMessage = this.validateData(this.valueValidator, value);
		this.updateStatus();
	}

	/*
	 * Validates the Data given the validator and the data to validate.
	 */
	private String validateData(IInputValidator validator, String data)
	{
		String newError = null;
		if (validator != null)
		{
			newError = validator.isValid(data);
		}

		return newError;
	}

	/*
	 * Update the Status displayed in the Status Dialog
	 */
	private void updateStatus()
	{
		IStatus newStatus = Status.OK_STATUS;
		if (this.nameErrorMessage != null)
		{
			newStatus = new Status(Status.ERROR, this.pluginId, this.nameErrorMessage);
		}
		else if (this.valueErrorMessage != null)
		{
			newStatus = new Status(Status.ERROR, this.pluginId, this.valueErrorMessage);
		}

		this.updateStatus(newStatus);
	}

	// ========================================================================
	// ===================== EVENT LISTENERS
	// ========================================================================

	/*
	 * Modify Listener implementation that performs validation when the controls are modified
	 */
	private class ValidatingModifyListener implements ModifyListener
	{
		/**
		 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
		 */
		public void modifyText(ModifyEvent event)
		{
			if (event.getSource() == ResourceTypeParameterDialog.this.nameText)
			{ // Validate the Name
				ResourceTypeParameterDialog.this.validateParameterName();
			}
			else if (event.getSource() == ResourceTypeParameterDialog.this.valueText)
			{ // Validate the Value
				ResourceTypeParameterDialog.this.validateParameterValue();
			}
		}
	}

	/*
	 * Focus Listener implementation that performs validation when the controls lose focus
	 */
	private class ValidatingFocusListener extends FocusAdapter
	{
		/**
		 * @see org.eclipse.swt.events.FocusAdapter#focusLost(org.eclipse.swt.events.FocusEvent)
		 */
		@Override
		public void focusLost(FocusEvent event)
		{
			if (event.getSource() == ResourceTypeParameterDialog.this.nameText)
			{ // Trim the Name Field, and validate
				String name = ResourceTypeParameterDialog.this.nameText.getText();
				if (!name.trim().equals(name))
					ResourceTypeParameterDialog.this.nameText.setText(name.trim());

				ResourceTypeParameterDialog.this.validateParameterName();
			}
			else if (event.getSource() == ResourceTypeParameterDialog.this.valueText)
			{ // Validate the Value Field
				ResourceTypeParameterDialog.this.validateParameterValue();
			}
		}
	}
}
