package com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart;

import static com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType.MessageNode;
import static java.lang.System.getProperty;

import java.util.Comparator;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TreeSet;

import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.jbpm.gd.jpdl.model.Description;

import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.ResourceTypeEditorInput;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;

public class DocumentSectionPart extends ResourceTypeSectionPart {

	private static final int MESSAGE_UID_INDEX = 0;

	private static final int MESSAGE_NAME_INDEX = 1;

	private static final int MESSAGE_DESCRIPTION_INDEX = 2;

	private Set<NodeDefinitionDTO> nodes = null;

	private ResourceType resourceType = null;

	TableViewer messagesTableViewer;

	Table messagesTable;

	TableColumn nameColumn, uidColumn, descriptionColumn;

	Label connectionLabel, msgLabel, msgDetailLabel, nameLabel, descriptionLabel, resourcedescLabel;

	Label[] labels;

	// ========================================================================
	// ===================== CONSTRUCTORS
	// ========================================================================

	/**
	 * @param parentPart
	 *            The Part that contains this composite
	 * @param parent
	 *            The Parent Composite containing the section to be created
	 * @param toolkit
	 *            The toolkit to create the section with
	 * @param style
	 *            The style bits to create the section with
	 */
	public DocumentSectionPart(IEditorPart parentPart, Composite parent, FormToolkit toolkit, int style) {
		super(parentPart, parent, toolkit, style);

	}

	// ========================================================================
	// ===================== USER CONTROL INITIALIZATION
	// ========================================================================

	protected void createSectionClientControls(IManagedForm managedForm) {
		FormToolkit toolkit = managedForm.getToolkit();
		Display display = this.getSection().getDisplay();
		FontData fontdata = new FontData("Arial", 10, SWT.BOLD);
		Font font = new Font(display, fontdata);
		this.connectionLabel = 
			toolkit.createLabel(this.sectionClient, "Connection Type: " + resourceType.getConnectionType().getName());
		this.connectionLabel.setFont(font);

		fontdata = new FontData("Arial", 10, SWT.NONE);
		font = new Font(display, fontdata);

		this.resourcedescLabel = 
			toolkit.createLabel(this.sectionClient, getDescription(), SWT.WRAP);
		this.resourcedescLabel.setFont(font);

		fontdata = new FontData("Arial", 11, SWT.BOLD);
		font = new Font(display, fontdata);
		this.msgLabel = toolkit.createLabel(this.sectionClient, "Summary");
		this.msgLabel.setFont(font);

		/* The Messages Table and Buttons */
		this.initializeMessagesTable(managedForm);

		/* The Labels */
		msgDetailLabel = toolkit.createLabel(this.sectionClient, "Node Details");
		this.msgDetailLabel.setFont(font);

		nodes = (resourceType != null) ? resourceType.getNodes() : null;

		if (nodes == null)
			return;

		Set<NodeDefinitionDTO> sorted = new TreeSet<NodeDefinitionDTO>(new Comparator<NodeDefinitionDTO>() {
			@Override
			public int compare(NodeDefinitionDTO o1, NodeDefinitionDTO o2) {
				return o1.getName().compareTo(o2.getName());
			}
		});

		sorted.addAll(nodes);

		int i = 0;
		labels = new Label[nodes.size()];

		for (NodeDefinitionDTO dto : sorted) {
			labels[i++] = new Label(this.sectionClient, SWT.SEPARATOR | SWT.HORIZONTAL);
			
			NodeDefinition node = convertoNode(dto);
			String name = "";
			String description = "";
			
			if (node != null)
				name = node.getName();
			
			if (dto.getType() == MessageNode) {
				String msg = dto.getMessages().iterator().next().getDescription();
				
				if (msg != null)
					description =  msg + getProperty("line.separator");
			}
			
			Description desc = node.getDescription();
			
			if (desc != null)
				description += desc.getDescription();
			
			createNodeDetails(managedForm, name, description);
		}
	}

	/*
	 * Initialize the Messages Table
	 */
	private void initializeMessagesTable(IManagedForm managedForm) {
		FormToolkit toolkit = managedForm.getToolkit();
		int style = SWT.SINGLE;
		this.messagesTable = toolkit.createTable(this.sectionClient, style);

		toolkit.paintBordersFor(this.messagesTable);
		this.uidColumn = new TableColumn(messagesTable, SWT.LEFT, 0);
		this.uidColumn.setText("UID");
		this.uidColumn.setToolTipText("UID");
		this.uidColumn.setResizable(true);
		this.uidColumn.setMoveable(false);
		this.uidColumn.setWidth(110);

		this.nameColumn = new TableColumn(messagesTable, SWT.LEFT, 1);
		this.nameColumn.setText("Name");
		this.nameColumn.setToolTipText("Name");

		this.nameColumn.setResizable(true);
		this.nameColumn.setMoveable(false);
		this.nameColumn.setWidth(220);

		this.descriptionColumn = new TableColumn(messagesTable, SWT.LEFT, 2);
		this.descriptionColumn.setText("Description");
		this.descriptionColumn.setToolTipText("Description");
		this.descriptionColumn.setResizable(true);
		this.descriptionColumn.setMoveable(false);
		this.descriptionColumn.setWidth(1024);

		this.messagesTable.setHeaderVisible(true);
		this.messagesTable.setLinesVisible(true);

		this.messagesTableViewer = new TableViewer(messagesTable);
		this.messagesTableViewer.setLabelProvider(new MessageLabelProvider());
		this.messagesTableViewer.setContentProvider(new MessageContentProvider());
		this.messagesTableViewer.setComparator(new ViewerComparator());
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#initializeSectionClientLayout(org.eclipse.ui.forms.IManagedForm)
	 */
	@Override
	protected void initializeSectionClientLayout(IManagedForm managedForm) {
		/* Set the Layout for the Section Client in the Form */
		this.sectionClient.setLayout(this.createGridLayout(1, false));

		/* Set the Layout data for the widgets */
		GridData gridData = new GridData(SWT.BEGINNING, SWT.FILL, false, false);

		gridData.minimumHeight = 100;
		gridData.heightHint = 200;
		gridData.minimumWidth = 800;
		gridData.widthHint = 800;
		this.messagesTable.setLayoutData(gridData);

		int labelWidth = this.messagesTable.getSize().x;
		gridData = new GridData(SWT.FILL, SWT.BEGINNING, true, false);
		gridData.widthHint = labelWidth;
		this.connectionLabel.setLayoutData(gridData);

		gridData = new GridData(SWT.NONE, SWT.BEGINNING, true, false);
		gridData.widthHint = 900;
		this.resourcedescLabel.setLayoutData(gridData);

		nodes = resourceType.getNodes();

		for (int i = 0; i < nodes.size(); i++) {
			gridData = new GridData(SWT.FILL, SWT.BEGINNING, true, false);
			gridData.widthHint = labelWidth;
			gridData.horizontalSpan = 2;
			labels[i].setLayoutData(gridData);
		}
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#populateSectionData()
	 */
	@Override
	protected void populateSectionData() {
		this.messagesTableViewer.setInput(this.input);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#setWidgetPermissions()
	 */
	@Override
	protected void setWidgetPermissions() {
	}

	/*
	 * Get the Selection for the table
	 */

	// ========================================================================
	// ===================== INTERNAL CLASSES FOR TABLE VIEWER
	// ========================================================================

	/*
	 * The Label Provider for the Table
	 */
	private class MessageLabelProvider extends LabelProvider implements ITableLabelProvider {
		/**
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object, int)
		 */
		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}

		/**
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
		 */
		public String getColumnText(Object element, int columnIndex) {
			if (!(element instanceof MessageInput))
				return null;

			MessageInput input = (MessageInput) element;
			Message message = input.getMessage();
			if (message == null)
				return null;

			String label = "";
			switch (columnIndex) {
			case MESSAGE_UID_INDEX:
				label = message.getUid();
				break;

			case MESSAGE_NAME_INDEX:
				label = message.getName();
				break;

			case MESSAGE_DESCRIPTION_INDEX:
				label = message.getDescription();
				break;
			}

			if (label != null)
				label = label.trim();

			return label;
		}

		/**
		 * @see org.eclipse.jface.viewers.LabelProvider#getText(java.lang.Object)
		 */
		@Override
		public String getText(Object element) {
			if (!(element instanceof MessageInput))
				return null;

			MessageInput input = (MessageInput) element;
			Message message = input.getMessage();
			if (message == null)
				return null;

			return message.getName();
		}
	}

	/*
	 * Content Provider for the Messages
	 */
	private class MessageContentProvider implements IStructuredContentProvider {
		/**
		 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
		 */
		public Object[] getElements(Object inputElement) {
			Object[] elements = null;

			if (inputElement instanceof ResourceTypeEditorInput) {
				ResourceTypeEditorInput input = (ResourceTypeEditorInput) inputElement;
				ResourceType resourceType = input.getResourceType();
				boolean isEditable = input.canEdit();
				Set<Message> messages = (resourceType != null) ? resourceType.getMessages() : null;

				if (messages != null) {
					elements = new Object[messages.size()];
					int counter = 0;
					for (Message message : messages) {
						elements[counter++] = new MessageInput(message, isEditable);
					}
				}
			} else {
				elements = new Object[0];
			}

			return elements;
		}

		/**
		 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer,
		 *      java.lang.Object, java.lang.Object)
		 */
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			/* Do nothing */
		}

		/**
		 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
		 */
		public void dispose() {
		}
	}

	private String getDescription() {
		String description = resourceType.getDescription();

		if (description == null)
			return description;

		StringTokenizer token = new StringTokenizer(resourceType.getDescription(), "\t\n\r\f");
		StringBuilder builder = new StringBuilder();

		while (token.hasMoreTokens())
			builder.append(token.nextToken());

		return builder.toString();
	}

	private void createNodeDetails(IManagedForm managedForm, String name, String description) {
		FormToolkit toolkit = managedForm.getToolkit();
		Display display = this.getSection().getDisplay();
		FontData fontdata = new FontData("Arial", 9, SWT.BOLD | SWT.ITALIC);
		Font font = new Font(display, fontdata);
		this.nameLabel = toolkit.createLabel(this.sectionClient, name);
		this.nameLabel.setFont(font);

		fontdata = new FontData("Arial", 9, SWT.NONE);
		font = new Font(display, fontdata);
		this.descriptionLabel = toolkit.createLabel(this.sectionClient, description);
		this.descriptionLabel.setFont(font);
	}

	protected NodeDefinition convertoNode(NodeDefinitionDTO node) {
		NodeDefinition element = null;

		try {
			element = SemanticElementUtil.createNodeDefinition(node.getXml());
		} catch (Exception e) {
		}

		return element;
	}

	public void setResourceType(ResourceType rtype) {
		this.resourceType = rtype;
	}
}
