/*
 * Created on Sep 26, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart;

import java.util.Map;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.widgets.FormToolkit;

import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.eventhandler.FormPartDirtyModifyListener;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.eventhandler.TextTrimFocusListener;

/**
 * Section Part for the Resource Type Initialization Configuration
 * 
 * @author Vijay Silva
 */
public class InitializationSectionPart extends ResourceTypeSectionPart
{
	protected Label classLabel, parametersLabel;

	protected Button requiredButton;

	protected Group configGroup;

	protected Text classText;

	protected Composite parametersComposite;

	protected ParametersCompositeContainer parametersContainer;

	// ========================================================================
	// ===================== CONSTRUCTORS
	// ========================================================================

	/**
	 * @param parentPart
	 *            The Part that contains this composite
	 * @param parent
	 *            The Parent Composite containing the section to be created
	 * @param toolkit
	 *            The toolkit to create the section with
	 * @param style
	 *            The style bits to create the section with
	 */
	public InitializationSectionPart(IEditorPart parentPart, Composite parent, FormToolkit toolkit,
			int style)
	{
		super(parentPart, parent, toolkit, style);
	}

	// ========================================================================
	// ===================== OVERRIDDEN SECTION PART METHODS
	// ========================================================================

	/**
	 * @see org.eclipse.ui.forms.AbstractFormPart#commit(boolean)
	 */
	@Override
	public void commit(boolean onSave)
	{
		super.commit(onSave);

		/* Update the Resource Type with the data in the widgets */
		ResourceType resourceType = this.getResourceType();
		if (resourceType != null)
		{
			resourceType.setInitializationRequired(this.requiredButton.getSelection());
			resourceType.setInitializationStrategyClass(this.classText.getText());

			Map<String, String> parameters = this.parametersContainer.getParameters();
			resourceType.setInitializationParameters(parameters);
		}
	}

	// ========================================================================
	// ===================== USER CONTROL INITIALIZATION
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#createSectionClientControls(org.eclipse.ui.forms.IManagedForm)
	 */
	@Override
	protected void createSectionClientControls(IManagedForm managedForm)
	{
		FormToolkit toolkit = managedForm.getToolkit();

		/* The Required Button */
		this.requiredButton = toolkit.createButton(this.sectionClient, "Requires Initialization",
				SWT.CHECK | SWT.LEFT);
		this.requiredButton.addSelectionListener(new ButtonSelectionListener());

		/* The Group containing configuration data */
		this.configGroup = new Group(this.sectionClient, SWT.NONE);
		toolkit.adapt(this.configGroup);
		toolkit.paintBordersFor(this.configGroup);

		/* The Strategy Class Name */
		this.classLabel = toolkit.createLabel(this.configGroup, "Initialization Strategy Class: ");
		this.classText = toolkit.createText(this.configGroup, "");
		this.classText.setTextLimit(256);
		this.classText.addFocusListener(new TextTrimFocusListener());
		this.classText.addModifyListener(new FormPartDirtyModifyListener(this));

		/* The Parameters */
		this.parametersLabel = toolkit.createLabel(this.configGroup, "Initialization Parameters: ");
		this.parametersComposite = toolkit.createComposite(this.configGroup);
		toolkit.paintBordersFor(this.parametersComposite);
		this.parametersContainer = new ParametersCompositeContainer(this.editorPart, this, toolkit,
				this.parametersComposite);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#initializeSectionClientLayout(org.eclipse.ui.forms.IManagedForm)
	 */
	@Override
	protected void initializeSectionClientLayout(IManagedForm managedForm)
	{
		/* Set the Layout for the Section Client in the Form */
		this.sectionClient.setLayout(this.createGridLayout(1, false));

		/* Set the default Layout data for the widgets */
		this.requiredButton.setLayoutData(this.createGridData());
		this.configGroup.setLayoutData(this.createGridData());

		/* The Configuration Group Layout */
		this.configGroup.setLayout(this.createGridLayout(2, false));
		this.classLabel.setLayoutData(this.createLabelGridData());
		this.parametersLabel.setLayoutData(this.createLabelGridData());

		/* Set the Layout data for the ClassName Text */
		GridData layoutData = this.createGridData();
		layoutData.widthHint = this.getDefaultClassNameTextWidth();
		this.classText.setLayoutData(layoutData);

		/* Set the Layout data for the Parameters Parent Composite */
		this.parametersComposite.setLayoutData(this.createGridData());
	}

	// ========================================================================
	// ===================== DISPLAYING THE INPUT DATA
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#populateSectionData()
	 */
	@Override
	protected void populateSectionData()
	{
		ResourceType resourceType = this.getResourceType();
		boolean isEditable = this.isInputEditable();

		if (resourceType != null)
		{
			this.requiredButton.setSelection(resourceType.isInitializationRequired());
			String strategyClassName = resourceType.getInitializationStrategyClass();
			this.classText.setText(this.formatValue(strategyClassName, true));

			Map<String, String> parameters = resourceType.getInitializationParameters();
			this.parametersContainer.setInput(parameters, isEditable);
		}
		else
		{
			this.requiredButton.setSelection(false);
			this.classText.setText("");
			this.parametersContainer.setInput(null, isEditable);
		}
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#setWidgetPermissions()
	 */
	@Override
	protected void setWidgetPermissions()
	{
		boolean isEditable = this.isInputEditable();
		this.requiredButton.setEnabled(isEditable);
		this.classText.setEditable(isEditable);
		this.updateConfigControlAccess();
	}

	/*
	 * Set the access for the controls in the Configuration Group
	 */
	private void updateConfigControlAccess()
	{
		boolean isRequired = this.requiredButton.getSelection();
		this.configGroup.setEnabled(isRequired);
		this.classText.setEnabled(isRequired);
		this.parametersContainer.setEnabled(isRequired);
	}

	// ========================================================================
	// ===================== EVENT HANDLERS AND EVENT HANDLING
	// ========================================================================

	private void handleRequiredButtonSelected()
	{
		if (!this.isInputEditable())
			return;

		this.updateConfigControlAccess();
		this.configGroup.redraw();
	}

	/*
	 * Event Handler Class for the Button check
	 */
	private class ButtonSelectionListener implements SelectionListener
	{
		/*
		 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
		 */
		public void widgetDefaultSelected(SelectionEvent event)
		{
			InitializationSectionPart.this.handleRequiredButtonSelected();
		}

		/*
		 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
		 */
		public void widgetSelected(SelectionEvent event)
		{
			InitializationSectionPart.this.handleRequiredButtonSelected();
			if (!InitializationSectionPart.this.isDirty())
				InitializationSectionPart.this.markDirty();
		}
	}
}
