/**
 * UpdateMessageCommand.java
 * Created Apr 20, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.sync.resource;

import java.util.ArrayList;
import java.util.List;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiff;
import com.tandbergtv.watchpoint.studio.ui.sync.command.AbstractDiffCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.util.ISynchronizationContext;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;

/**
 * Updates properties of a message
 * 
 * @author Sahil Verma
 */
public class UpdateMessageCommand extends AbstractDiffCommand {

	protected NodeDefinitionDTO node;
	
	protected NodeDefinition templateNode;
	
	/**
	 * @param name
	 * @param diff
	 * @param context
	 */
	public UpdateMessageCommand(String name, IDiff diff, ISynchronizationContext context) {
		super(name, diff, context);
	}

	public UpdateMessageCommand(NodeDefinition templateNode, NodeDefinitionDTO node) {
		this.templateNode = templateNode;
		this.node = node;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void execute() {
		/* This is the current node */
		NodeDefinition currentNode = createSemanticElement();

		templateNode.setDefinitionName(node.getName());
		
		// Remove all variables that are not present on the Node anymore
		List<WPVariable> toRemoveVars = new ArrayList<WPVariable>(); // to avoid concurrent modification exception
		for (WPVariable templateVariable : templateNode.getVariables()) {
			if (!currentNode.hasVariable(templateVariable.getMappedName())) {
				toRemoveVars.add(templateVariable);
			}
		}
		IMessageContainer container = (IMessageContainer) templateNode.getNode();
		for (WPVariable toRemove : toRemoveVars) {
			if (toRemove.isCompositeKey()) {
				container.removeCompositeKey(toRemove.getMappedName());
			} else {
				container.removeVariable(toRemove.getMappedName());
			}
		}
		
		// Add the new variables
		for (WPVariable newVariable : currentNode.getVariables()) {
			WPVariable variable = templateNode.getVariable(newVariable.getMappedName());

			if (variable == null) {
				if (newVariable.isCompositeKey()) {
					// For composite keys, the variable in the template must be mapped with the same name as the node name 
					templateNode.setCompositeKey(newVariable.getMappedName(), newVariable.getMappedName());
				} else {
					((IMessageContainer) templateNode.getNode()).addVariable(newVariable);
				}
			} else {
				// If variable exists, adjust the correct variable type...
				// for backward compatibility reasons, type NULL must be considered "String"
				if (variable.getType() == null 
				|| !variable.getType().equals(newVariable.getType())) {
					variable.setType(newVariable.getType());
				}
			}
		}
	}

	protected NodeDefinition createSemanticElement() {
		NodeDefinition element = null;

		try {
			element = SemanticElementUtil.createNodeDefinition(node.getXml());
			element.setId(node.getId());
			element.setResourceType(node.getResourceType().getId());
		} catch (Exception e) {
		}

		return element;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return node + " " + diff.getKind();
	}
}
