/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.view.resourcegroup;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IBaseLabelProvider;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.wizard.IWizard;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.ide.IDE;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IResourceGroupService;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer;
import com.tandbergtv.watchpoint.studio.ui.view.DefaultContentProvider;
import com.tandbergtv.watchpoint.studio.ui.view.ViewAction;
import com.tandbergtv.watchpoint.studio.ui.wizard.ResourceGroupExportWizard;
import com.tandbergtv.watchpoint.studio.ui.wizard.resourcegroup.NewResourceGroupWizard;

/**
 * @author Raj Prakash
 * @author DReddy
 *
 */
public class ResourceGroupExplorer extends AbstractTreeViewExplorer {

	private IResourceGroupService rgService;
	private Action exportAction;
	private IResourceTypeService rtService;
	private ResourceGroupContentProvider contentProvider = new ResourceGroupContentProvider();
	private Map<String,List<ResourceGroup>> resourceTypeGroups = new HashMap<String,List<ResourceGroup>>();
	
	public ResourceGroupExplorer() {
		ServiceFactory serviceFactory = ServiceFactory.createFactory();
		this.rgService = serviceFactory.createResourceGroupService();
		this.rtService = serviceFactory.createResourceTypeService();
	}
	
	/* (non-Javadoc)
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#getInput()
	 */
	@Override
	protected List<? extends IWatchPointDTO> getInput() {
		this.resourceTypeGroups = getResourceGroupsByResourceType();
		List<ResourceType> resourceTypeList = rtService.getResourceTypeList();
		return resourceTypeList;
	}

	/* (non-Javadoc)
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#getContentProvider()
	 */
	@Override
	protected IContentProvider getContentProvider() {
		return this.contentProvider;
	}

	/* (non-Javadoc)
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#getLabelProvider()
	 */
	@Override
	protected IBaseLabelProvider getLabelProvider() {
		return new ResourceGroupLabelProvider();
	}

	/* (non-Javadoc)
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#create()
	 */
	@Override
	protected void create() {
		Object selection = ((TreeSelection) getSelection()).getFirstElement();
		String suggestedFolder = "";
		if (selection instanceof ResourceType) {
			ResourceType rtSelected = (ResourceType) selection;
			suggestedFolder = rtSelected.getProjectName();
		} else if (selection instanceof ResourceGroup) {
			ResourceGroup rgSelected = (ResourceGroup) selection;
			suggestedFolder = rgSelected.getProjectName();
		}
		IWorkbench workbench = super.getSite().getWorkbenchWindow().getWorkbench();
		NewResourceGroupWizard wizard = new NewResourceGroupWizard(suggestedFolder);
		wizard.init(workbench, new StructuredSelection());
		WizardDialog dialog = new WizardDialog(workbench.getActiveWorkbenchWindow().getShell(), wizard);
		dialog.open();
	}
	
	/* (non-Javadoc)
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#view()
	 */
	@Override
	protected void view() {
		// Not used anymore
	}

	/* (non-Javadoc)
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#edit()
	 */
	@Override
	protected void edit() {
		Object element = ((TreeSelection) getSelection()).getFirstElement();
		if (element instanceof ResourceGroup) {
			ResourceGroup rgSelected = (ResourceGroup) element;
			
			IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(rgSelected.getPath()));
			try {
				IDE.openEditor(super.getSite().getWorkbenchWindow().getActivePage(), file);
			} catch (PartInitException e) {
				Utility.handleException("Error opening ResourceGroupEditor.", e);
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#delete()
	 */
	@Override
	protected void delete() {
		// Not used anymore
	}
	
	/*
	 * Fills context menus for the nodes in the tree viewer.
	 * 
	 * @param mgr menu manager of the treeviewer
	 */
	@Override
	protected void fillContextMenu(IMenuManager mgr) {
		mgr.add(createAction);
		mgr.add(editAction);
		mgr.add(exportAction);
	}
	
	/*
	 * Create toolbar.
	 */
	@Override
	protected void createToolbar() {
		IToolBarManager tbManager = getViewSite().getActionBars().getToolBarManager();
		tbManager.add(createAction);
		tbManager.add(editAction);
		tbManager.add(exportAction);
		tbManager.add(refreshAction);
	}

	/*
	 * Creates all the toolbar actions.
	 */
	@Override
	protected void createActions() {
		super.createActions();
		
		/* Create 'Export' Action */
		exportAction = new ViewAction(this, "Export") {
			public void performAction() {
				ResourceGroup rgSelected = (ResourceGroup) ((TreeSelection) getSelection()).getFirstElement();
				if (rgSelected == null) {
					MessageDialog.openInformation(getSite().getShell(), "Select Resource Group", "Please select a Resource Group to export.");
					return;
				}

				openWizard(new ResourceGroupExportWizard(rgSelected));
			}
		};
		exportAction.setImageDescriptor(Utility.getImageDescriptor("export.png"));
	}

	/**
	 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	public void selectionChanged(SelectionChangedEvent event) {
		ISelection selection = event.getSelection();
		Object element =((TreeSelection) selection).getFirstElement();
		if(element instanceof ResourceType){
			ResourceType rgSelected = (ResourceType) element;
			if(rgSelected != null){
				super.deleteAction.setEnabled(false);
				super.createAction.setEnabled(true);
				super.editAction.setEnabled(false);
				super.viewAction.setEnabled(false);
				exportAction.setEnabled(false);
			}
		}else{
			ResourceGroup rgSelected = (ResourceGroup) element;
			if(rgSelected != null){
				super.createAction.setEnabled(true);
				super.editAction.setEnabled(true);
				super.viewAction.setEnabled(true);
				exportAction.setEnabled(true);
				super.deleteAction.setEnabled(true);
			}
		}
	}

	private boolean openWizard(IWizard wizard) {
		IWorkbench workbench = super.getSite().getWorkbenchWindow().getWorkbench();
		if(wizard instanceof INewWizard)
			((INewWizard)wizard).init(workbench, new StructuredSelection());

		WizardDialog dialog = new WizardDialog(workbench.getActiveWorkbenchWindow().getShell(),
				wizard);
		return (dialog.open() == WizardDialog.OK) ? true : false;
	}
	
	// ========================================================================
	// ===================== THE CONTENT PROVIDER
	// ========================================================================

	/*
	 * The Resource Group Content Provider
	 */
	private class ResourceGroupContentProvider extends DefaultContentProvider
	{
		/**
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
		 */
		public Object[] getChildren(Object element)
		{
			Object[] result = null;

			if (element instanceof ResourceType)
			{
				ResourceType resourceType = (ResourceType) element;
				List<ResourceGroup> resourceGroups = resourceTypeGroups.get(resourceType.getSystemId());
				if (resourceGroups != null)
				{
					result = resourceGroups.toArray();
				}
			}
			else
			{
				result = super.getChildren(element);
			}

			return result;
		}

		/**
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
		 */
		public Object getParent(Object element)
		{
			Object parent = null;

			if (element instanceof ResourceGroup)
			{
				ResourceGroup resourceGroup = (ResourceGroup) element;
				IResourceTypeService rtService = ServiceFactory.createFactory().createResourceTypeService();
				parent = rtService.getResourceTypeBySystemId(resourceGroup.getSystemId());
			}
			else
			{
				parent = super.getParent(element);
			}

			return parent;
		}

		/**
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
		 */
		public boolean hasChildren(Object element)
		{
			boolean result = false;

			if (element instanceof ResourceType)
			{
				ResourceType resourceType = (ResourceType) element;
				List<ResourceGroup> resourceGroups =resourceTypeGroups.get(resourceType.getSystemId());
				result = (resourceGroups != null && resourceGroups.size() > 0);
			}
			else
			{
				result = super.hasChildren(element);
			}

			return result;
		}
	}
	
	/*
	 * Gets all the Resource Groups and creates a  list for each of the resource type
	 */
	private Map<String,List<ResourceGroup>> getResourceGroupsByResourceType() {
		List<ResourceGroup> resourceGroups = rgService.getAllResourceGroups();
		Map<String,List<ResourceGroup>> resourceTypeGroups = new HashMap<String,List<ResourceGroup>>();
		for (ResourceGroup resourceGroup : resourceGroups)
		{
			if (resourceGroup.getSystemId()!=null && !resourceGroup.getSystemId().equals(String.valueOf(ResourceType.DEFAULT_ID))) 
			{
				List<ResourceGroup> resourceGroupList = resourceTypeGroups.get(resourceGroup.getSystemId());
				if(resourceGroupList==null)
				{
					resourceGroupList = new ArrayList<ResourceGroup>();
					resourceGroupList.add(resourceGroup);
					resourceTypeGroups.put(resourceGroup.getSystemId(),resourceGroupList);
				}else
				{
					resourceGroupList.add(resourceGroup);
					resourceTypeGroups.put(resourceGroup.getSystemId(),resourceGroupList);
				}

			}

		}
		return resourceTypeGroups;
	}		
}