/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.wizard;

import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.tandbergtv.watchpoint.studio.dto.AdaptorType;
import com.tandbergtv.watchpoint.studio.dto.ConnectionType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.util.UIValidator;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.wizard.resource.fields.IResourceTypeInputField;

/**
 * @author Patrik Araujo
 * 
 */
public class NewResourceTypeProjectWizardPage extends WizardPage {

	private Text nameText, systemIDText, schemaPath;
	private Button selectFile;

	private ModifyListener validationListener;
	private SelectionListener selectFileListener;

	private IResourceTypeService rtService;
	private boolean validationEnabled = false;

	private List<IResourceTypeInputField> extFields = new ArrayList<IResourceTypeInputField>();

	public NewResourceTypeProjectWizardPage() {
		super("Resource Type");
		setTitle("Create Resource Type");
		setDescription("Create a new Resource Type");
		this.validationListener = new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				validate();
			}
		};
		final IConfigurationElement[] configurationElements = Platform.getExtensionRegistry()
				.getConfigurationElementsFor("org.jbpm.gd.jpdl.resourceTypeShemaProvider");

		this.selectFileListener = new SelectionListener() {
			public void widgetDefaultSelected(SelectionEvent arg0) {
			}

			@SuppressWarnings("finally")
			public void widgetSelected(SelectionEvent arg0) {
				// A extension file choose dialog here
				if (configurationElements.length == 1) {
					IConfigurationElement element = configurationElements[0];
					try {
						Object classObject = element.createExecutableExtension("class");

						if (classObject instanceof IResourceTypeSchemaProvider) {
							IResourceTypeSchemaProvider schemaProvider = (IResourceTypeSchemaProvider) classObject;
							String filePath = schemaProvider.getSchemaFilePath(getShell());
							if (filePath != null && !"".equals(filePath)) {
								schemaPath.setText(filePath);
							}
						}
					} catch (Exception e) {
						// do nothing and turn to the default file choose dialog
					} finally {
						return;
					}
				}

				// Default File choose dialog
				FileDialog dlg = new FileDialog(selectFile.getShell());
				dlg.setText("Open");
				String[] filterExt = { "*.xsd" };
				dlg.setFilterExtensions(filterExt);
				String path = dlg.open();
				if (path == null) {
					return;
				}
				schemaPath.setText(path);

			}
		};

		this.rtService = ServiceFactory.createFactory().createResourceTypeService();
	}

	/**
	 * 
	 * @return Name (null is returned if the value is an empty string)
	 */
	public String getResourceTypeName() {
		return convertEmptyToNull(nameText.getText());
	}

	public void setResourceTypeName(String rtName) {
		nameText.setText(rtName);
	}

	/**
	 * Set the default values for fields that are required but were hidden by
	 * removing the extension point declaration.
	 * @param resourceType The resource type object to set the values.
	 */
	private void setDefaultValues(ResourceType resourceType){
		// Set connection type
		if(resourceType.getConnectionType() == null){
			List<ConnectionType> connTypes = this.rtService.getAllConnectionTypes();
			if (connTypes != null && !connTypes.isEmpty()) {
				for (ConnectionType connType : connTypes) {
					if(connType.getName().equals("URL")){
						resourceType.setConnectionType(connType);
					}
				}
			}
		}
	}
	
	public ResourceType getResourceTypeObject(){
		ResourceType rt = new ResourceType();
		rt.setCreateDate(new Date());
		rt.setName(getResourceTypeName());
		rt.setSystemId(getResourceTypeSystemID());
		rt.setVersion(1);
		
		/* Checks if was selected a schema file and sets the adaptor type to default */
		if(this.getFile() != null){
			rt.setAdaptorType( AdaptorType.DEFAULT );
		}
		/* Sets the values to the resource type object */
		for (IResourceTypeInputField extField : extFields) {
			extField.setObjectValues(rt);
		}
		/* Set default values for hidden fields that are required */
		setDefaultValues(rt);
		
		return rt;
	}

	/**
	 * @return System ID (null is returned if the value is an empty string)
	 */
	public String getResourceTypeSystemID() {
		return convertEmptyToNull(systemIDText.getText());
	}

	public String getSelectedFilePath() {
		return schemaPath.getText();
	}

	public File getFile() {
		String filePath = schemaPath.getText();
		if (filePath.length() == 0) {
			return null;
		}
		File file = new File(filePath);
		return file;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.wizard.WizardPage#isPageComplete()
	 */
	public boolean isPageComplete() {
		/*
		 * Checking if the name is entered is to make sure the finish button is
		 * disabled when the wizard starts up
		 */
		return (getErrorMessage() == null && nameText.getText().length() > 0);
	}

	private void loadExtensionControls(){
		IConfigurationElement[] configurationElements = Platform.getExtensionRegistry()
		.getConfigurationElementsFor("org.jbpm.gd.jpdl.resourceTypeFields");
		
		extFields.clear();
		
		/* Loads each extension point field according to the order defined in plugin.xml */
		for (int i = 0; i < configurationElements.length; i++) {
			try {
				// Creates an instance for the class defined by the 'class' parameter from the
				// extension point calling the 0-argument public constructor
				Object classObject = configurationElements[i].createExecutableExtension("class");
				IResourceTypeInputField extField = (IResourceTypeInputField)classObject;
				extFields.add(extField);
			} catch (CoreException e) {
				Utility.reportError("Error loading extension point resource type wizard fields.", e);
			}
		}
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets
	 * .Composite)
	 */
	public void createControl(Composite parent) {
		initializeDialogUnits(parent);
		Composite composite = createClientArea(parent);
		createCustomControls(composite);
		loadExtensionControls();
		createExtensionControls(composite);
		setValuesAndState();
		setExtFieldsValidationListener();
		this.validationEnabled = true;
		setControl(composite);
		Dialog.applyDialogFont(composite);
	}

	/*
	 * Sets the layout of the client area.
	 * 
	 * @param parent the parent composite of the client area
	 * 
	 * @return the composite representing the client area
	 */
	private Composite createClientArea(Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);
		FormLayout layout = new FormLayout();
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		composite.setLayout(layout);
		return composite;
	}
	
	private Composite createExtensionArea(Composite parent){
		Composite extensionArea = new Composite(parent, SWT.NONE);
		FormLayout layout = new FormLayout();
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		extensionArea.setLayout(layout);
		
		FormData extensionAreaFD = new FormData();
		extensionAreaFD.top = new FormAttachment(schemaPath, 10);
		extensionAreaFD.left = new FormAttachment(0, 0);
		extensionAreaFD.right = new FormAttachment(100, 0);
		extensionArea.setLayoutData(extensionAreaFD);
		
		return extensionArea;
	}
	
	/**
	 * Creates the controls defined by extension points. <br>
	 * For each class defined by the extension point is given a composite area, which is the 
	 * area defined in the page to hold the controls created by extension point and a 
	 * default layout data to be set to the control. <br>
	 * Each control can modify the layout data for specific control requirements, 
	 * for example, to define a custom control width .   
	 * @param parent The composite to be used for control creation
	 */
	private void createExtensionControls(Composite parent){
		Composite extensionArea = createExtensionArea(parent);
		
		/* Add the fields loaded by extension points */
		for (int i = 0; i < extFields.size(); i++) {
			IResourceTypeInputField current = extFields.get(i);
			// Layout data for each control
			FormData defaultFD = new FormData();
			defaultFD.right = new FormAttachment(100, 0 );
			defaultFD.left = new FormAttachment(0, 0 );
			if(i == 0){
				current.createControls(extensionArea, defaultFD).setLayoutData(defaultFD);
			}
			else{
				IResourceTypeInputField previous = extFields.get(i-1);
				Composite previousComposite = previous.getComposite();
				defaultFD.top = new FormAttachment(previousComposite, 10);
				extFields.get(i).createControls(extensionArea, defaultFD).setLayoutData(defaultFD);
			}
		}
	}

	private void createCustomControls(Composite parent) {
		// control creation

		// name label
		Label nameLabel = new Label(parent, SWT.NONE);
		nameLabel.setText("Name : ");

		// name text
		nameText = new Text(parent, SWT.BORDER);
		nameText.addModifyListener(validationListener);

		// system id label
		Label systemIDLabel = new Label(parent, SWT.NONE);
		systemIDLabel.setText("System ID : ");

		// system id text
		systemIDText = new Text(parent, SWT.BORDER);
		systemIDText.setTextLimit(2);
		systemIDText.addModifyListener(validationListener);

		// schema label
		Label schemaLabel = new Label(parent, SWT.NONE);
		schemaLabel.setText("Schema : ");

		// schema path text
		schemaPath = new Text(parent, SWT.BORDER);
		schemaPath.setEditable(false);
		schemaPath.addModifyListener(validationListener);

		// select file button
		selectFile = new Button(parent, SWT.NONE);
		selectFile.setText("...");
		selectFile.addSelectionListener(selectFileListener);

		// layout

		// name text
		FormData nameTextFD = new FormData();
		nameTextFD.left = new FormAttachment(30, 2);
		nameTextFD.right = new FormAttachment(100, -10);
		nameText.setLayoutData(nameTextFD);

		// system id label
		FormData systemIDLabelFD = new FormData();
		systemIDLabelFD.top = new FormAttachment(nameText, 10);
		systemIDLabel.setLayoutData(systemIDLabelFD);

		// system id text
		FormData systemIDTextFD = new FormData();
		systemIDTextFD.top = new FormAttachment(nameText, 10);
		systemIDTextFD.left = new FormAttachment(30, 2);
		systemIDTextFD.right = new FormAttachment(100, -10);
		systemIDText.setLayoutData(systemIDTextFD);

		// schema label
		FormData schemaLabelFD = new FormData();
		schemaLabelFD.top = new FormAttachment(systemIDText, 10);
		schemaLabel.setLayoutData(schemaLabelFD);

		// schema path
		FormData schemaPathFD = new FormData();
		schemaPathFD.top = new FormAttachment(systemIDText, 10);
		schemaPathFD.left = new FormAttachment(30, 2);
		schemaPathFD.right = new FormAttachment(selectFile, -5);
		schemaPath.setLayoutData(schemaPathFD);

		// select file button
		FormData selectFileFD = new FormData();
		selectFileFD.top = new FormAttachment(systemIDText, 8);
		selectFileFD.right = new FormAttachment(100, -10);
		selectFile.setLayoutData(selectFileFD);
		
	}
	
	/**
	 * Sets a modify listener for the control, then the page can be validated 
	 * at the same time that the user is filling the fields.
	 */
	private void setExtFieldsValidationListener(){
		for (IResourceTypeInputField extField : extFields) {
			extField.setValidationListener(validationListener);
		}
	}
	
	/**
	 * Initialize the control values.<br>
	 * Each class is responsible for initialize the control values. 
	 */
	private void setValuesAndState() {
		for (IResourceTypeInputField extField : extFields) {
			extField.initValues();
		}
	}

	@Override
	public void setVisible(boolean visible) {
		String projName = ((NewResourceTypeProjectWizard) getWizard()).getProjectName();
		nameText.setText(projName);
		nameText.setEnabled(false);
		super.setVisible(visible);
	}

	private void validate() {
		if (!this.validationEnabled)
			return;

		String errorMessage = null;
		
		errorMessage = ( !schemaPath.getText().equals("") && !schemaPath.getText().endsWith(".xsd") ) ? 
				"Schema must be a .xsd file" : null;
		if (errorMessage != null) {
			updateErrorMessage(errorMessage);
			return;
		}
		
		errorMessage = UIValidator.validateValue("Name", nameText.getText(), UIValidator.Validation.NOT_BLANK,
				UIValidator.Validation.TRIMMED, UIValidator.Validation.MAX_STRING_LENGTH);
		if (errorMessage != null) {
			updateErrorMessage(errorMessage);
			return;
		}

		errorMessage = UIValidator.validateValue("System ID", systemIDText.getText(), UIValidator.Validation.NOT_BLANK,
				UIValidator.Validation.TRIMMED, UIValidator.Validation.ALPHA_NUMERIC);
		if (errorMessage != null) {
			updateErrorMessage(errorMessage);
			return;
		}
		
		// Calls the validate method for each control defined by extension points and gets
		// the error message. 
		for (IResourceTypeInputField extField : extFields) {
			errorMessage = extField.validate();
			if (errorMessage != null) {
				updateErrorMessage(errorMessage);
				return;
			}
		}

		updateErrorMessage(null);
	}

	private void updateErrorMessage(String errorMessage) {
		setErrorMessage(errorMessage);
		getContainer().updateButtons();
	}

	private String convertEmptyToNull(String str) {
		return ((str.length() == 0) ? null : str);
	}

}
