/**
 * ImportWizardFirstPage.java
 * Created Feb 9, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.wizard.imports;

import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Label;
import org.jbpm.gd.jpdl.Logger;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.external.imports.ImportDataProvider;
import com.tandbergtv.watchpoint.studio.ui.wizard.AbstractWizardPage;

/**
 * Prompts for directory from which items will be imported
 * 
 * @author Sahil Verma
 */
public class ImportWizardFirstPage extends AbstractWizardPage {
	
	private static final int MAXENTRIES = 5;
	
	private Combo combo;
	
	private Button browseButton;
	
	private Label label;
	
	private Map<Class<? extends IWatchPointDTO>, ImportDataProvider> map;
	
	private Class<? extends IWatchPointDTO> clazz;
	
	private Collection<? extends IWatchPointDTO> models;
	
	private static final String DIRECTORY = "directory"; //$NON-NLS-1$
	
	/**
	 * Creates a page
	 * 
	 * @param pageName
	 */
	protected ImportWizardFirstPage(String pageName) {
		super(pageName);
		setTitle("Import location");
		setMessage("File system location of items to be imported");
		this.map = new HashMap<Class<? extends IWatchPointDTO>, ImportDataProvider>();
	}

	/**
	 * @param pageName
	 * @param clazz
	 */
	public ImportWizardFirstPage(String pageName, Class<? extends IWatchPointDTO> clazz) {
		this(pageName);
		this.clazz = clazz;
	}

	public void addProvider(Class<? extends IWatchPointDTO> clazz, ImportDataProvider provider) {
		this.map.put(clazz, provider);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isPageComplete() {
		return getBaseDirectory() != null && getBaseDirectory().trim().length() > 0;
	}
	
	public Collection<? extends IWatchPointDTO> loadModel(String location) {
		fetchItems(location);

		return models;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void createControl(Composite parent) {
		initializeDialogUnits(parent);
		Composite container = createClientArea(parent);
		
		Dialog.applyDialogFont(container);
		initialize();
		setControl(container);
	}
	
	/**
	 * @param parent
	 * @return
	 */
	protected Composite createClientArea(Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);
		FormLayout layout = new FormLayout();
		
		layout.marginWidth = 10;
		layout.marginHeight = 50;
		
		composite.setLayout(layout);
		
		FormData formData = new FormData();
		
		this.label = new Label(composite, SWT.NONE);
		this.label.setText("Directory:");
		formData.left = new FormAttachment(0);
		this.label.setLayoutData(formData);
		
		this.browseButton = new Button(composite, SWT.PUSH);
		this.browseButton.setText("Browse");
		this.browseButton.setEnabled(true);
		
		formData = new FormData();
		formData.top = new FormAttachment(0);
		formData.right = new FormAttachment(100, -10);
		
		this.browseButton.setLayoutData(formData);

		this.combo = new Combo(composite, SWT.DROP_DOWN);
		
		formData = new FormData();
		formData.top = new FormAttachment(0);
		formData.left = new FormAttachment(this.label, 5);
		formData.right = new FormAttachment(this.browseButton, -10);
		
		this.combo.setLayoutData(formData);
		
		return composite;
	}

	protected void initialize() {
		IDialogSettings settings = getDialogSettings();
		
		initializeCombo(settings, DIRECTORY, combo);
		hookListeners();
	}
	
	/**
	 * Attach listeners to widgets on the page
	 */
	protected void hookListeners() {
		this.browseButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				IPath path = chooseDirectory();
				
				if (path != null)
					combo.setText(path.toOSString());
			}
		});
		
		this.combo.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				validate();
			}
		});
	}
	
	/**
	 * @return a chosen path from the directory dialog invoked from browse button
	 */
	protected IPath chooseDirectory() {
		DirectoryDialog dialog = new DirectoryDialog(getShell());
		
		dialog.setFilterPath(combo.getText());
		dialog.setText("Choose import directory");
		dialog.setMessage("Select a directory containing items to import");
		
		String res = dialog.open();
		
		if (res != null)
			return new Path(res);
		
		return null;
	}
	
	protected void fetchItems(final String directory) {
		IRunnableWithProgress op = new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) {
				monitor.beginTask("Calculating items to import...", IProgressMonitor.UNKNOWN);
				ImportDataProvider provider = map.get(clazz);
				
				provider.setImportLocation(directory);
				models = provider.getItems();
				monitor.done();
			}
		};
		
		try {
			getContainer().run(true, false, op);
		} catch (InvocationTargetException e) {
			Logger.logError(e.getCause());
			/* FIXME Message dialog */
		} catch (InterruptedException e) {
		}
	}
	
	/**
	 * Returns the directory from which we're going to import plugins
	 * 
	 * @return
	 */
	public  String getBaseDirectory() {
		return this.combo.getText();
	}
	
	/**
	 * Validates the drop location
	 */
	protected void validate() {
		if (browseButton.getSelection()) {
			IPath curr = new Path(combo.getText());
			
			if (curr.segmentCount() == 0 && curr.getDevice() == null) {
				setErrorMessage("Missing directory");
				setPageComplete(false);
				return;
			}
			
			if (!Path.ROOT.isValidPath(combo.getText())) {
				setErrorMessage("Invalid folder");
				setPageComplete(false);
				return;
			}

			if (!curr.toFile().isDirectory()) {
				setErrorMessage("Not a directory");
				setPageComplete(false);
				return;
			}
		}
		
		setErrorMessage(null);
		setPageComplete(true);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void saveSettings(IDialogSettings settings) {
		saveCombo(settings, DIRECTORY, combo);
	}

	/**
	 * Initializes the combo box with all saved entries and highlights the first available one
	 * 
	 * @param settings
	 * @param key
	 * @param combo
	 */
	protected void initializeCombo(IDialogSettings settings, String key, Combo combo) {
		for (int i = 0; i < MAXENTRIES + 1; i++) {
			String text = settings.get(key + String.valueOf(i));
			
			if (text != null && combo.indexOf(text) == -1)
				combo.add(text);
		}
		
		if (combo.getItemCount() > 0)
			combo.setText(combo.getItem(0));
	}

	/**
	 * Saves most recent entries to the settings plus current
	 * 
	 * @param settings
	 * @param key
	 * @param combo
	 */
	protected void saveCombo(IDialogSettings settings, String key, Combo combo) {
		if (combo.getText().trim().length() > 0) {
			settings.put(key + String.valueOf(0), combo.getText().trim());
			String [] items = combo.getItems();
			int count = Math.min(items.length, MAXENTRIES);
			
			for (int i = 0; i < count; i++)
				settings.put(key + String.valueOf(i + 1), items[i].trim());
		}
	}
}
