/*
 * Created on Sep 10, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.util;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.channels.FileChannel;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

/**
 * Utility class that provides methods to deal with files.
 * 
 * @author Vijay Silva
 */
public final class FileUtil
{
	private static final Logger logger = Logger.getLogger(FileUtil.class);

	private static final long COPY_BUF_MAX_SIZE = (64 * 1024 * 1024) - (32 * 1024);

	/*
	 * Cannot instantiate
	 */
	private FileUtil()
	{
	}

	/**
	 * Export the workspace folder contents to the Filesystem. 
	 * 	
	 * @param destinationFolder The destination folder
	 * @param sourceMember The member where the files will be get from.
	 * @param recursive flag indicating if wants to export the sub-folders
	 * @throws IOException
	 * @throws CoreException
	 */
	public static void exportWorkspaceFolderToFilesystem(IResource sourceMember, File destinationFolder, boolean recursive) throws IOException, CoreException {
		if (sourceMember instanceof IFile) {
			IFile file = (IFile) sourceMember;
			byte[] contents = FileUtil.readFile(file.getContents());
			File destination = new File(destinationFolder, sourceMember.getName());
			writeToFile(contents, destination);
		} else if (sourceMember instanceof IFolder) {
			IFolder sourceFolder = (IFolder) sourceMember;
			for (IResource childMember : sourceFolder.members()) {
				File newDestinationFolder = destinationFolder;
				if (childMember instanceof IFolder) {
					newDestinationFolder = new File(destinationFolder, childMember.getName());
					newDestinationFolder.mkdirs();
				}

				if (recursive) {
					exportWorkspaceFolderToFilesystem(childMember, newDestinationFolder, recursive);
				}
			}
		}
	}
	
	/**
	 * Import the filesystem folder contents to the Workspace. 
	 * 	
	 * @param sourceFolder The source folder in the Filesystem
	 * @param destinationFolder The destination folder on the Workspace.
	 * @param recursive flag indicating if wants to import the sub-folders
	 * @throws IOException
	 * @throws CoreException
	 */
	public static void importFilesystemFolderToWorkspace(File sourceFolder, IContainer destinationFolder, boolean recursive) throws IOException, CoreException {
		if (!destinationFolder.exists()) {
			if (destinationFolder instanceof IFolder) {
				IFolder iFolder = (IFolder) destinationFolder;
				iFolder.create(true, true, null);
			}
		}
		if (sourceFolder.isFile()) {
			byte[] contents = FileUtil.readFile(new FileInputStream(sourceFolder));
			IFile destinationFile = destinationFolder.getFile(new Path(sourceFolder.getName()));
			ByteArrayInputStream inputStream = new ByteArrayInputStream(contents);
			if (destinationFile.exists()) {
				destinationFile.setContents(inputStream, true, false, null);
			} else {
				destinationFile.create(inputStream, true, null);
			}
		} else if (sourceFolder.isDirectory()) {
			for (File childFile : sourceFolder.listFiles()) {
				IContainer childFolder = destinationFolder;

				if (childFile.isDirectory()) {
					childFolder = destinationFolder.getFolder(new Path(childFile.getName()));
				}
				if (recursive) {
					importFilesystemFolderToWorkspace(childFile, childFolder, recursive);
				}
			}
		}
	}
	
	/**
	 * Writes the byte array to the file specified by the filePath.
	 * 
	 * @param data The data to be written
	 * @param file The file in which the data is to be written
	 * @throws IOException 
	 */
	public static void writeToFile(byte[] data, File file) throws IOException
	{
		FileOutputStream fos = new FileOutputStream(file);
		fos.write(data);
		fos.flush();
		fos.close();
	}
	
	/**
	 * Deletes the contents of a Folder (without deleting the folder itself). Deletes all
	 * sub-folders and their contents if the deleteSubFolders is true.
	 * 
	 * @param folder
	 *            The folder who's contents must be deleted
	 * @param deleteSubFolders
	 *            flag to indicate if sub-folders must be deleted, true for deletion, false
	 *            otherwise
	 * @return true if the delete operation succeeds, false otherwise
	 */
	public static boolean deleteFolderContents(File folder, boolean deleteSubFolders)
	{
		if (!folder.isDirectory())
			return false;

		File[] contents = folder.listFiles();
		boolean result = true;
		if (contents != null)
		{
			for (int i = 0; i < contents.length; i++)
			{
				if (contents[i].isDirectory() && deleteSubFolders)
				{
					result &= deleteFolderContents(contents[i], deleteSubFolders);
					result &= contents[i].delete();
				}
				else
				{
					result &= contents[i].delete();
				}
			}
		}

		return result;
	}

	/**
	 * Copy the file specified by the source location to the file specified to the target location.
	 * If the source location is a folder, the target location will be created as a folder. The
	 * target location must not exist before the copy operation.
	 * 
	 * @param source
	 *            The Source file
	 * @param target
	 *            The Target file
	 * @param copySubFolders
	 *            flag indicating that sub-folders must be copied (if the source is a folder)
	 * @return true if the operation succeeds, false otherwise
	 */
	public static boolean copyFileOrFolder(File source, File target, boolean copySubFolders, boolean overwrite)
	{
		boolean result = true;

		if (source.isDirectory())
		{
			if (target.exists() && !target.isDirectory() && !overwrite)
			{
				String msg = "The target location is an existing file, failed to copy folder: "
						+ source.getAbsolutePath() + " to: " + target.getAbsolutePath();
				logger.warn(msg);
				return false;
			}
			else 
				if (!target.exists() && !target.mkdirs())
			{
				String msg = "Failed to create folder: " + target.getAbsolutePath()
						+ " when attempting to copy folder: " + source.getAbsolutePath();
				logger.warn(msg);
				return false;
			}

			File[] contents = source.listFiles();
			if (contents != null)
			{
				for (File content : contents)
				{
					if (content.isDirectory() && !copySubFolders)
						continue;

					File targetContent = new File(target, content.getName());
					if (!copyFileOrFolder(content, targetContent, copySubFolders, overwrite))
					{
						result = false;
						break;
					}
				}
			}
		}
		else
		{
			if (target.exists() && !overwrite)
			{
				String msg = "The target location already exists, failed to copy file: "
						+ source.getAbsolutePath() + " to: " + target.getAbsolutePath();
				logger.warn(msg);
				return false;
			}

			try
			{
				copyFile(source, target);
			}
			catch (IOException ex)
			{
				String msg = "Failed when copying file from: " + source.getAbsolutePath() + " to: "
						+ target.getAbsolutePath();
				logger.warn(msg, ex);
				result = false;
			}
		}

		return result;
	}
	
	public static boolean copyFileOrFolderCreatingTargetDirs(File source, File target, boolean copySubFolders, boolean overwrite)
	{
		boolean result = true;

		if (source.isDirectory())
		{
			if (target.exists() && !target.isDirectory() && !overwrite)
			{
				String msg = "The target location is an existing file, failed to copy folder: "
						+ source.getAbsolutePath() + " to: " + target.getAbsolutePath();
				logger.warn(msg);
				return false;
			}
			else 
				if (!target.exists() && !target.mkdirs())
			{
				String msg = "Failed to create folder: " + target.getAbsolutePath()
						+ " when attempting to copy folder: " + source.getAbsolutePath();
				logger.warn(msg);
				return false;
			}

			File[] contents = source.listFiles();
			if (contents != null)
			{
				for (File content : contents)
				{
					if (content.isDirectory() && !copySubFolders)
						continue;

					File targetContent = new File(target, content.getName());
					if (!copyFileOrFolder(content, targetContent, copySubFolders, overwrite))
					{
						result = false;
						break;
					}
				}
			}
		}
		else
		{
			if (target.exists() && !overwrite)
			{
				String msg = "The target location already exists, failed to copy file: "
						+ source.getAbsolutePath() + " to: " + target.getAbsolutePath();
				logger.warn(msg);
				return false;
			}
			try
			{
				if(!target.isDirectory() && !target.getParentFile().exists()){
					target.getParentFile().mkdirs();
				}
				copyFile(source, target);
			}
			catch (IOException ex)
			{
				String msg = "Failed when copying file from: " + source.getAbsolutePath() + " to: "
						+ target.getAbsolutePath();
				logger.warn(msg, ex);
				result = false;
			}
		}

		return result;
	}
	
	/**
	 * Copy the file specified by the source location to the file specified to the target location.
	 * If the source location is a folder, the target location will be created as a folder. The
	 * target location will be overwritten (merged).
	 * 
	 * @param source
	 * @param target
	 * @param copySubFolders
	 * @return true if the operation succeeds, false otherwise
	 */
	public static boolean overwrite(File source, File target, boolean copySubFolders)
	{
		boolean result = true;

		if (source.isDirectory())
		{
			if (target.exists() && !target.isDirectory())
			{
				String msg = "The target location is an existing file, failed to copy folder: "
						+ source.getAbsolutePath() + " to: " + target.getAbsolutePath();
				logger.warn(msg);
				return false;
			}
			else if (!target.exists() && !target.mkdirs())
			{
				String msg = "Failed to create folder: " + target.getAbsolutePath()
						+ " when attempting to copy folder: " + source.getAbsolutePath();
				logger.warn(msg);
				return false;
			}

			File[] contents = source.listFiles();
			if (contents != null)
			{
				for (File content : contents)
				{
					if (content.isDirectory() && !copySubFolders)
						continue;

					File targetContent = new File(target, content.getName());
					if (!overwrite(content, targetContent, copySubFolders))
					{
						result = false;
						break;
					}
				}
			}
		}
		else
		{
			try
			{
				copyFile(source, target);
			}
			catch (IOException ex)
			{
				String msg = "Failed when copying file from: " + source.getAbsolutePath() + " to: "
						+ target.getAbsolutePath();
				logger.warn(msg, ex);
				result = false;
			}
		}

		return result;
	}
	
	/**
	 * Writes out the XML to the specified path
	 * 
	 * @param xml
	 * @param path
	 * @throws TransformerException
	 * @throws IOException
	 */
	public static void writeXml(String xml, String path) throws TransformerException, IOException
	{
		File targetFile = new File(path);
		Writer writer = null;
		TransformerFactory factory = TransformerFactory.newInstance();

		try
		{
			Transformer transformer = factory.newTransformer();

			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");
			transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");

			writer = new OutputStreamWriter(new FileOutputStream(targetFile), "UTF-8");

			transformer.transform(new StreamSource(new StringReader(xml)), new StreamResult(writer));
		}
		finally
		{
			if (writer != null)
				try
				{
					writer.close();
				}
				catch (IOException e) {}
		}
	}

	/**
	 * Parses a DOM from an XML file
	 * 
	 * @param path
	 * @return
	 * @throws TransformerException
	 */
	public static Document readDocument(String path) throws TransformerException
	{
		return readDocument(new StreamSource(new File(path)));
	}
	
	/**
	 * Parses a DOM from an Input Stream 
	 * 
	 * @param path
	 * @return
	 * @throws TransformerException
	 */
	public static Document readDocument(InputStream in) throws TransformerException
	{
		return readDocument(new StreamSource(in));
	}
	
	/**
	 * Parses a DOM from an Input Stream 
	 * 
	 * @param path
	 * @return
	 * @throws TransformerException
	 */
	private static Document readDocument(StreamSource in) throws TransformerException
	{
		Document document = null;
		
		try {
			document = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();
			Transformer transformer = TransformerFactory.newInstance().newTransformer();
			
			transformer.transform(in, new DOMResult(document));
		} catch (ParserConfigurationException e) {
			throw new Error("Oops");
		}
		
		return document;
	}
	
	public static String convertDocument(Node node) throws TransformerException
	{
		StringWriter w = new StringWriter();
		Transformer transformer = TransformerFactory.newInstance().newTransformer();
		
		Document document;
		
		if (node instanceof Document)
			document = Document.class.cast(node);
		else
			document = node.getOwnerDocument();
		
		try {
			if (document.getXmlEncoding() != null) {
				transformer.setOutputProperty(OutputKeys.ENCODING, document.getXmlEncoding());
			}
		} catch (Exception e) {
			// Ignore error, method getXmlEncoding() is not supported by some Document implementations.
		}
		
		transformer.transform(new DOMSource(node), new StreamResult(w));
		
		return w.toString();
	}
	
	/**
	 * Reads bytes from the specified file
	 * 
	 * @param filePath
	 * @return
	 * @throws IOException
	 */
	public static byte[] readFile(String filePath) throws IOException
	{
		File file = new File(filePath);
		int fileLengthInt = (int) file.length();
		int READ_SIZE = 4096;
		
		int arraySize = fileLengthInt;
		byte[] fileContents = new byte[arraySize];
		BufferedInputStream bis = null;
		
		try
		{
			bis = new BufferedInputStream(new FileInputStream(file));
			int offset = 0;
			while (true)
			{
				int remainingArraySize = arraySize - offset;
				int numBytesToRead = (READ_SIZE > remainingArraySize) ? remainingArraySize : READ_SIZE;
				int bytesRead = bis.read(fileContents, offset, numBytesToRead);
				if(bytesRead == -1)
					break;
				offset += bytesRead;
				if(offset == arraySize)
					break;
			}
			
			return fileContents;
		}
		finally
		{
			if(bis != null)
			{
				try
				{
					bis.close();
				}
				catch(IOException ioe)
				{
				}
			}
		}
	}
	
	/**
	 * Reads bytes from the specified file
	 * 
	 * @param filePath
	 * @return
	 * @throws IOException
	 */
	public static byte[] readFile(InputStream in) throws IOException
	{
		BufferedInputStream bis = null;
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		try {
			bis = new BufferedInputStream(in);
			int i = 0;
			while ( (i = bis.read()) != -1 ) {
				out.write((byte) i);
			}
			
			return out.toByteArray();
		}
		finally
		{
			if(bis != null)
			{
				try
				{
					bis.close();
				}
				catch(IOException ioe)
				{
				}
			}
		}
	}

	/**
	 * Write the DOM out to the specified file path.
	 * 
	 * @param document
	 * @param targetFilePath
	 * @throws TransformerException
	 * @throws IOException
	 */
	public static void writeDocument(Document document, String targetFilePath)
	throws TransformerException, IOException
	{
		File targetFile = new File(targetFilePath);
		writeDocument(document, new FileOutputStream(targetFile));
	}
	
	/**
	 * Write the DOM out to the specified file path.
	 * 
	 * @param document
	 * @param targetFilePath
	 * @throws TransformerException
	 * @throws IOException
	 */
	public static void writeDocument(Document document, OutputStream out)
	throws TransformerException, IOException
	{
		Writer writer = null;
		
		try
		{
			TransformerFactory factory = TransformerFactory.newInstance();
			Transformer transformer = factory.newTransformer();
			
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");
			transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
			transformer.setOutputProperty("{http://xml.apache.org/xalan}line-separator","\n");
			
			writer = new OutputStreamWriter(out, "UTF-8");
			transformer.transform(new DOMSource(document), new StreamResult(writer));
		}
		finally
		{
			if (writer != null)
			{
				try
				{
					writer.close();
				}
				catch (IOException ex)
				{
					String msg = "Failed to close stream";
					logger.warn(msg, ex);
				}
			}
		}
	}
	
	/*
	 * Copy a file from the source File to the target File.
	 */
	public static void copyFile(File source, File target) throws IOException
	{
		FileInputStream inStream = null;
		FileOutputStream outStream = null;

		try
		{
			inStream = new FileInputStream(source);
			outStream = new FileOutputStream(target);
			FileChannel inputChannel = inStream.getChannel();
			FileChannel outputChannel = outStream.getChannel();

			long inputChannelSize = inputChannel.size();
			long marker = 0;
			while (marker < inputChannelSize)
			{
				marker += inputChannel.transferTo(marker, COPY_BUF_MAX_SIZE, outputChannel);
			}
		}
		finally
		{
			closeStream(inStream, source.getAbsolutePath());
			closeStream(outStream, target.getAbsolutePath());
		}
	}

	/*
	 * Close the File Input / Output stream
	 */
	private static void closeStream(Closeable stream, String filePath)
	{
		try
		{
			if (stream != null)
				stream.close();
		}
		catch (IOException ex)
		{
			logger.warn("Failed to close the file stream for file: " + filePath
					+ ", ignoring error.", ex);
		}
	}
}
