package com.tandbergtv.watchpoint.studio.validation.impl;

import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.TEMPLATE_START_NODE_VARIABLE_UNUSED;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.TEMPLATE_SUBPROCESS_EXPRESSION_NOT_ALLOWED;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.TEMPLATE_SUBPROCESS_INVALID_EXPRESSION;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.TEMPLATE_SUBPROCESS_UNMAPPED_VARIABLE;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.TEMPLATE_SUBPROCESS_VARIABLE_MUST_BE_MAPPED_AS_READ;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.TEMPLATE_SUBPROCESS_VARIABLE_NOT_DECLARED_CHILD;
import static com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode.TEMPLATE_SUBPROCESS_VARIABLE_NOT_DECLARED_PARENT;
import static java.util.Arrays.asList;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.compiler.IProblem;
import org.jbpm.gd.common.model.GenericElement;
import org.jbpm.gd.common.model.SemanticElement;
import org.jbpm.gd.jpdl.model.Action;
import org.jbpm.gd.jpdl.model.Decision;
import org.jbpm.gd.jpdl.model.Fork;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;
import org.jbpm.gd.jpdl.model.ProcessState;
import org.jbpm.gd.jpdl.model.StartState;
import org.jbpm.gd.jpdl.model.SubProcess;
import org.jbpm.gd.jpdl.model.Transition;
import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.service.INodeDefinitionService;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.IWorkflowTemplateService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.actionconfig.ActionClassConfiguration;
import com.tandbergtv.watchpoint.studio.ui.actionconfig.ActionConfigurationManager;
import com.tandbergtv.watchpoint.studio.ui.actionconfig.ActionVariableConfiguration;
import com.tandbergtv.watchpoint.studio.ui.model.AssignNode;
import com.tandbergtv.watchpoint.studio.ui.model.AutomaticTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.DataCollectionManager;
import com.tandbergtv.watchpoint.studio.ui.model.IDueDateElement;
import com.tandbergtv.watchpoint.studio.ui.model.LoopNode;
import com.tandbergtv.watchpoint.studio.ui.model.MailNode;
import com.tandbergtv.watchpoint.studio.ui.model.ManualTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.NodeGroup;
import com.tandbergtv.watchpoint.studio.ui.model.SemanticElementVisitor;
import com.tandbergtv.watchpoint.studio.ui.model.WPAbstractNode;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.model.tree.ITree;
import com.tandbergtv.watchpoint.studio.ui.properties.DataType;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiff;
import com.tandbergtv.watchpoint.studio.ui.sync.compare.NodeDefinitionComparer;
import com.tandbergtv.watchpoint.studio.ui.sync.compare.RollbackNodeDefinitionComparer;
import com.tandbergtv.watchpoint.studio.ui.sync.compare.SuperStateComparer;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.util.FileUtil;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;
import com.tandbergtv.watchpoint.studio.validation.IValidationRule;
import com.tandbergtv.watchpoint.studio.validation.IValidator;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRuleInstantiationException;
import com.tandbergtv.watchpoint.studio.validation.ValidatorFactory;
import com.tandbergtv.watchpoint.studio.validation.model.ActionVariable;
import com.tandbergtv.watchpoint.studio.validation.model.ContainerNode;
import com.tandbergtv.watchpoint.studio.validation.model.NodeDefinitionInstance;
import com.tandbergtv.watchpoint.studio.validation.model.SuperStateNodeDefinitionInstance;
import com.tandbergtv.watchpoint.studio.validation.model.TemplateLoopNodeInstance;
import com.tandbergtv.watchpoint.studio.validation.model.TemplateNodeDefinitionInstance;
import com.tandbergtv.watchpoint.studio.validation.rules.actionvariable.ActionVariableValueRule;

public class SemanticElementValidatorVisitor extends SemanticElementVisitor {

    private final List<ValidationMessage> messages = new ArrayList<ValidationMessage>();
    private final Logger logger = Logger.getLogger(SemanticElementValidatorVisitor.class);
    private final ValidationService validationService;

    private static ValidatorFactory validatorFactory = ValidatorFactory.createFactory();

    public SemanticElementValidatorVisitor(ValidationService validationService) {
        this.validationService = validationService;
    }

    public List<ValidationMessage> getMessages() {
        return messages;
    }

    @Override
    public void visit(SemanticElement semanticElement) {
        if (!(semanticElement instanceof WorkflowTemplate)) {
            messages.addAll(validatorFactory.createValidator(NodeElement.class).validate((NodeElement) semanticElement));
        }

        super.visit(semanticElement);
    }

    /*
     * Method to validate a Decision Node, as well as the expressions in the Decision Node and its
     * outgoing Transitions. May be used for decision nodes in Templates.
     * The input variables are the list of variables in that may be used in the
     * Expressions of the Transition Condition or the Decision.
     */
    public void visit(Decision decision) {
    	if (Utility.getNodeDefinition(decision) != null) {
            return;
        }
    	
        /* validate the node itself */
        messages.addAll(validatorFactory.createValidator(Decision.class).validate(decision));

        /* validate the decision expression */
        WorkflowTemplate template = Utility.getTemplate(decision);
        if (template != null) {
            messages.addAll(validationService.validateDecisionExpression(decision, template));

        } else {
            NodeDefinition nodeDefinition = Utility.getNodeDefinition(decision);
            if (nodeDefinition != null) {
                messages.addAll(validationService.validateDecisionExpression(decision, nodeDefinition));
            }
        }

        /* validate the condition expressions */
        Transition[] transitions = decision.getTransitions();
        if (transitions != null) {
            for (Transition transition : transitions) {
                validateCondition(decision, transition);
            }
        }
    }

    public void visit(AutomaticTaskNode node) {
        messages.addAll(validatorFactory.createValidator(AutomaticTaskNode.class).validate(node));

        /* Validate each of the Automatic Task Node Action Variables */
        Action action = node.getAction();
        GenericElement[] elements = (action != null) ? action.getGenericElements() : null;
        if (elements != null) {
            for (GenericElement element : elements) {
                messages.addAll(validateActionVariableValue(node, element));
            }
        }
    }

    private List<ValidationMessage> validateActionVariableValue(AutomaticTaskNode node, GenericElement element) {
        String className = (node != null && node.getAction() != null) ? node.getAction().getClassName() : null;
        String variableName = element.getName();

        ActionConfigurationManager manager = ActionConfigurationManager.getInstance();
        ActionClassConfiguration actionClass = manager.getActionClass(className);
        ActionVariableConfiguration variableConfiguration = null;
        if (actionClass != null) {
            List<ActionVariableConfiguration> variables = actionClass.getVariables();
            if (variables != null) {
                for (ActionVariableConfiguration currentConfiguration : variables) {
                    if (currentConfiguration.getName().equals(variableName)) {
                        variableConfiguration = currentConfiguration;
                        break;
                    }
                }
            }
        }

        /* Find the Rule to execute for the value */
        ActionVariableValueRule rule = createActionVariableRule(variableConfiguration);
        ActionVariable actionVariable = new ActionVariable(node, element, variableConfiguration);
        return rule.validateRule(actionVariable);
    }

    private ActionVariableValueRule createActionVariableRule(ActionVariableConfiguration variable) {
        ActionVariableValueRule valueRule = null;

        String ruleClassName = (variable != null) ? variable.getValidationRuleClassName() : null;
        if (ruleClassName != null && ruleClassName.trim().length() > 0) {
            try {
                IValidationRule<?> rule = validatorFactory.createValidationRule(ruleClassName);
                if (rule instanceof ActionVariableValueRule) {
                    valueRule = (ActionVariableValueRule) rule;
                } else {
                    String msg = "The Rule: " + ruleClassName + " does not extend class: "
                            + ActionVariableValueRule.class.getName()
                            + ", cannot use specified rule for Action Variable validation"
                            + ", using default implementation.";
                    logger.warn(msg);
                }
            } catch (ValidationRuleInstantiationException ex) {
                String msg = "Failed to validate ActionVariableValueRule: " + ruleClassName
                        + ", using default implementation.";
                logger.warn(msg, ex);
            }
        }

        if (valueRule == null) {
            valueRule = new ActionVariableValueRule();
        }

        return valueRule;
    }

    private ContainerNode createContainerNode(NodeElementContainer container, NodeElement node) {
        List<WPVariable> variables = null;
        if (container instanceof WorkflowTemplate) {
            variables = validationService.getContainerVariables((WorkflowTemplate) container);

        } else if (container instanceof NodeDefinition) {
            variables = (List<WPVariable>) validationService.getContainerVariables((NodeDefinition) container);
        }

        return new ContainerNode(container, variables, node);
    }

    public void visit(MailNode mailNode) {
        messages.addAll(validatorFactory.createValidator(MailNode.class).validate(mailNode));
    }

    public void visit(LoopNode loopNode) {
        messages.addAll(validatorFactory.createValidator(LoopNode.class).validate(loopNode));

        NodeElementContainer parent = Utility.getNodeDefinition(loopNode);
        if (parent == null) {
            parent = Utility.getTemplate(loopNode);
        }
        TemplateLoopNodeInstance nodeWrapper = new TemplateLoopNodeInstance(loopNode, parent);
        messages.addAll(validatorFactory.createValidator(TemplateLoopNodeInstance.class).validate(nodeWrapper));

        validateLoopNodeExpression(loopNode, loopNode.getParent());
    }

    public void visit(StartState startState) {
        messages.addAll(validatorFactory.createValidator(StartState.class).validate(startState));

        validateUnusedVariables((WorkflowTemplate) startState.getParent(), messages);
    }

    private void validateUnusedVariables(WorkflowTemplate template, List<ValidationMessage> messages) {
        if (template.getStartState() == null || template.getStartState().getTask() == null) {
            return;
        }

        CheckVariableUsageVisitor visitor = new CheckVariableUsageVisitor(template.getStartState().getTask().getController().getVariables());
        template.accept(visitor);

        for (Variable variable : visitor.getUnusedVariables()) {
            List<String> parameters = new ArrayList<String>();
            parameters.add(variable.getName());
            ValidationMessageAdder.getInstance().addValidationMessage(messages, variable, TEMPLATE_START_NODE_VARIABLE_UNUSED, parameters);
        }
    }

    public void visit(ManualTaskNode node) {
        /* Validate the node */
        messages.addAll(validatorFactory.createValidator(ManualTaskNode.class).validate(node));

        /* Validate the due date expression */
        validateDueDate(node);

        /* Validate the mapping between the node variables and the allowed list of variables */
        ContainerNode containerNode = createContainerNode(node.getParent(), node);
        messages.addAll(validatorFactory.createValidator(containerNode).validate(containerNode));
    }

    public void visit(NodeDefinition node) {
        if (NodeDefinitionType.MessageNode == node.getNodeType()) {
            String url = DataCollectionManager.getInstance().getReader().getMessageNodeUrlByUid(node.getUid());
            if (StringUtils.isNotBlank(url) && !Utility.isFileExist(url)) {
                ValidationMessageAdder.getInstance().addValidationMessage(messages, node,
                        ValidationMessageCode.NODE_DEFINITION_MESSAGE_NOT_FOUND, Arrays.asList(node.getName()));
                return;
            }
        } else if (NodeDefinitionType.SuperState == node.getNodeType()) {
            String url = DataCollectionManager.getInstance().getReader()
                    .getSuperstateUrlByName(node.getDefinitionName());
            if (StringUtils.isNotBlank(url) && !Utility.isFileExist(url)) {
                ValidationMessageAdder.getInstance().addValidationMessage(messages, node,
                        ValidationMessageCode.SUPERSTATE_NOT_FOUND, Arrays.asList(node.getName()));
                return;
            }
        }

        if (node.getParent() != null) {
            NodeDefinitionInstance nodeWrapper = new NodeDefinitionInstance(node, Utility.getTemplateOrSuperState(node));
            validateNodeDefinitionInstance(nodeWrapper);
            validateTemplateResourceTypeDependency(nodeWrapper);        
        }
    }

    private void validateNodeDefinitionInstance(NodeDefinitionInstance node) {
        NodeDefinition nodeDefinition = node.getNodeInstance();

        /* Use the Definition Name to determine if type is set */
        String definitionName = nodeDefinition.getDefinitionName();
        if (definitionName == null || definitionName.trim().length() == 0)
        {
            ValidationMessageCode code = ValidationMessageCode.NODE_NODE_DEFINITION_TYPE_UNDEFINED;
            ValidationMessageAdder.getInstance().addValidationMessage(messages, nodeDefinition, code);

            // no more validation required
            return;
        } 
        
        /*
         * The validation framework allows only one set of rules per class. Node Definition needs 3
         * separate sets of rules for Node Definitions in template, super-states and independent
         * node definitions. In order to generate 3 different sets of rules, use 3 classes but pass
         * the same type of object (NodeDefinition) to each class ['Creative Solution']. This breaks
         * the generics used for the validator.
         */

        /* validate the node definition instance */
        IValidator validator = validatorFactory.createValidator(Utility.getNodeGroup(nodeDefinition) != null ? SuperStateNodeDefinitionInstance.class : TemplateNodeDefinitionInstance.class);
        messages.addAll(validator.validate(nodeDefinition));

        /* validate the due date and duration expressions */
        NodeElementContainer container = node.getContainer();
        validateDueDate(nodeDefinition);

        /* validate that all variables in the node definition instance are mapped */
        ContainerNode containerNode = createContainerNode(node.getContainer(), nodeDefinition);
        messages.addAll(validatorFactory.createValidator(containerNode).validate(containerNode));
    }

    private void validateTemplateResourceTypeDependency(NodeDefinitionInstance node) {
        IResourceTypeService service = ServiceFactory.createFactory().createResourceTypeService();

        NodeDefinition nodeDefinition = node.getNodeInstance();
        /* Use the Definition Name to determine if type is set */
        String definitionName = nodeDefinition.getDefinitionName();
        if (definitionName != null && definitionName.trim().length() > 0) {
            String systemUID = null;
            if (nodeDefinition.getNodeType().equals(NodeDefinitionType.SuperState)) {
                NodeGroup group = (NodeGroup) nodeDefinition.getNode();
                
                for (NodeElement ssElement : group.getNodeElements()) {
                    if (ssElement instanceof NodeDefinition) {
                        NodeDefinition nd = (NodeDefinition) ssElement;
                        if (nd.getUid() != null) {
                            systemUID = Utility.getSystemIDByMessageUID(nd.getUid());
                            break;
                        }
                    }
                }
            } else if (nodeDefinition.getNodeType().equals(NodeDefinitionType.MessageNode)) {
                if (nodeDefinition.getUid() != null) {
                    systemUID = Utility.getSystemIDByMessageUID(nodeDefinition.getUid());
                }
            }
            
            if (systemUID != null) {
                ResourceType resourceType = service.getResourceTypeBySystemId(systemUID);
                if (resourceType == null) {
                    // Resource type not found.
                    ValidationMessageCode code = ValidationMessageCode.NODE_DEFINITION_TYPE_NOT_FOUND;
                    List<String> params = Arrays.asList(new String[] {definitionName});
                    ValidationMessageAdder.getInstance().addValidationMessage(messages, nodeDefinition, code, params);
                } else {
                    // Resource type found, verify if it hasErrors
                    if (resourceType.isHasErrors()) {
                        ValidationMessageCode code = ValidationMessageCode.NODE_DEFINITION_TYPE_IS_INVALID;
                        List<String> params = Arrays.asList(new String[] {resourceType.getName()});
                        ValidationMessageAdder.getInstance().addValidationMessage(messages, nodeDefinition, code, params);
                    } else {
                        // Resource type is ok, validate the nodedefinitions
                        if (nodeDefinition.getNodeType().equals(NodeDefinitionType.SuperState)) {
                            validateSuperStateUpToDate(resourceType, nodeDefinition);
                        } else if (nodeDefinition.getNodeType().equals(NodeDefinitionType.MessageNode)) {
                            validateMessageUpToDate(resourceType, nodeDefinition, false);
                            /* Validate the message used as Rollback */
                            if (nodeDefinition.getRollbackUid() != null && !nodeDefinition.getRollbackUid().trim().isEmpty()) {
                                validateMessageUpToDate(resourceType, nodeDefinition, true);
                            }
                        }
                            
                    }
                }
            }
        }
    }

    private void validateSuperStateUpToDate(ResourceType resourceType, NodeDefinition superState) {
        String definitionName = superState.getDefinitionName();
        
        NodeDefinitionDTO remoteNode = null;
        
        for (NodeDefinitionDTO node : resourceType.getNodes()) {
            if (node.getType().equals(NodeDefinitionType.SuperState)) {
                if (node.getName() != null && node.getName().equals(definitionName)) {
                    remoteNode = node;
                    break;
                }
            }
        }
        if (remoteNode == null) {
            // SuperState wasn't found 
            ValidationMessageCode code = ValidationMessageCode.SUPERSTATE_NOT_FOUND;
            List<String> params = Arrays.asList(new String[] {definitionName});
            ValidationMessageAdder.getInstance().addValidationMessage(messages, superState, code, params);
        } else {
            // SuperState found, verify if its up-to-date (same variables, name, etc..)
            try {
                NodeDefinition remoteNodeUIModel = SemanticElementUtil.createNodeDefinition(remoteNode.getXml());
                NodeDefinitionComparer comparer = new SuperStateComparer();
                ITree<IDiff> diff = comparer.compare(superState, remoteNodeUIModel, null);
                
                if (diff.getRoot().hasChildren()) {
                    ValidationMessageCode code = ValidationMessageCode.NODEDEFINITION_SUPERSTATE_OUTDATED;
                    List<String> params = Arrays.asList(new String[] {resourceType.getName(), definitionName});
                    ValidationMessageAdder.getInstance().addValidationMessage(messages, superState, code, params);
                }
            } catch (Exception e) {
            }
        }
    }

    private void validateMessageUpToDate(ResourceType resourceType, NodeDefinition message, boolean isRollback) {
        INodeDefinitionService nodeService = ServiceFactory.createFactory().createNodeDefinitionService();
        String definitionName = message.getDefinitionName();
        List<NodeDefinitionDTO> remoteNodes = nodeService.getNodeDefinitionsByMessageUID(message.getUid(), NodeDefinitionType.MessageNode);
        if (remoteNodes == null || remoteNodes.isEmpty()) {
            // ResourceType was found, but message wasn't. 
            ValidationMessageCode code = ValidationMessageCode.NODE_DEFINITION_MESSAGE_NOT_FOUND;
            List<String> params = Arrays.asList(new String[] {definitionName});
            ValidationMessageAdder.getInstance().addValidationMessage(messages, message, code, params);
        } else if (remoteNodes.size() == 1) {
            // Message found, verify if its up-to-date (same variables, name, etc..)
            // If there is more than 1 NodeDefinition for the Message then ResourceType must be broken.
            NodeDefinitionDTO remoteNode = remoteNodes.get(0);
            try {
                NodeDefinition remoteNodeUIModel = SemanticElementUtil.createNodeDefinition(remoteNode.getXml());
                NodeDefinitionComparer comparer = null;
                if (!isRollback) {
                    comparer = new NodeDefinitionComparer();
                } else {
                    comparer = new RollbackNodeDefinitionComparer();
                }
                ITree<IDiff> diff = comparer.compare(message, remoteNodeUIModel, null);
                
                if (diff.getRoot().hasChildren()) {
                    ValidationMessageCode code = ValidationMessageCode.NODEDEFINITION_MESSAGE_OUTDATED;
                    List<String> params = Arrays.asList(new String[] {resourceType.getName(), definitionName});
                    ValidationMessageAdder.getInstance().addValidationMessage(messages, message, code, params);
                }
            } catch (Exception e) {
            }
        }
    }

    public void visit(ProcessState node) {
        messages.addAll(validatorFactory.createValidator(ProcessState.class).validate(node));
        validateProcessStateNode(Utility.getTemplate(node), node);
    }

    private void validateProcessStateNode(WorkflowTemplate parentTemplate, ProcessState node) {
        SubProcess subproc = node.getSubProcess();
        String templatename = subproc.getName();
        if (templatename.indexOf(" - v") > 0) {
            templatename = templatename.substring(0,templatename.indexOf(" - v"));
        }
        IWorkflowTemplateService service = ServiceFactory.createFactory().createWorkflowTemplateService();
        WorkflowTemplateDTO childTemplateDTO = null;
        try {
            List<WorkflowTemplateDTO> childTemplatesDTO = service.getTemplateByName(templatename);
            
            if (childTemplatesDTO.isEmpty()) {
                return;
            }
        
            childTemplateDTO = childTemplatesDTO.get(0);
            IFile templateFile = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(childTemplateDTO.getPath()));

            String templateContent = new String(FileUtil.readFile(templateFile.getContents()));
            WorkflowTemplate childTemplate = SemanticElementUtil.createWorkflowTemplate(templateContent);               
            Variable[] vars = null;
            if (childTemplate.getStartState().getTask() != null) {
                vars = childTemplate.getStartState().getTask().getController().getVariables();
            }
            Map<String, Variable> childVars = createVariableMap("name", vars);
            Map<String, Variable> mappedVars = createVariableMap("mappedName", node.getVariables());
            vars = null;
            if (parentTemplate.getStartState().getTask() != null) {
                vars = parentTemplate.getStartState().getTask().getController().getVariables();
            }

            for (Variable parentMappedVar : mappedVars.values()) {
                DataType returnType = null;
                // Validates if child does not contain the mapped var
                if (!childVars.containsKey(parentMappedVar.getMappedName())) {
                    List<String> params = Arrays.asList(new String[] {childTemplate.getName(), parentMappedVar.getMappedName()});
                    ValidationMessageAdder.getInstance().addValidationMessage(messages, node, TEMPLATE_SUBPROCESS_VARIABLE_NOT_DECLARED_CHILD, params);
                } else {
                    returnType = DataType.valueOf(((WPVariable) childVars.get(parentMappedVar.getMappedName()))
                            .getType());
                }

                // First validate if the mapped var exists in the parent template or is an expression
                IProblem[] problems = Utility.isScriptValid(ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(childTemplateDTO.getPath())).getProject(), returnType, parentMappedVar.getName(), vars);
                if (problems != null) {
                    // Note that the expr parser isn't very strict...
                    List<String> params = null;
                    for (IProblem problem : problems) {
                        params = new ArrayList<String>();
                        params.add(parentMappedVar.getMappedName());
                        params.add(problem.getMessage());
                        ValidationMessageAdder.getInstance().addValidationMessage(messages, node, TEMPLATE_SUBPROCESS_INVALID_EXPRESSION, params);
                    }
                } else {
                    if (parentMappedVar.isWritable()) {
                        // Valid expr, but can't use here
                        List<String> params = asList(new String[] { parentMappedVar.getMappedName() });
                        ValidationMessageAdder.getInstance().addValidationMessage(messages, node, TEMPLATE_SUBPROCESS_EXPRESSION_NOT_ALLOWED, params);
                    } else {
                        // WARN, not error. May be a valid expression or a typo, it's difficult to tell.
                        List<String> params = Arrays.asList(new String[] {childTemplate.getName(), parentMappedVar.getName()});
                        ValidationMessageAdder.getInstance().addValidationMessage(messages, node, TEMPLATE_SUBPROCESS_VARIABLE_NOT_DECLARED_PARENT, params);
                    }
                }
            }

            for (Variable childVar : childVars.values()) {
                if (childVar.isRequired()) {
                    Variable mappedVar = mappedVars.get(childVar.getName());
                    if (mappedVar != null) {
                        // means that variable is mapped, check if it is mapped
                        // as "Read"
                        if (!mappedVar.isReadable()) {
                            List<String> params = Arrays.asList(new String[] { childTemplate.getName(),
                                    childVar.getName() });
                            ValidationMessageAdder.getInstance().addValidationMessage(messages, node,
                                    TEMPLATE_SUBPROCESS_VARIABLE_MUST_BE_MAPPED_AS_READ, params);
                        }
                    } else {
                        // If at least one of the variables is not mapped, raise 
                        // the error and stops the iteration
                        List<String> params = Arrays
                                .asList(new String[] { childTemplate.getName(), childVar.getName() });
                        ValidationMessageAdder.getInstance().addValidationMessage(messages, node,
                                TEMPLATE_SUBPROCESS_UNMAPPED_VARIABLE, params);
                    }
                }
            }
        } catch (Exception ie) {
            logger.debug(ie.getMessage());
        }
    }
 
    private Map<String, Variable> createVariableMap(String keyProperty, Variable[] vars) {
        Map<String, Variable> m = new HashMap<String, Variable>();
        if (vars != null) {
            for (Variable var : vars) {
                String key = "";
                if ("name".equals(keyProperty)) {
                    key = var.getName();
                } else if ("mappedName".equals(keyProperty)) {
                    key = var.getMappedName();
                }
                m.put(key, var);
            }
        }
        return m;
    }

    public void visit(Fork node) {
        messages.addAll(validatorFactory.createValidator(Fork.class).validate(node));
    }

    public void visit(AssignNode node) {
        messages.addAll(validatorFactory.createValidator(AssignNode.class).validate(node));
    }

    private void validateDueDate(WPAbstractNode node) {
        WorkflowTemplate template = Utility.getTemplate(node);
        if (template != null) {
            messages.addAll(validationService.validateDueDate((IDueDateElement) node, template));

        } else {
            NodeDefinition nodeDefinition = Utility.getNodeDefinition(node);
            if (nodeDefinition != null) {
                messages.addAll(validationService.validateDueDate((IDueDateElement) node, nodeDefinition));
            }
        }
    }

    private void validateCondition(WPAbstractNode node, Transition transition) {
        WorkflowTemplate template = Utility.getTemplate(node);
        if (template != null) {
            messages.addAll(validationService.validateCondition(node, transition, template));

        } else {
            NodeDefinition nodeDefinition = Utility.getNodeDefinition(node);
            if (nodeDefinition != null) {
                messages.addAll(validationService.validateCondition(node, transition, nodeDefinition));
            }
        }
    }

    private void validateLoopNodeExpression(LoopNode loopNode, NodeElementContainer container) {
        if (container instanceof WorkflowTemplate) {
            messages.addAll(validationService.validateLoopNodeExpression(loopNode, (WorkflowTemplate) container));

        } else if (container instanceof NodeDefinition) {
            messages.addAll(validationService.validateLoopNodeExpression(loopNode, (NodeDefinition) container));
        }
    }
}