/*
 * Created on Jul 11, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;
import org.jbpm.gd.common.model.GenericElement;
import org.jbpm.gd.jpdl.model.Decision;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.StartState;
import org.jbpm.gd.jpdl.model.Transition;

import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.ui.actionconfig.ActionClassConfiguration;
import com.tandbergtv.watchpoint.studio.ui.actionconfig.ActionConfigurationManager;
import com.tandbergtv.watchpoint.studio.ui.actionconfig.ActionVariableConfiguration;
import com.tandbergtv.watchpoint.studio.ui.model.AutomaticTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.IDueDateElement;
import com.tandbergtv.watchpoint.studio.ui.model.LoopNode;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WPSuperStateVariableDecorator;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.util.ModelHandler;
import com.tandbergtv.watchpoint.studio.util.FileUtil;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;
import com.tandbergtv.watchpoint.studio.validation.IValidationRule;
import com.tandbergtv.watchpoint.studio.validation.IValidationService;
import com.tandbergtv.watchpoint.studio.validation.IValidator;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationRuleInstantiationException;
import com.tandbergtv.watchpoint.studio.validation.ValidationServiceException;
import com.tandbergtv.watchpoint.studio.validation.ValidatorFactory;
import com.tandbergtv.watchpoint.studio.validation.graph.GraphFactory;
import com.tandbergtv.watchpoint.studio.validation.graph.IWatchPointGraph;
import com.tandbergtv.watchpoint.studio.validation.model.ActionVariable;
import com.tandbergtv.watchpoint.studio.validation.model.ConditionExpression;
import com.tandbergtv.watchpoint.studio.validation.model.DecisionExpression;
import com.tandbergtv.watchpoint.studio.validation.model.DueDateExpression;
import com.tandbergtv.watchpoint.studio.validation.model.LoopNodeExpression;
import com.tandbergtv.watchpoint.studio.validation.rules.actionvariable.ActionVariableValueRule;

/**
 * Implementation for the IValidationService interface.
 * 
 * @author Vijay Silva
 */
public class ValidationService implements IValidationService
{
	// The logger
	private static final Logger logger = Logger.getLogger(ValidationService.class);

	// The factory to create object validators
	private ValidatorFactory validatorFactory;

	// The factory to create graphs that can be validated
	private GraphFactory graphFactory;

	/**
	 * Constructor
	 */
	public ValidationService() {
		validatorFactory = ValidatorFactory.createFactory();
		graphFactory = GraphFactory.createFactory();
	}

	// ========================================================================
	// ===================== TEMPLATE VALIDATION
	// ========================================================================

	/**
	 * Builds the model object from the DTO object and does the validation on the model object.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationService#validateTemplate(com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO)
	 */
	public List<ValidationMessage> validateTemplate(WorkflowTemplateDTO template)
	{
		try {
			IFile xmlFile = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(template.getPath())); // FIXME Blows up if someone closed the project during build
			String xml = new String(FileUtil.readFile(xmlFile.getContents()));
			WorkflowTemplate templateSE = this.createWorkflowTemplate(xml);
			templateSE.setLocation(template.getPath());
			return this.validateTemplate(templateSE);
		} catch (Exception e) {
		    e.printStackTrace();
			logger.error("Error validating template " + template.getName(), e);
		}
		
		return new ArrayList<ValidationMessage>();
	}

	/**
	 * Validates the Template model object. Ensures that the template properties and the graph
	 * structure are valid. Also validates each of the template nodes.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationService#validateTemplate(com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate)
	 */
	public List<ValidationMessage> validateTemplate(WorkflowTemplate template)
	{
		List<ValidationMessage> allMessages = new ArrayList<ValidationMessage>();

		/* validate the template */
		addMessages(allMessages, validatorFactory.createValidator(WorkflowTemplate.class).validate(template));

		/* validate the template graph structure */
		IWatchPointGraph graph = this.graphFactory.createGraph(template);
		addMessages(allMessages, validatorFactory.createValidator(graph).validate(graph));

		SemanticElementValidatorVisitor visitor = new SemanticElementValidatorVisitor(this);
		template.accept(visitor);
		
		addMessages(allMessages, visitor.getMessages());
		
		return allMessages;
	}

	// ========================================================================
	// ===================== NODE DEFINITION VALIDATION
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationService#validateNodeDefinition(com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition)
	 */
	public List<ValidationMessage> validateNodeDefinition(NodeDefinition nodeDefinition)
	{
		List<ValidationMessage> allMessages = new ArrayList<ValidationMessage>();
		List<ValidationMessage> messages = null;

		/* validate the node definition */
		messages = this.validatorFactory.createValidator(NodeDefinition.class).validate(
				nodeDefinition);
		this.addMessages(allMessages, messages);

		/* validate the due date and duration expressions */
		allMessages.addAll(validateDueDate(nodeDefinition, nodeDefinition));

        if (nodeDefinition.getNodeType() == NodeDefinitionType.SuperState) {
			/* validate the node definition graph */
			IWatchPointGraph graph = this.graphFactory.createGraph(nodeDefinition);
			messages = this.validatorFactory.createValidator(graph).validate(graph);
			this.addMessages(allMessages, messages);
			
			SemanticElementValidatorVisitor visitor = new SemanticElementValidatorVisitor(this);
			nodeDefinition.accept(visitor);

			allMessages.addAll(visitor.getMessages());

        } else if (nodeDefinition.getNodeType() == NodeDefinitionType.MessageNode) {
			/* TODO: validate the Automatic Task Node in the node definition */
		}

		return allMessages;
	}
	
    // ========================================================================
	// ===================== RESOURCE TYPE VALIDATION
	// ========================================================================

	/**
	 * Validate the Resource Type DTO and all of the contained Message DTO objects.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationService#validateResourceType(com.tandbergtv.watchpoint.studio.dto.ResourceType)
	 */
	public List<ValidationMessage> validateResourceType(ResourceType resourceType)
	{
		List<ValidationMessage> allMessages = new ArrayList<ValidationMessage>();

		addMessages(allMessages, validatorFactory.createValidator(ResourceType.class).validate(resourceType));

		Set<Message> resourceTypeMessages = resourceType.getMessages();
        if (resourceTypeMessages != null) {
            for (Message message : resourceTypeMessages) {
                addMessages(allMessages, validateMessage(message));
            }
        }

		return allMessages;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationService#validateMessage(com.tandbergtv.watchpoint.studio.dto.Message)
	 */
	public List<ValidationMessage> validateMessage(Message message)
	{
		IValidator<Message> validator = this.validatorFactory.createValidator(Message.class);
		return validator.validate(message);
	}

	// ========================================================================
	// ===================== RESOURCE GROUP VALIDATION
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationService#validateResourceGroup(com.tandbergtv.watchpoint.studio.dto.ResourceGroup)
	 */
	public List<ValidationMessage> validateResourceGroup(ResourceGroup group)
	{
		return this.validatorFactory.createValidator(ResourceGroup.class).validate(group);
	}

	// ========================================================================
	// ===================== ACTION VARIABLE VALIDATION
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationService#validateActionVariableName(GenericElement)
	 */
	public List<ValidationMessage> validateActionVariableName(GenericElement element)
	{
		ActionVariable variable = new ActionVariable(element);
		return this.validatorFactory.createValidator(variable).validate(variable);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationService#validateActionVariableValue(AutomaticTaskNode,
	 *      GenericElement)
	 */
	public List<ValidationMessage> validateActionVariableValue(AutomaticTaskNode node,
			GenericElement element)
	{
		String className = (node != null && node.getAction() != null) ? node.getAction()
				.getClassName() : null;
		String variableName = element.getName();

		ActionConfigurationManager manager = ActionConfigurationManager.getInstance();
		ActionClassConfiguration actionClass = manager.getActionClass(className);
		ActionVariableConfiguration variableConfiguration = null;
		if (actionClass != null)
		{
			List<ActionVariableConfiguration> variables = actionClass.getVariables();
			if (variables != null)
			{
				for (ActionVariableConfiguration currentConfiguration : variables)
				{
					if (currentConfiguration.getName().equals(variableName))
					{
						variableConfiguration = currentConfiguration;
						break;
					}
				}
			}
		}

		/* Find the Rule to execute for the value */
		ActionVariableValueRule rule = this.createActionVariableRule(variableConfiguration);
		ActionVariable actionVariable = new ActionVariable(node, element, variableConfiguration);
		return rule.validateRule(actionVariable);
	}

	/*
	 * Create the Rule used to validate the value of the Action Variable
	 */
	private ActionVariableValueRule createActionVariableRule(ActionVariableConfiguration variable)
	{
		ActionVariableValueRule valueRule = null;

		String ruleClassName = (variable != null) ? variable.getValidationRuleClassName() : null;
		if (ruleClassName != null && ruleClassName.trim().length() > 0)
		{
			try
			{
				IValidationRule<?> rule = this.validatorFactory.createValidationRule(ruleClassName);
				if (rule instanceof ActionVariableValueRule)
				{
					valueRule = (ActionVariableValueRule) rule;
				}
				else
				{
					String msg = "The Rule: " + ruleClassName + " does not extend class: "
							+ ActionVariableValueRule.class.getName()
							+ ", cannot use specified rule for Action Variable validation"
							+ ", using default implementation.";
					logger.warn(msg);
				}
			}
			catch (ValidationRuleInstantiationException ex)
			{
				String msg = "Failed to validate ActionVariableValueRule: " + ruleClassName
						+ ", using default implementation.";
				logger.warn(msg, ex);
			}
		}

		if (valueRule == null)
			valueRule = new ActionVariableValueRule();

		return valueRule;
	}
	
	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/* Adds new list of validation messages to the list of all existing validation messages. */
	private void addMessages(List<ValidationMessage> allMessages, List<ValidationMessage> messages)
	{
		if (messages != null)
			allMessages.addAll(messages);
	}

    private WorkflowTemplate createWorkflowTemplate(String xml) {
        try {
            return SemanticElementUtil.createWorkflowTemplate(xml);
        } catch (Exception ex) {
            String msg = "Failed to construct Semantic Element: WorkflowTemplate from the XML String.";
            throw new ValidationServiceException(msg, ex);
        }
	}

    public List<ValidationMessage> validateCondition(NodeElement node, Transition transition, WorkflowTemplate template) {
        List<WPVariable> variables = this.getContainerVariables(template);
        ConditionExpression expression = new ConditionExpression(template, variables, node,
                transition);
        return validatorFactory.createValidator(expression).validate(expression);
    }

    public List<ValidationMessage> validateCondition(NodeElement node, Transition transition,
            NodeDefinition nodeDefinition) {
        List<WPVariable> variables = (List<WPVariable>) getContainerVariables(nodeDefinition);
        ConditionExpression expression = new ConditionExpression(nodeDefinition, variables, node, transition);
        return validatorFactory.createValidator(expression).validate(expression);
    }


    public List<ValidationMessage> validateLoopNodeExpression(LoopNode loopNode, WorkflowTemplate template) {
        List<WPVariable> variables = getContainerVariables(template);
        LoopNodeExpression expression = new LoopNodeExpression(template, variables, loopNode);
        return validatorFactory.createValidator(expression).validate(expression);
    }

    public List<ValidationMessage> validateLoopNodeExpression(LoopNode loopNode, NodeDefinition nodeDefinition) {
        List<WPVariable> variables = (List<WPVariable>) getContainerVariables(nodeDefinition);
        LoopNodeExpression expression = new LoopNodeExpression(nodeDefinition, variables, loopNode);
        return validatorFactory.createValidator(expression).validate(expression);
    }


    public List<ValidationMessage> validateDueDate(IDueDateElement node, WorkflowTemplate template) {
        List<WPVariable> variables = getContainerVariables(template);
        DueDateExpression expression = new DueDateExpression(template, variables, node);
        return validatorFactory.createValidator(expression).validate(expression);
    }

    public List<ValidationMessage> validateDueDate(IDueDateElement node, NodeDefinition nodeDefinition) {
        DueDateExpression expression = new DueDateExpression(nodeDefinition, null, node);
        return validatorFactory.createValidator(expression).validate(expression);
    }

    public List<ValidationMessage> validateDecisionExpression(Decision decision, WorkflowTemplate template) {
        List<WPVariable> variables = getContainerVariables(template);
        DecisionExpression expression = new DecisionExpression(template, variables, decision);
        return validatorFactory.createValidator(expression).validate(expression);
    }

    public List<ValidationMessage> validateDecisionExpression(Decision decision, NodeDefinition nodeDefinition) {
        List<WPVariable> variables = (List<WPVariable>) getContainerVariables(nodeDefinition);
        DecisionExpression expression = new DecisionExpression(nodeDefinition, variables, decision);
        return validatorFactory.createValidator(expression).validate(expression);
    }


    public List<WPVariable> getContainerVariables(WorkflowTemplate template) {
        List<WPVariable> variables = new ArrayList<WPVariable>();

        StartState startState = template.getStartState();
        for (WPVariable variable : ModelHandler.getStartStateValidVariables(startState)) {
            variables.add(variable);
        }

        return variables;
    }

    public List<? extends WPVariable> getContainerVariables(NodeDefinition nodeDefinition) {
        List<WPSuperStateVariableDecorator> variables = new ArrayList<WPSuperStateVariableDecorator>();

        if (nodeDefinition.getNodeType() == NodeDefinitionType.SuperState) {
            List<WPVariable> nodeDefinitionVariables = nodeDefinition.getVariables();
            if (nodeDefinitionVariables != null) {
                for (WPVariable variable : nodeDefinitionVariables) {
                    variables.add(new WPSuperStateVariableDecorator(variable));
                }
            }
        }

        return variables;
    }
}
