/*
 * Created on Nov 13, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.nodedefinition;

import java.util.ArrayList;
import java.util.List;

import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;
import com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.NodeVariablesRule;

/**
 * Validates the variables of a Super-State Node Definition
 * 
 * @author Vijay Silva
 */
public class SuperStateVariablesRule extends NodeVariablesRule<NodeDefinition>
{
	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.NodeVariablesRule#getVariables(org.jbpm.gd.jpdl.model.NodeElement)
	 */
	@Override
	protected List<Variable> getVariables(NodeDefinition node)
	{
		List<Variable> variables = new ArrayList<Variable>();

		if (node.getNodeType() == NodeDefinitionType.SuperState)
		{
			List<WPVariable> nodeVariables = node.getVariables();
			if (nodeVariables != null)
				variables.addAll(nodeVariables);
		}

		return variables;
	}

	// ========================================================================
	// ===================== DISABLING UNWANTED VALIDATIONS
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.NodeVariablesRule#isValidatingBlankNames(org.jbpm.gd.jpdl.model.NodeElement)
	 */
	@Override
	protected boolean isValidatingBlankNames(NodeDefinition node)
	{
		return false;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.NodeVariablesRule#isValidatingNameBlank(org.jbpm.gd.jpdl.model.NodeElement)
	 */
	@Override
	protected boolean isValidatingNameBlank(NodeDefinition node)
	{
		return false;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.NodeVariablesRule#isValidatingNameValue(org.jbpm.gd.jpdl.model.NodeElement)
	 */
	@Override
	protected boolean isValidatingNameValue(NodeDefinition node)
	{
		return false;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.NodeVariablesRule#isValidatingMappedNameBlank(org.jbpm.gd.jpdl.model.NodeElement)
	 */
	@Override
	protected boolean isValidatingMappedNameBlank(NodeDefinition node)
	{
		return false;
	}

	// ========================================================================
	// ===================== DISABLING UNWANTED VALIDATIONS
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.NodeVariablesRule#getBlankMappedNamesCode(org.jbpm.gd.jpdl.model.NodeElement)
	 */
	@Override
	protected ValidationMessageCode getBlankMappedNamesCode(NodeDefinition node)
	{
		return ValidationMessageCode.SUPERSTATE_VARIABLES_BLANK_MAPPED_NAMES;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.NodeVariablesRule#getMappedNameInvalidCode(org.jbpm.gd.jpdl.model.NodeElement)
	 */
	@Override
	protected ValidationMessageCode getMappedNameInvalidCode(NodeDefinition node)
	{
		return ValidationMessageCode.VARIABLE_MAPPED_NAME_INVALID_IN_SUPERSTATE;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.NodeVariablesRule#getTypeBlankCode(org.jbpm.gd.jpdl.model.NodeElement)
	 */
	@Override
	protected ValidationMessageCode getTypeBlankCode(NodeDefinition node)
	{
		return ValidationMessageCode.VARIABLE_TYPE_BLANK_IN_SUPERSTATE;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.NodeVariablesRule#getTypeInvalidCode(org.jbpm.gd.jpdl.model.NodeElement)
	 */
	@Override
	protected ValidationMessageCode getTypeInvalidCode(NodeDefinition node)
	{
		return ValidationMessageCode.VARIABLE_TYPE_INVALID_IN_SUPERSTATE;
	}

	// ========================================================================
	// ===================== NAME VALIDATION
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.validation.rules.nodeelement.NodeVariablesRule#validateVariables(org.jbpm.gd.jpdl.model.NodeElement,
	 *      java.util.List, java.util.List)
	 */
	@Override
	protected void validateVariables(NodeDefinition node, List<Variable> variables,
			List<ValidationMessage> messages)
	{
		super.validateVariables(node, variables, messages);

		this.validateNamesNotConfigured(node, variables, messages);
	}

	/*
	 * Validate that all the Super State Variables do not have a name configured.
	 */
	private void validateNamesNotConfigured(NodeDefinition node, List<Variable> variables,
			List<ValidationMessage> messages)
	{
		if (variables != null)
		{
			boolean hasValues = false;

			for (Variable variable : variables)
			{
				String name = variable.getName();
				if (name != null && name.trim().length() > 0)
				{
					hasValues = true;
					break;
				}
			}

            if (hasValues) {
                ValidationMessageAdder.getInstance().addValidationMessage(messages, node, ValidationMessageCode.SUPERSTATE_VARIABLES_NAME_DEFINED);
            }
		}
	}
}
