/*
 * Created on Aug 28, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.resourcetype;

import java.util.ArrayList;
import java.util.List;

import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * Rule that performs validation of the Resource Type properties related to management
 * configuration.
 * 
 * @author Vijay Silva
 */
public class ManagementPropertiesRule extends ValidationRule<ResourceType>
{
	/**
	 * Validates the Management properties (HeartBeat, Initialization) of the Resource Type.
	 * 
	 * @param target
	 *            the Resource Type to validate
	 * 
	 * @return The list of validation messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	public List<ValidationMessage> validateRule(ResourceType target)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		this.validateInitializationProperties(target, messages);
		this.validateHeartbeatProperties(target, messages);

		return messages;
	}

	/*
	 * validate the initialization parameters
	 */
	protected void validateInitializationProperties(ResourceType resourceType,
			List<ValidationMessage> messages)
	{
		if (!resourceType.isInitializationRequired())
			return;

		/* Initialization required, validate properties */
		String strategyClass = resourceType.getInitializationStrategyClass();
		if (strategyClass == null || strategyClass.trim().length() == 0)
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, ValidationMessageCode.RESOURCE_TYPE_INITIALIZATION_STRATEGY_BLANK);
		}
		else if (strategyClass.length() > 256)
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, ValidationMessageCode.RESOURCE_TYPE_INITIALIZATION_STRATEGY_TOO_LONG);
		}
	}

	/*
	 * validate the heart beat parameters
	 */
	protected void validateHeartbeatProperties(ResourceType resourceType,
			List<ValidationMessage> messages)
	{
		if (!resourceType.isHeartbeatRequired())
			return;

		/* HeartBeat is required, need to check properties */
		if (resourceType.getHeartbeatConnectionType() == null)
		{
			ValidationMessageCode code = ValidationMessageCode.RESOURCE_TYPE_HEARTBEAT_CONNECTION_TYPE_BLANK;
			ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, code);
		}

		String strategyClass = resourceType.getHeartbeatStrategyClass();
		if (strategyClass == null || strategyClass.trim().length() == 0)
		{
			ValidationMessageCode code = ValidationMessageCode.RESOURCE_TYPE_HEARTBEAT_STRATEGY_BLANK;
			ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, code);
		}
		else if (strategyClass.length() > 256)
		{
			ValidationMessageCode code = ValidationMessageCode.RESOURCE_TYPE_HEARTBEAT_STRATEGY_TOO_LONG;
			ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, code);
		}
	}
}
