package com.tandbergtv.watchpoint.studio.debugger.runtime;

import java.util.List;
import java.util.Map;
import java.util.Stack;

import org.jbpm.graph.def.Node;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.graph.def.Transition;
import org.jbpm.graph.exe.ExecutionContext;
import org.jbpm.graph.exe.ProcessInstance;
import org.jbpm.graph.node.Fork;
import org.jbpm.graph.node.Join;

import com.tandbergtv.watchpoint.studio.debugger.core.graph.Loop2;
import com.tandbergtv.watchpoint.studio.debugger.core.graph.exe.ExecutionContext2;
import com.tandbergtv.watchpoint.studio.debugger.model.NodeSimulationConfig;
import com.tandbergtv.watchpoint.studio.debugger.model.SimulationType;
import com.tandbergtv.watchpoint.studio.debugger.runtime.nodesimulation.NodeSimulator;
import com.tandbergtv.watchpoint.studio.debugger.runtime.output.TemplateSimulationOutputWriter;

/**
 * 		Simulates the Template in a recursive way.
 * 
 * @author <a href="mailto:francisco.bento.silva.neto@ericsson.com">efrasio - Francisco Bento da Silva Neto</a>
 *
 */
public class RecursiveTemplateSimulator extends TemplateSimulator {

	private Stack<Integer> forkState = new Stack<Integer>();
	private ExecutionContext executionContext;
	private TemplateSimulationOutputWriter writer;
	
	private Loop2 loop;
	private boolean hasFailed = false;
	
	public RecursiveTemplateSimulator(ProcessDefinition template,
									  Map<String, NodeSimulationConfig> simulationConfigMap,
									  TemplateSimulationOutputWriter writer) {
		super(template, simulationConfigMap);
		this.writer = writer;
	}
	
	public void runNode(Node node) {
		NodeSimulator nodeSimulator = simulatorFactory.getSimulatorFor(node);
		writer.nodeSimulationStarted(executionContext, node, nodeSimulator);
		if (nodeSimulator != null) {
			if (!SimulationType.SKIP_NODE.equals(nodeSimulator.getSimulationConfig().getSimulationType())) {
				nodeSimulator.startSimulation(executionContext);
			}
			hasFailed = nodeSimulator.fail();
		}
		writer.nodeSimulationCompleted(executionContext, node, nodeSimulator);

		if (hasFailed) {
			return;
		}			

		if (node instanceof Join) {
			int currentForkState = forkState.pop();
			currentForkState--;
			if (currentForkState > 0) {
				forkState.push(currentForkState);
				return;
			}
		} else if (node instanceof Fork) {
			forkState.push(node.getLeavingTransitions().size());
		} 
		
		if (node instanceof Loop2) {
			this.loop = (Loop2) node;
		} else if (loop != null) {
			if (!loop.evaluateExpression(executionContext)) {
				// its inside a loop and loop condition has failed: break loop 
				return;
			}
		}
		
		List<?> transitions = null;
		if (nodeSimulator != null) {
			transitions = nodeSimulator.getLeavingTransitions();
		} else {
			transitions = node.getLeavingTransitions();
		}
		if (transitions != null) {
			for (Object obj : transitions) {
				Transition transition = (Transition) obj;
				if (transition.getTo() != null) {
					if (hasFailed) {
						return;
					}
					if (node != loop) {
						// its not a loop, run single node
						runNode(transition.getTo());
					} else {
						// its a loop, loops over loop nodes. If the loop evaluation fails, skip loop at any time
						while (loop.evaluateExpression(executionContext)) {
							runNode(transition.getTo());
							if (hasFailed) {
								return;
							}
							writer.nodeSimulationStarted(executionContext, node, nodeSimulator);
							loop.incrementIndex(executionContext);
							writer.nodeSimulationCompleted(executionContext, node, nodeSimulator);
						}
						loop = null;
						Transition loopExit = (Transition) node.getLeavingTransitions().get(0);
						runNode(loopExit.getTo());
					}
				}
			}
		}
	}
	
	@Override
	public void startSimulation() {
		ProcessInstance processInstance = new ProcessInstance(template);
		executionContext = new ExecutionContext2(processInstance.getRootToken());
		
		writer.templateSimulationStarted(executionContext, template);
		Node currentNode = template.getStartState();
		runNode(currentNode);
		writer.templateSimulationCompleted(executionContext, template);
	}

}
