/*
 * Created on Jun 25, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.dataaccess;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;

/**
 * The Data Access Interface for the NodeDefinitionDTO entity.
 * 
 * @author Vijay Silva
 */
public interface NodeDefinitionDTODAI extends DataAccessInterface<NodeDefinitionDTO, Long>
{
	/**
	 * Gets the list of Node Definitions for a Resource Type.
	 * 
	 * @param resourceTypeId
	 *            The key for the ResourceType
	 * 
	 * @return The List of matching Node Definitions.
	 */
	List<NodeDefinitionDTO> findByResourceType(long resourceTypeId);

	/**
	 * Gets the list of Single-Node Node Definitions for a Resource Type.
	 * 
	 * @param resourceTypeId
	 *            The key for the ResourceType
	 * 
	 * @return The List of matching Node Definitions.
	 */
	List<NodeDefinitionDTO> findSingleNodeDefinitionsByResourceType(long resourceTypeId);
	
	/**
	 * Gets the list of Single-Node Node Definitions for a Resource Type.
	 * 
	 * @param resourceTypeId
	 *            The key for the ResourceType
	 *        incoming
	 * 
	 * @return The List of matching Node Definitions.
	 */
	List<NodeDefinitionDTO> findSingleNodeDefinitionsByResourceTypeAndMessageType(long resourceTypeId, boolean incoming);

	/**
	 * Gets the list of existing Node Definitions given the list of Node Definition Keys.
	 * 
	 * @param keys
	 *            the Node Definition Keys
	 * 
	 * @return The List of existing Node Definitions with matching keys.
	 */
	List<NodeDefinitionDTO> findByKeys(Collection<Long> keys);

	/**
	 * Finds all node definitions which are parents of the the node definition whose id is passed.
	 * 
	 * @param nodeDefinitionId
	 *            The key of the node definition whose parents are being searched
	 * @return List of parent node definitions
	 */
	public List<NodeDefinitionDTO> findByParentUsage(long nodeDefinitionId);
	
	/**
	 * Get the Node Definition DTO by its Path.
	 * 
	 * @param name
	 *            The Node Definition DTO Path
	 * 
	 * @return NodeDefinitionDTO with matching path.
	 */
	NodeDefinitionDTO findByPath(String path);

	/**
	 * Finds all super state node definitions that are mapped to exactly the same given set of messages.
	 * 
	 * @param messages
	 * 			Set of Messages
	 * @return	List of Node Definitions
	 */
	List<NodeDefinitionDTO> findSuperStateNodeDefinitionsByMessages(Set<Message> messages);

	/**
	 * Finds all single-node node definitions that are mapped to the same message.
	 * 
	 * @param id
	 * 			Id of the message
	 * @return	List of Node Definitions
	 */
	List<NodeDefinitionDTO> findSingleNodeDefinitionsByMessage(long id);
	
	/**
	 * Gets the count of the Node Definitions that currently use the specified node definition as a
	 * child / nested node definition.
	 * 
	 * @param nodeDefinitionId
	 *            The Node Definition Id
	 * 
	 * @return The count of parent node definitions that use the node definition with matching Id
	 */
	int getCountByParentUsage(long nodeDefinitionId);

	/**
	 * Gets the count of the number of Node Definitions with matching name
	 * 
	 * @param name
	 *            The Node Definition name to match against.
	 * 
	 * @return The Count of matching Node Definitions.
	 */
	int getCountByName(String name);

	/**
	 * Gets a count of Node Definitions for a Resource Type.
	 * 
	 * @param resourceTypeId
	 *            The key for the ResourceType
	 * 
	 * @return The count of matching Node Definitions.
	 */
	int getCountByResourceType(long resourceTypeId);

	/**
	 * Gets a count of Node Definitions that use a particular Message.
	 * 
	 * @param messageId
	 *            The key for the Message
	 * 
	 * @return The count of matching Node Definitions.
	 */
	int getCountByMessage(long messageId);

	/**
	 * Deletes all references to this node definition by parent node definitions.
	 * 
	 * @param nodeDefinitionId
	 *            the key of the node definition hwose references are to be deleted
	 */
	void deleteNodeDefinitionReferences(long nodeDefinitionId);


	/**
	 * Get all the NodeDefinitions for a message
	 * 
	 * @param messageUID
	 *            the messageUID
	 *            
	 * @param type
	 * 			  the Node type
	 */
	List<NodeDefinitionDTO> findByMessageUID(String messageUID, NodeDefinitionType type);
}
