/**
 * NodeDefinitionParser.java
 * Created Feb 18, 2010
 */
package com.tandbergtv.watchpoint.studio.external.fs;

import static com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType.MessageNode;
import static com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType.SuperState;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants.GPD_EXTENSION;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants.XML_EXTENSION;
import static javax.xml.xpath.XPathConstants.NODESET;

import java.io.File;
import java.io.FilenameFilter;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.xml.transform.TransformerException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathException;
import javax.xml.xpath.XPathFactory;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.tandbergtv.watchpoint.studio.application.StudioRuntimeException;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.util.FileUtil;

/**
 * Parses node definition XML and GPD files
 * 
 * @author Sahil Verma
 */
public class NodeDefinitionParser {

	private static final String NODE_DEFINITION_XPATH = "//super-state/nodeDefinition";

	private static final String RECEIVE_XPATH = "//receive";

	private static final String SEND_XPATH = "//send";

	private static final String DEFINITION_NAME = "definitionName";

	private static final String ID = "id";

	private static final String NAME = "name";

	private static final String RESOURCE_TYPE = "resourceType";

	private static final String SUPER_STATE = "super-state";

	private static final String AUTO_TASK = "auto-task";

	private String location;

	private Set<Message> messages;

	private Set<NodeDefinitionDTO> nodes;

	private Map<NodeDefinitionDTO, Node> deferred;

	private static final Logger logger = Logger.getLogger(NodeDefinitionParser.class);

	/**
	 * Creates the parser
	 * 
	 * @param location
	 * @param messages
	 */
	public NodeDefinitionParser(String location, Set<Message> messages) {
		super();
		this.location = location;
		this.messages = messages;
		this.nodes = new HashSet<NodeDefinitionDTO>();
		this.deferred = new HashMap<NodeDefinitionDTO, Node>();
	}

	/**
	 * Parses node definitions
	 * 
	 * @return
	 */
	public Collection<NodeDefinitionDTO> parse() {
		File dir = new File(location);

		if (!dir.exists() || !dir.isDirectory())
			return nodes;

		File[] files = dir.listFiles(new FilenameFilter() {
			public boolean accept(File dir, String name) {
				return name.toLowerCase().endsWith("xml");
			}
		});

		if (files == null)
			return nodes;

		for (int i = 0; i < files.length; i++) {
			File file = files[i];
			
			try {
				NodeDefinitionDTO node = parse(file);

				if (node != null)
					nodes.add(node);
				
				logger.debug("Parsed node " + node.getName());
			} catch (Exception e) {
				logger.warn("Ignored node definition file " + file.getAbsolutePath() + " - " + e.getMessage());
			}
		}

		/* Go through superstates again */
		for (NodeDefinitionDTO superstate : this.deferred.keySet()) {
			try {
				parseSuperState(superstate, this.deferred.get(superstate));
			} catch (Exception e) {
				this.nodes.remove(superstate);
				logger.warn("Ignored superstate node definition (" + superstate.getName() + ") - " + e.getMessage());
			}
		}

		return this.nodes;
	}

	/**
	 * Parses the node definition XML. Also looks for the associated GPD file.
	 * 
	 * @param file
	 * @return
	 * @throws TransformerException 
	 * @throws XPathException 
	 */
	public NodeDefinitionDTO parse(File file) throws TransformerException, XPathException {
		NodeDefinitionDTO node = new NodeDefinitionDTO();
		Document document = FileUtil.readDocument(file.getAbsolutePath());
		File gpd = new File(file.getAbsolutePath().replace(XML_EXTENSION, GPD_EXTENSION));

		parseGpd(node, gpd);
		parseNodeDefinition(node, document.getDocumentElement());
		// Set node path
		IPath path = new Path(file.getAbsolutePath());
		path = path.removeFirstSegments(path.segmentCount() - 4);
		path = path.setDevice("");
		node.setPath(path.toString());
		return node;
	}

	protected void parseNodeDefinition(NodeDefinitionDTO node, Node n) throws TransformerException,
			XPathException {
		parseXml(node, n);
		parseProperties(node, n);

		/* Superstates don't have messages, only their child nodes do. We'll revisit superstates
		 * once all other nodes have been processed */
		if (node.getType() == SuperState) {
			this.deferred.put(node, n);
			return;
		}

		parseMessage(node, n);
	}

	/**
	 * Children of the specified superstate that are node definitions have already been parsed. This
	 * method establishes the relationship between the superstate and its children.
	 * 
	 * @param node
	 * @param n
	 * @throws XPathException
	 * @throws TransformerException
	 */
	protected void parseSuperState(NodeDefinitionDTO node, Node n) throws XPathException,
			TransformerException {
		XPath xpath = XPathFactory.newInstance().newXPath();
		NodeList nodelist = (NodeList) xpath.evaluate(NODE_DEFINITION_XPATH, n, NODESET);

		for (int i = 0; i < nodelist.getLength(); i++) {
			Element e = (Element) nodelist.item(i);
			String name = e.getAttribute(DEFINITION_NAME);

			for (NodeDefinitionDTO dto : this.nodes) {
				if (name.equals(dto.getName())) {
					node.addChild(dto);
					node.addMessage(dto.getMessages().iterator().next());
					break;
				}
			}
		}
	}

	/**
	 * Parses properties of the specified node definition using the XML element
	 * 
	 * @param node
	 * @param document
	 */
	protected void parseProperties(NodeDefinitionDTO node, Node n) {
		Element e = ((Element) n);
		String name = e.getAttribute(NAME);
		node.setName(name);

		NodeList children = e.getElementsByTagName(SUPER_STATE);

		if (children == null || children.getLength() == 0) {
			node.setType(MessageNode);
		} else {
			node.setType(SuperState);
		}
	}

	protected void parseXml(NodeDefinitionDTO node, Node n) throws TransformerException {
		Element e = ((Element) n);

		e.setAttribute(RESOURCE_TYPE, null);

		if (e.hasAttribute(ID))
			e.setAttribute(ID, null); /* We'll have to change the ids later */

		String xml = FileUtil.convertDocument(n);

		node.setXml(xml);
	}

	/**
	 * @param file
	 * @return
	 * @throws TransformerException 
	 */
	protected void parseGpd(NodeDefinitionDTO node, File file) throws TransformerException {
		if (!file.exists())
			throw new StudioRuntimeException("No GPD found for node " + node.getName());

		Document document = FileUtil.readDocument(file.getAbsolutePath());
		String xml = FileUtil.convertDocument(document);

		node.setGpd(xml);
	}

	protected void parseMessage(NodeDefinitionDTO node, Node n) throws XPathException {
		String uid = parseUid(n);

		if (uid == null)
			return;

		Message message = getMessage(uid);

		if (message == null)
			throw new StudioRuntimeException("No message uid [" + uid + "] found for " + node.getName());

		Set<Message> messages = new HashSet<Message>();

		messages.add(message);
		node.setMessages(messages);
	}

	protected String parseUid(Node n) throws XPathException {
		Element e = (Element) n;
		NodeList autotasks = e.getChildNodes();

		for (int i = 0; i < autotasks.getLength(); i++) {
			Node autotask = autotasks.item(i);

			if (!(autotask instanceof Element && autotask.getNodeName().equals(AUTO_TASK)))
				continue;

			XPath xpath = XPathFactory.newInstance().newXPath();
			String uid = (String) xpath.evaluate(SEND_XPATH, autotask, XPathConstants.STRING);

			if (uid == null || uid.trim().length() == 0)
				uid = (String) xpath.evaluate(RECEIVE_XPATH, autotask, XPathConstants.STRING);

			return uid.trim();
		}

		return null;
	}

	private Message getMessage(String uid) {
		for (Message message : this.messages) {
			if (message.getUid().equals(uid))
				return message;
		}

		return null;
	}
}
